/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author JJ Frankovich
 */

/**
 * API Status: **Private**
 * @module nmodule/webChart/rc/line/ZoomLayer
 */
define(["d3", "nmodule/webChart/rc/line/XAxisLayer", "nmodule/webChart/rc/webChartUtil"], function (d3, XAxisLayer, webChartUtil) {
  "use strict";

  /**
   * A Layer in a Line Chart to handle Zooming
   *
   * @class
   * @alias module:nmodule/webChart/rc/line/ZoomLayer
   */
  var ZoomLayer = function ZoomLayer(graph) {
    this.$graph = graph;
  };
  ZoomLayer.prototype.redraw = function () {
    // Bind events to the line-area of the chart
    var that = this,
      graph = that.$graph,
      jq = graph.widget().jq().find(".line-container-overlay-group"),
      chart = graph.chartd3(),
      zoom = d3.zoom(),
      scaleX = graph.getScaleX(),
      model = graph.widget().model();
    var initialXDomain;
    var selection = chart.select('.line-container-overlay');
    zoom(selection);
    function zoomStart(event) {
      var sourceEvent = event.sourceEvent,
        valueScalesLocked = false;
      var hasAlt = sourceEvent && sourceEvent.altKey;
      var hasShiftOrCtrl = sourceEvent && (sourceEvent.shiftKey || sourceEvent.ctrlKey);
      model.mapValueScales(function (valueScale) {
        if (valueScale.isLocked()) {
          valueScalesLocked = true;
        }
      });
      if (valueScalesLocked) {
        hasShiftOrCtrl = true;
        hasAlt = false;
      }
      if (!hasAlt) {
        initialXDomain = scaleX.domain();
      } else {
        initialXDomain = null;
      }
      var index = 0;
      model.mapValueScales(function (valueScale) {
        if (!hasShiftOrCtrl) {
          var valueZoom = d3.zoom();
          if (!hasShiftOrCtrl && index === 0) {
            valueZoom = zoom; //no x to use as base zoom, so use first valueScale
          }
          valueScale.zoom = valueZoom;
          valueScale.initialDomain = valueScale.scale().domain();
        } else {
          valueScale.zoom = null;
          valueScale.initialDomain = null;
        }
        index++;
      });
      chart.select('.line-container-overlay-group').call(function (selection) {
        var isWheelEvent = sourceEvent && sourceEvent.type === "wheel";
        if (isWheelEvent) {
          return; //no pointerup event for a mouse wheel
        }

        //store original cursor
        var altCursor;
        selection.attr("originalCursor", jq.css("cursor"));
        if (!hasAlt && !hasShiftOrCtrl) {
          altCursor = "move";
        } else if (hasAlt) {
          altCursor = "n-resize";
        } else if (hasShiftOrCtrl) {
          altCursor = "w-resize";
        }
        selection.attr("altCursor", altCursor);
      }).on("pointerup.pan", function () {
        var select = d3.select(this);
        jq.css("cursor", select.attr("originalCursor"));
        select.attr("originalCursor", null);
        select.attr("altCursor", null);
      });
    }
    zoom.on('start', zoomStart, {
      passive: true
    });
    zoom.on('end', function () {
      //d3 v7 has some issues on the first zoom after a zoom end, this seems to correct it.
      //This will be something to check on new version to see if this workaround can be removed.
      selection.property("__zoom", d3.zoomIdentity.scale(1));
    });
    //for easier testing
    selection.nodes()[0]._zoomStart = zoomStart;
    zoom.on('zoom', function (event) {
      var oldDomain,
        checkDomainResults,
        zoomModified = false;
      if (event && initialXDomain) {
        scaleX.domain(initialXDomain);
        var newScale = event.transform.rescaleX(scaleX);
        scaleX.domain(newScale.domain());
        zoomModified = true;
      }
      model.mapValueScales(function (valueScale) {
        var valueZoom = valueScale.zoom;
        if (valueZoom) {
          var initialValueDomain = valueScale.initialDomain;
          var scale = valueScale.scale();
          scale.domain(initialValueDomain);
          var _newScale = event.transform.rescaleY(scale);
          scale.domain(_newScale.domain());
          zoomModified = true;
        }
      });
      oldDomain = zoom.lastXZoomDomain || scaleX.domain();
      checkDomainResults = XAxisLayer.checkDomain(scaleX.domain(), oldDomain);
      if (checkDomainResults.tooHigh || checkDomainResults.tooLow || checkDomainResults.tooSmall) {
        scaleX.domain(checkDomainResults.newDomain);
      } else {
        zoom.lastXZoomDomain = scaleX.domain();
        zoomModified = true;
      }
      if (zoomModified) {
        graph.widget().manualZoom();
      }
    }, {
      passive: true
    });
    chart.select('.line-container-overlay-group').on("pointermove.pan", function () {
      var select = d3.select(this);
      jq.css("cursor", select.attr("altCursor"));
      select.on("pointermove.pan", null);
    });
  };
  return ZoomLayer;
});
