/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Scott Hoye
 */

define(['baja!', 'Promise', 'lex!search'], function (baja, Promise, lexs) {
  'use strict';

  var lex = lexs[0],
    SCOPE_ORD_FACET_KEY = 'scopeOrd';

  /**
   * API Status: **Private**
   *
   * Provides some utilities to deal with search scopes
   *
   * @exports nmodule/search/rc/SearchScope
   */
  var SearchScope = {};

  /**
   * Given a SearchScope value, look up the display name to use for it.  The first
   * attempt is to lookup the Lexicon and retrieve the value there.  The fallback is
   * to use the display name provided on the SearchScope object.
   *
   * @private
   * @inner
   * @param searchScope is the the value for which to retrieve the display name
   * @param index is the index of the search scope in the search scope enum set
   * @returns {Promise} that will always resolve with the display name determined
   */
  function getScopeDisplayName(searchScope, index) {
    var defaultName = searchScope.getScopeName(),
      lexKey = searchScope.getScopeLexiconKey(),
      lexModule = searchScope.getScopeLexiconModule();
    if (!lexKey || !lexModule) {
      // If no lexicon specified on the SearchScope, use the display name
      return Promise.resolve({
        displayName: defaultName,
        index: index
      });
    } else {
      return baja.lex({
        module: lexModule
      }).then(function (lex) {
        return {
          displayName: lex.get(lexKey) || defaultName,
          index: index
        };
      })["catch"](function (ignore) {
        return {
          displayName: defaultName,
          index: index
        };
      });
    }
  }

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * Returns the scopes supported by the given SearchService as
   * an EnumSet, with default scope selections if none are specified.
   *
   * @param {baja.Component} searchService is the SearchService instance
   * @param {Array|baja.EnumSet} scopeSelections are the optional scope selections as an Array of
   * ORD strings or a {baja.EnumSet} instance for which to derive the scope selections
   * @returns {Promise} which when resolved supplies a {baja.EnumSet} EnumSet that
   * contains the scopes supported by the SearchService with ordinal selections for the
   * selected scopes
   */
  SearchScope.getScopesAsEnumSet = function (searchService, scopeSelections) {
    if (scopeSelections && baja.hasType(scopeSelections)) {
      // If scopeSelections is a baja.EnumSet instance, extract the selected
      // ORDs from it
      scopeSelections = this.getScopeOrds(scopeSelections, /*useStrings*/true);
    }
    return searchService.getSearchScopes().then(function (scopes) {
      var promises = [],
        selectedOrdinals = [],
        defaultOrdinals = [],
        ordinals = [],
        tags = [],
        facetKeys = ['emptyDisplayText'],
        facetValues = [lex.get('SearchWidget.searchScopeDefault')],
        i = 0;
      scopes.getSlots().each(function (slot) {
        var searchScope = scopes.get(slot),
          scopeOrdStr = searchScope.getScopeOrd().toString(),
          isDefault = searchScope.getIsDefault();
        ordinals.push(i);
        if (scopeSelections) {
          if (scopeSelections.indexOf(scopeOrdStr) >= 0) {
            selectedOrdinals.push(i);
          }
        } else if (isDefault) {
          selectedOrdinals.push(i);
        }
        if (isDefault) {
          defaultOrdinals.push(i);
        }
        facetKeys.push(SCOPE_ORD_FACET_KEY + i);
        facetValues.push(scopeOrdStr);

        // The following is just a placeholder in the tags array. In the subsequent promise
        // callback, it will be reassigned to a proper value by array index. We have to do
        // this because there is no guarantee on the order that the promises resolve, so this
        // ensures the tags array has the correct order as the ordinals array.
        tags.push(scopeOrdStr);
        promises.push(getScopeDisplayName(searchScope, i).then(function (obj) {
          tags[obj.index] = baja.SlotPath.escape(obj.displayName);
        }));
        i++;
      });
      if (selectedOrdinals.length === 0) {
        // If no ordinals are selected, revert to the default ordinals.  If we ever decide
        // that it is acceptable to search with an empty scope selection, then we can remove this
        selectedOrdinals = defaultOrdinals;
      }
      return Promise.all(promises).then(function () {
        return baja.EnumSet.make({
          ordinals: selectedOrdinals,
          range: baja.EnumRange.make({
            ordinals: ordinals,
            tags: tags,
            options: baja.Facets.make(facetKeys, facetValues)
          })
        });
      });
    });
  };

  /**
   * Returns an array of baja.Ords translated from the selected ordinals of the given
   * scope EnumSet (as returned by 'SearchScope.getScopesAsEnumSet()').
   *
   * @param {baja.EnumSet} scopeEnumSet is the EnumSet of scopes containing selected ordinals
   * which represent the scopes which should be translated to scope ORDs.
   * @param {Boolean} optional parameter that when true, causes the result Array to use the
   * String forms of the scope ORDs instead of the {baja.Ord} form.
   * @returns {Array} array of baja.Ords that are translated from the selected
   * ordinals in the given scopeEnumSet
   */
  SearchScope.getScopeOrds = function (scopeEnumSet, useStrings) {
    var ordinals = scopeEnumSet.getOrdinals(),
      facets = scopeEnumSet.getRange().getOptions(),
      ords = [],
      i;
    for (i = 0; i < ordinals.length; i++) {
      if (useStrings) {
        ords.push(facets.get(SCOPE_ORD_FACET_KEY + ordinals[i]));
      } else {
        ords.push(baja.Ord.make(facets.get(SCOPE_ORD_FACET_KEY + ordinals[i])));
      }
    }
    return ords;
  };

  /**
   * Returns a String containing the encoding of the selected ORDs from the given
   * scope EnumSet.
   *
   * @param {baja.EnumSet} scopeEnumSet is the EnumSet of scopes containing selected ordinals
   * which represent the scopes whose ORDs should be encoded to a String.
   * @returns {String} the encoded String representation of the selected scope ORDs
   */
  SearchScope.encodeSelectedScopes = function (scopeEnumSet) {
    var ordinals = scopeEnumSet.getOrdinals(),
      facets = scopeEnumSet.getRange().getOptions(),
      ords = [],
      i;
    for (i = 0; i < ordinals.length; i++) {
      ords.push(facets.get(SCOPE_ORD_FACET_KEY + ordinals[i]));
    }
    return JSON.stringify(ords);
  };

  /**
   * Returns an {Array} containing the ORDs as Strings from the given
   * JSON encoding, or null if the given encoding is empty.
   *
   * @param {String} encodedScopes is the JSON String encoding of a list of scope ORD strings.
   * @returns {Array} the decoded array of scope ORD Strings
   */
  SearchScope.decodeSelectedScopes = function (encodedScopes) {
    if (encodedScopes) {
      return JSON.parse(encodedScopes);
    } else {
      return null;
    }
  };
  return SearchScope;
});
