# All About BajaScript Communications

BajaScript's entire job is to provide an API to shuttle information back and forth from the station in a way that 1) accurately reflects the station's configuration and 2) is reasonably concordant with the way the Niagara Framework itself works.

There are a lot of moving parts, but all of them are fairly simple. This document will just attempt to capture them all in one place to provide an accurate picture of the way the whole thing works.

## The BOX Protocol

BOX itself is very similar to FOX, but expressed in JSON. The essence of BOX can be expressed as _messages_ and _frames_.

### BOX Messages

A BOX message can be thought of as a method call. A request from the client, in the form of a BOX message, is sent to the station; the appropriate method is called, and the results are returned to the client.

A BOX message is simply a JSON object that looks like this:

```javascript
const msg = {
  r: 'requestType',
  c: 'channelName',
  k: 'keyName',
  r: messageNumber,
  b: body
}
```

These properties represent:

- Request Type: the type of message. Requests from client to station will have `rt` as the request type.
- Channel Name: the name of the `BBoxChannel` that will handle this request on the server. These are literal slot names and are enumerated as frozen slots on `BBoxService` itself.
- Key Name: essentially, the name of the method that is being called on the `BBoxChannel`. Inside each channel's `service()` method, the key will be examined, and the appropriate method called.
- Message Number: a number that simply identifies the message (typically monotonically increasing) - this number is included in the response to this call, so that the response can be matched up with the originating request back in the client.
- Body: essentially, the argument to the method that is being called. If the method does not take an argument, this might be omitted.

The response to a BOX message has _exactly_ the same structure as the message itself - but the body attribute (`b`) is the value being returned. The request type attribute will also differ, and be one of these values:

- `rp`: a direct response to a request
- `e`: an error message
- `u`: an unsolicited message

### BOX Frames

Messages are packaged up into BOX frames for sending across the wire. A BOX frame is a JSON object that looks like this:

```javascript
const frame = {
  p: 'box',
  v: 'boxVersion',
  n: frameNumber,
  m: [ msg1, msg2 ]
}
```

These properties represent:

- Protocol: this is always `box` and marks this as a BOX frame.
- Version: marker indicating what BOX version is in use.
- Frame Number: a number that simply identifies the frame (typically monotonically increasing). The number is included in the response to this frame, so it can be matched up with the originating frame in the client.
- Messages: an array of messages or message responses in this frame.

Historically, a BOX frame corresponded to one POST or WebSocket message - basically, a BOX frame represented a network call, in which individual messages were sent to the station. The response to a BOX frame is also a BOX frame, where the messages array contains the message responses.

With muxing enabled (more on this later), there's no longer necessarily a one-to-one-relationship between BOX Frames and network calls.

## BOX Muxing

The paradigm of using one BOX frame per network call brought with it several limitations.

First, the number of messages you can send at once is limited to what can fit in a WebSocket message. If you need to send 1,000 messages, you can't just send them up in one frame: you have to manually break them down into chunks, and you can only guess at the number and size of messages that can fit into one WebSocket message.

Second, a message that overflows the size of a WebSocket message can't be sent or received at all. Imagine adding a huge Component, like a fully-populated driver network. If that network encoded to an amount of data bigger than a WebSocket message, it will fail.

Third, it requires the developer to constantly be thinking about whether individual calls can be batched together into one. Forgetting to do this can bring about significant performance problems, because individual network calls are one of the hardest things for a JACE to deal with. If you `component.add()` 1,000 times and forget to manually batch them, it's going to be slow going.

BOX muxing solves all of these problems. This section will describe the different mechanisms in play when muxing is turned on.

### BoxEnvelopes and Fragments

To solve the problem of individual BOX frames overflowing WebSocket messages, BoxEnvelopes and Fragments were introduced.

A Fragment is simply a chunk of raw data, represented in the browser as a `Uint8Array`, that can be sent out in a single network call. When WebSockets are in use, the maximum size of a Fragment is the maximum size of a WebSocket message. A Fragment doesn't know what BSON or JSON are - it's just data.

A BoxEnvelope contains a collection of Fragments that add up to one piece of data. Since each Fragment can be sent in its own WebSocket message, a really huge BOX frame can be split up and sent using multiple WebSocket messages.

In BOX muxing, a BoxEnvelope will contain one BOX frame.

### BoxEnvelopeMux and BoxEnvelopeDemux

To send a BoxEnvelope across the wire, each individual Fragment it contains will be sent as an individual network call (muxing). Those Fragments will be reassembled on the other side to reconstitute the data in the original BoxEnvelope (demuxing). In this way, a BOX frame that contains a huge number of messages, or one huge message, that would not otherwise fit in a single WebSocket message, can be successfully transmitted.

This mux/demux pattern happens on both sides, in the `baja/env/mux` folder in BajaScript world, and the `com.tridium.box.mux` package in Java world.

### BoxMessageRelay and Implicit Batching

To fix the problem of too many network calls when manual batching is forgotten, BOX muxing also introduces the idea of implicit batching.

Rather than forcing the developer to load messages into a Batch object, BajaScript now introduces a tiny delay (a "debounce") after every BOX message to allow more BOX messages to come along. If a lot of BOX messages come along very close to each other in time, they will naturally coalesce into a single network call - no manual batching required.

This behavior is implemented in BoxMessageRelay. It introduces the tiny debounce necessary for BOX messages to coalesce, and once the delay has elapsed, it will send those messages to the muxer/demuxer to send across the wire.