/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Reused WebSocket Connection.
 *
 * API Status: **Private**
 * @module baja/env/ReusedWebSocketConnection
 */
define(["bajaScript/comm", "bajaScript/baja/comm/Callback", "bajaScript/env/Connection", "bajaScript/env/WebSocketConnection"], function (baja, Callback, Connection, WebSocketConnection) {
  "use strict";

  /**
   * A connection that reuses an existing WebSocket for a connection.
   *
   * @class
   * @extends module:baja/env/Connection
   * @alias module:baja/env/ReusedWebSocketConnection
   * @param {module:baja/env/Connection~ConnectionEvents} handlers handlers
   * for connection events.
   * @param {module:baja/env/ConnectionManager} connectionManager the connection
   * manager responsible for processing BoxFrames via incoming websocket
   * messages
   * @param {WebSocket} internalConnection the existing parent connection to
   * reuse
   */
  var _ReusedWebSocketConnection = function ReusedWebSocketConnection(handlers, connectionManager, internalConnection) {
    baja.callSuper(_ReusedWebSocketConnection, this, arguments);
    if (!connectionManager) {
      throw new Error('ConnectionManager required');
    }
    if (!internalConnection) {
      throw new Error('reused WebSocket required');
    }
    this.$internalConnection = internalConnection;
  };
  baja.subclass(_ReusedWebSocketConnection, WebSocketConnection);

  /**
   * Authenticates by creating a new server session.
   * @returns {Promise}
   */
  _ReusedWebSocketConnection.prototype.doAuthenticate = function () {
    var cb = new baja.comm.Callback();
    baja.comm.makeServerSession(cb);
    return cb.promise();
  };

  /**
   * @returns {boolean} true if the reused WebSocket is secure.
   */
  _ReusedWebSocketConnection.prototype.isSecure = function () {
    return /^wss/i.test(this.$internalConnection.url);
  };

  /**
   * Piggyback onto all the event handlers from the parent connection.
   */
  _ReusedWebSocketConnection.prototype.start = function () {
    var connection = this.$internalConnection;
    var handlers = this.$handlers;
    for (var name in handlers) {
      if (handlers.hasOwnProperty(name)) {
        connection.addEventListener(name, handlers[name], false);
      }
    }

    // Check current state to see if the Socket is already open.
    if (connection.readyState === Connection.WS_OPEN) {
      handlers.open();
    }
  };
  _ReusedWebSocketConnection.prototype.close = function () {
    var connection = this.$internalConnection;
    var handlers = this.$handlers;
    for (var name in handlers) {
      if (handlers.hasOwnProperty(name)) {
        connection.removeEventListener(name, handlers[name], false);
      }
    }
    this.$handlers.close();
  };
  return _ReusedWebSocketConnection;
});
