/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * API Status: **Private**
 * @module baja/env/Connection
 */
define(["bajaScript/comm", 'bajaScript/env/mux/BoxFrameCoalescer', "bajaPromises"], function (baja, BoxFrameCoalescer, Promise) {
  "use strict";

  /**
   * The abstract base class for all full-duplex browser connections.
   *
   * @abstract
   * @class
   * @alias module:baja/env/Connection
   * @see module:baja/env/ConnectionManager
   * @param {module:baja/env/Connection~ConnectionEvents} handlers handlers
   * for connection events.
   * @param {module:baja/env/ConnectionManager} connectionManager the connection
   * manager responsible for processing BoxFrames via incoming websocket
   * messages
   */
  var Connection = function Connection(handlers, connectionManager) {
    var that = this;
    that.$connectionManager = connectionManager;
    that.$internalConnection = null;
    that.$handlers = handlers;
  };
  var WS_OPEN = Connection.WS_OPEN = 1;

  /**
   * Authenticate with the server so that the connection can be started.
   * 
   * @private
   * @returns {Promise|*}
   */
  Connection.prototype.authenticate = function () {
    var that = this;
    return Promise.resolve(that.doAuthenticate()).then(function (res) {
      that.$authenticated = true;
      return res;
    });
  };

  /**
   * Called to authenticate with the server prior to starting the connection.
   * By default, does nothing. Override this function; call authenticate().
   *
   * @private
   * @abstract
   * @returns {Promise|*}
   */
  Connection.prototype.doAuthenticate = function () {};

  /**
   * Called to start the connection. By default, does nothing.
   * 
   * @private
   * @abstract
   * @returns {Promise|*}
   */
  Connection.prototype.start = function () {};

  /**
   * Called to close the connection.
   *
   * @private
   */
  Connection.prototype.close = function () {
    var that = this;
    if (that.$internalConnection && typeof that.$internalConnection.close === "function") {
      that.$internalConnection.close();
    }
  };

  /**
   * Called to send some data to the connection.
   *
   * @private
   * @param {module:baja/env/Connection~FrameData} frameData
   */
  Connection.prototype.send = function (frameData) {
    if (!this.isOpen()) {
      throw new Error("Connection unavailable!");
    }
    if (baja.$mux.isEnabled()) {
      // queue up the frame to be sent later
      this.$getCoalescer().sendFrameData(frameData);
    } else {
      // var that = this;
      // setTimeout(function () {
      //   that.$internalConnection.send(frameData.frame.toString());
      // }, webSocketLatency);
      this.$internalConnection.send(frameData.frame.toString());
    }
  };

  /**
   * @param {module:baja/env/mux/BoxEnvelope~BoxFragment} fragment
   */
  Connection.prototype.receiveFragment = function (fragment) {
    this.$fragmentHandler(fragment);
  };

  /**
   * @private
   * @returns {module:baja/env/mux/BoxFrameCoalescer}
   */
  Connection.prototype.$getCoalescer = function () {
    var _this = this;
    var coalescer = this.$coalescer;
    if (!coalescer) {
      var mux = baja.$mux;
      var connectionManager = this.$connectionManager;
      coalescer = this.$coalescer = new BoxFrameCoalescer({
        connectionManager: connectionManager,
        maxEnvelopeSize: mux.getMaxEnvelopeSize(),
        maxMessageSize: mux.getMaxMessageSize(),
        minDelay: mux.getMinDelay(),
        //minDelay,
        maxDelay: mux.getMaxDelay(),
        sendToServer: function sendToServer(arr) {
          _this.$internalConnection.send(arr.buffer);
        },
        onIncomingFrame: function onIncomingFrame(frame) {
          connectionManager.receiveFrame(frame);
        },
        setFragmentHandler: function setFragmentHandler(handler) {
          _this.$fragmentHandler = handler;
        }
      });
    }
    return coalescer;
  };

  /**
   * @private
   * @returns {module:baja/env/Connection~InternalConnection|null} the internal
   * raw underlying connection object or null if there isn't one.
   */
  Connection.prototype.getInternalConnection = function () {
    return this.$internalConnection;
  };

  /**
   * @private
   * @return {Boolean} true if the underlying connection is currently in an open
   * state.
   */
  Connection.prototype.isOpen = function () {
    return !!(this.$authenticated && this.$internalConnection && this.$internalConnection.readyState === WS_OPEN);
  };

  /**
   * Return whether this connection is considered secure. Returns false by default; subclasses
   * should override this to declare their own checks.
   *
   * @returns {boolean} true if the connection is considered secure.
   * @since Niagara 4.14
   */
  Connection.prototype.isSecure = function () {
    return false;
  };

  /**
   * Interface for the actual implementation of the data connection of a
   * Connection. It is responsible for sending data across the wire. For
   * instance, a standard WebSocket implements this interface.
   *
   * @interface module:baja/env/Connection~InternalConnection
   */

  /**
   * @function
   * @name module:baja/env/Connection~InternalConnection#send
   * @param {string} data
   */

  /**
   * @function
   * @name module:baja/env/Connection~InternalConnection#close
   */

  /**
   * @typedef module:baja/env/Connection~FrameData
   * @property {baja.comm.BoxFrame} frame the BoxFrame to be sent
   * @property {baja.comm.Callback} cb the callback to be completed when the
   * frame has sent and returned
   * @property {function} processed a function called when the frame has sent
   * and returned - this is used by the framework to send the next frame in the
   * queue
   */

  /**
   * Interface for the events a connection may emit.
   * @interface module:baja/env/Connection~ConnectionEvents
   */

  /**
   * Fired when the Connection is opened.
   * @function
   * @name module:baja/env/Connection~ConnectionEvents#open
   * @param {Event} [e]
   */

  /**
   * Fired when the Connection receives a message from the server.
   * @function
   * @name module:baja/env/Connection~ConnectionEvents#message
   * @param {MessageEvent|{ data: string }} e
   */

  /**
   * Fired when the Connection is closed normally.
   * @function
   * @name module:baja/env/Connection~ConnectionEvents#close
   * @param {CloseEvent|*} [e]
   */

  /**
   * Fired when the Connection is closed due to an error.
   * @function
   * @name module:baja/env/Connection~ConnectionEvents#error
   * @param {Event} [e]
   */

  return Connection;
});
