/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * API Status: **Private**
 * @module baja/env/BrowserCommsManager
 */
define(["bajaBrowserEnvUtil", "bajaScript/comm", "bajaScript/env/ConnectionManager"], function (envUtil, baja, ConnectionManager) {
  "use strict";

  var BOX_URI = '/box/';

  /**
   * This is the primary public actor for sending BOX frames to the station in a
   * browser environment.
   *
   * @class
   * @alias module:baja/env/BrowserCommsManager
   */
  var BrowserCommsManager = function BrowserCommsManager() {
    var that = this;
    /** @type {module:baja/env/Connection~FrameData[]} */
    var asyncQueue = that.$asyncQueue = [];

    /**
     * Asynchronously processes the next message in the queue.
     *
     * @private
     */
    that.processNext = function () {
      // Remove the first element the queue
      asyncQueue.shift();
      if (asyncQueue.length > 0) {
        // Remove and return the first element of the async queue
        that.sendAsync(asyncQueue[0]);
      }
    };
    that.$connectionManager = new ConnectionManager(that.processNext);
  };

  /**
   * Sends the message asynchronously.
   *
   * @param {module:baja/env/Connection~FrameData} frameData The frame data to send.
   * @private
   */
  BrowserCommsManager.prototype.sendAsync = function (frameData) {
    // Flag that we're sending the data
    frameData.$sending = true;

    // If we can't send the frame through a connection then send it via HTTP
    if (!this.$connectionManager.send(frameData)) {
      postToBox(false, frameData.frame, frameData.cb, frameData.processed);
    }
  };

  /**
   * Sends the message as a beacon.
   *
   * @param {module:baja/env/Connection~FrameData} frameData The frame data to send.
   * @private
   */
  BrowserCommsManager.prototype.sendBeacon = function (frameData) {
    // Flag that we're sending the data
    frameData.$sending = true;

    // If we can't send the frame through a connection then send it via HTTP
    if (!this.$connectionManager.sendBeacon(frameData)) {
      return envUtil.sendBeacon(BOX_URI, frameData.cb, String(frameData.frame), frameData.processed);
    }
  };

  /**
   * Browser entry point for sending one BoxFrame to the station.
   *
   * @see baja.comm.BoxFrame
   *
   * @param {baja.comm.BoxFrame} boxFrame The box frame to send.
   * @param {baja.comm.Batch|baja.comm.Callback|Object} callback an object
   * containing ok/fail callbacks.
   */
  BrowserCommsManager.prototype.sendBoxFrame = function (boxFrame, callback) {
    var frameData,
      processNext = this.processNext,
      asyncQueue = this.$asyncQueue,
      syncFlag = boxFrame.$sync,
      isBeacon = syncFlag === 'beacon',
      isAsync = !syncFlag || isBeacon;
    if (isAsync) {
      frameData = {
        frame: boxFrame,
        cb: callback
      };

      // if muxing is enabled, the WebSocketConnection itself will handle the
      // async queueing.
      if (boxFrame.$queue && !baja.$mux.isEnabled()) {
        frameData.processed = processNext;

        // Push the frame onto our async queue (adds to end of array).
        asyncQueue.push(frameData);

        // If there are no outstanding frames then simply process the queue (otherwise
        // rely on comms to make processNext request).
        if (asyncQueue.length === 1) {
          this[isBeacon ? 'sendBeacon' : 'sendAsync'](asyncQueue[0]);
        }
      } else {
        this[isBeacon ? 'sendBeacon' : 'sendAsync'](frameData);
      }
    } else {
      // For synchronous network calls, always make an HTTP call.
      postToBox(syncFlag, boxFrame, callback, processNext);
    }
  };
  function postToBox(syncFlag, boxFrame, callback, processNext) {
    var data = String(boxFrame);
    envUtil.sendHttp("POST", BOX_URI, !syncFlag, callback, data, processNext);
  }

  /**
   * Returns the connection manager.
   *
   * @private
   *
   * @returns {module:baja/env/ConnectionManager} The connection manager.
   */
  BrowserCommsManager.prototype.getConnectionManager = function () {
    return this.$connectionManager;
  };
  return BrowserCommsManager;
});
