/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/tag/Relation
 */
define(["bajaScript/sys", "bajaScript/baja/tag/Id", "bajaScript/baja/tag/TagSet", "bajaPromises"], function (baja, Id, TagSet, Promise) {
  "use strict";

  /**
   * A relation is a taggable Id and Entity pair.
   *
   * @class
   * @alias module:baja/tag/Relation
   *
   * @param {module:baja/tag/Id|String} id The Id of the relation. A
   * qname string can also be passed in.
   * @param {String|baja.Ord} entityOrd The target entity ORD for the relation.
   * @param {Object} [tags] The tags for the relation or a function
   * that returns a promise that resolves to a tags object.
   * @param {Boolean} inbound whether this relation is inbound (inbound = false implied outbound)
   */
  var Relation = function Relation(id, entityOrd, tags, inbound) {
    var that = this;
    that.$id = typeof id === "string" ? new Id(id) : id;
    that.$entityOrd = typeof entityOrd === "string" ? baja.Ord.make(entityOrd) : entityOrd;
    that.$tags = tags;
    that.$inbound = inbound;
  };

  /**
   * @returns {module:baja/tag/Id} The relation's Id
   * or null if there isn't one.
   */
  Relation.prototype.getId = function () {
    return this.$id;
  };

  /**
   * @returns Returns a Promise that resolves to a Relation's Entity.
   */
  Relation.prototype.toEndpoint = function () {
    return this.$entityOrd.get();
  };

  /**
   * @returns Returns the ORD to the Relation's Entity.
   */
  Relation.prototype.getEndpointOrd = function () {
    return this.$entityOrd;
  };

  /**
   * @returns {Promise} A promise that resolves to a Tags object.
   */
  Relation.prototype.tags = function () {
    var that = this;
    that.$tags = that.$tags || new TagSet();
    return that.$tagsProm = that.$tagsProm || Promise.resolve(that.$tags);
  };

  /**
   * Test for outbound relation relative to entity where the relation is stored
   * @since Niagara 4.15
   *
   * @returns {Boolean} true if relation is an outbound relation
   */
  Relation.prototype.isOutbound = function () {
    return !this.$inbound;
  };

  /**
   * Test for inbound relation relative to entity where the relation is stored
   * @since Niagara 4.15
   *
   * @returns {Boolean}  true if relation is an inbound relation
   */
  Relation.prototype.isInbound = function () {
    return !!this.$inbound;
  };

  /**
   * Test for inbound relation relative to entity where the relation is stored
   * @since Niagara 4.15
   *
   * @returns {Boolean}  true if relation is an inbound relation
   */
  Relation.prototype.getInbound = function () {
    return this.isInbound();
  };

  /**
   * @param o The object used for comparison.
   * @returns {Boolean} Returns true if the object is equal to this relation.
   */
  Relation.prototype.equals = function (o) {
    return !!(o && o instanceof Relation && this.$id.equals(o.$id) && this.$entity.equals(o.$entity));
  };
  return Relation;
});
