/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.LocalScheme}.
 * @module baja/ord/LocalScheme
 */
define(["bajaScript/sys", "bajaScript/baja/ord/OrdQuery", "bajaScript/baja/ord/OrdScheme", "bajaScript/baja/ord/OrdTarget", "bajaScript/baja/ord/ordUtil"], function (baja, OrdQuery, OrdScheme, OrdTarget, ordUtil) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper,
    trimToStart = ordUtil.trimToStart;

  /**
   * Local Host ORD Scheme.
   * 
   * This scheme resolves to the local host. The local host represents the 
   * Station BajaScript is directly connected to.
   *
   * @class
   * @alias baja.LocalScheme
   * @extends baja.OrdScheme
   * @private
   */
  var LocalScheme = function LocalScheme() {
    callSuper(LocalScheme, this, arguments);
  };
  subclass(LocalScheme, OrdScheme);

  /**
   * Default Local ORD Scheme instance.
   * @private
   * @type {baja.LocalScheme}
   */
  LocalScheme.DEFAULT = new LocalScheme();

  /**
   * Called when an ORD is resolved.
   *
   * @private
   *
   * @see baja.OrdScheme#resolve
   *
   * @param {module:baja/ord/OrdTarget} target  the current ORD Target.
   * @param {baja.OrdQuery} query  the ORD Query used in resolving the ORD.
   * @param {module:baja/ord/OrdQueryListCursor} cursor  the ORD Query List 
   * cursor used for helping to asynchronously resolve the ORD.
   * @param {Object} options  options used for resolving an ORD.
   */
  LocalScheme.prototype.resolve = function (target, query, cursor, options) {
    var newTarget = new OrdTarget(target);
    newTarget.object = baja.nav.localhost;
    cursor.resolveNext(newTarget, options);
  };

  /**
   * Return an ORD Query for the scheme.
   *
   * @returns {baja.OrdQuery}
   */
  LocalScheme.prototype.parse = function (schemeName, body) {
    return new OrdQuery({
      scheme: this,
      schemeName: schemeName,
      body: body,
      isHost: true,
      isSession: true,
      normalize: trimToStart
    });
  };
  return LocalScheme;
});
