/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.Object}.
 * @module baja/obj/Object
 */
define(['bajaScript/sys', 'bajaPromises'], function (baja, Promise) {
  'use strict';

  var subclass = baja.subclass,
    callSuper = baja.callSuper,
    BaseBajaObj = baja.BaseBajaObj;

  /**
   * Represents `baja:Object` in BajaScript.
   * 
   * Since this Constructor represents an abstract class, it should never
   * be directly used to create a new Object.
   *
   * @class
   * @alias baja.Object
   * @extends baja.BaseBajaObj
   */
  var BObject = function BObject() {
    callSuper(BObject, this, arguments);
  };
  subclass(BObject, BaseBajaObj);

  /**
   * Equivalence test.
   * 
   * `equivalent()` is used to compare if two Objects have equivalent
   * state, but might not want to return true for equals since it
   * it has implied semantics for many operations.  The default
   * implementation returns the result of {@link baja.Object#equals}.
   * 
   * @param obj
   * @returns {Boolean}
   */
  BObject.prototype.equivalent = function (obj) {
    // By default, just perform equals
    return this.equals(obj);
  };

  /**
   * Return the Object's Icon.
   * 
   * @returns {baja.Icon}
   */
  BObject.prototype.getIcon = function () {
    return this.getType().getIcon();
  };

  /**
   * Returns a promise that resolves to the agent list for this Object.
   *
   * @see baja.registry.getAgents
   *
   * @param  {Array<String>} [is] An optional array of filters to add to the
   * agent query.
   * @param  {baja.comm.Batch} [batch] An optional object used to batch network
   * calls together.
   * @returns {Promise} A promise that will resolve with the Agent Info.
   */
  BObject.prototype.getAgents = function (is, batch) {
    return baja.registry.getAgents("type:" + this.getType().toString(), is, batch);
  };

  /**
   * Gets the friendly type display name for this object.
   *
   *
   * @see baja.Type#getDisplayName
   *
   * @since Niagara 4.10
   *
   * @param {Object} [cx] a context to be passed down to Type
   * @returns {Promise.<string>|string} If no context is provided, the type
   * display name is returned synchronously as a string. If context provided,
   * the type display name is resolved via a promise as a string. 
   */
  BObject.prototype.getTypeDisplayName = function (cx) {
    return this.getType().getDisplayName(cx);
  };

  /**
   * Returns the String representation of this Object.
   *
   * When implementing `toString()` on one of your own Objects, it should
   * adhere to the following contract.
   *
   * - When called with no arguments, it must return a string directly.
   * - It can also be called with an object literal. In this case, it *may*
   *   return a Promise to be resolved with a string, or it may return a string
   *   directly as normal. This case is sort of analogous to
   *   `javax.baja.sys.Localizable#toString(Context)`.
   *
   * Therefore, when calling `toString()` on an Object of unknown Type using an
   * object literal as an argument, it's important to wrap the call in
   * `Promise.resolve()`.
   *
   * @function toString
   * @memberOf baja.Object.prototype
   *
   * @param {Object} [cx] optional context information to be used when
   * formatting the string
   *
   * @returns {String|Promise.<String>} a string (if no context passed), or
   * either a string or a Promise (if context passed).
   *
   * @example
   * <caption>When no context is passed, toString() must always return a string
   * directly.</caption>
   *
   * var bool = comp.get('boolean');
   * baja.outln(bool.toString()); // 'false'
   *
   * @example
   * <caption>When a context is passed, some Objects may return a Promise.
   * </caption>
   *
   * var bool = comp.get('boolean');
   * bool.toString({ trueText: 'Yes', falseText: 'No' })
   *   .then(function (str) {
   *     baja.outln(str); // 'No'
   *   });
   *
   * @example
   * <caption>If you don't know the type of the Object, use Promise.resolve()
   * for safety.</caption>
   *
   * var displayFacets = comp.get('displayFacets'),
   *     value = comp.get('value');
   * Promise.resolve(value.toString(displayFacets.toObject()))
   *   .then(function (str) {
   *     baja.outln(str); // formatted according to displayFacets
   *   });
   */

  /**
   * Get the type of this instance.
   *
   * @function
   * @name getType
   * @memberOf baja.Object#
   * @returns {Type}
   */

  return BObject;
});
