/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Aaron Laniewicz
 */

/**
 * Defines {@link baja.DaysOfWeekBits}.
 * @module baja/obj/DaysOfWeekBits
 */
define(['bajaScript/baja/obj/objUtil', 'bajaScript/baja/obj/Simple', 'bajaScript/sys', 'bajaPromises', 'lex!'], function (objUtil, Simple, baja, Promise, lexjs) {
  'use strict';

  var callSuper = baja.callSuper,
    strictArg = baja.strictArg,
    subclass = baja.subclass,
    cacheDecode = objUtil.cacheDecode,
    cacheEncode = objUtil.cacheEncode,
    allBits = 127; // 0b1111111

  /**
   * Represents a `baja:DaysOfWeekBits` in BajaScript.
   *
   * When creating a `Simple`, always use the `make()` method instead of
   * creating a new Object.
   *
   * @class
   * @alias baja.DaysOfWeeksBits
   * @extends baja.Simple
   */
  var DaysOfWeekBits = function DaysOfWeekBits(bits) {
    callSuper(DaysOfWeekBits, this, arguments);
    this.$bits = strictArg(bits, Number);
  };
  subclass(DaysOfWeekBits, Simple);

  /**
   * Make a `DaysOfWeekBits`.
   *
   * @param {Number} bits - the bits value.
   *
   * @returns {baja.DaysOfWeekBits}
   */
  DaysOfWeekBits.make = function (bits) {
    switch (bits) {
      case 0:
        return DaysOfWeekBits.EMPTY;
      case allBits:
        return DaysOfWeekBits.DEFAULT;
      default:
        return new DaysOfWeekBits(bits);
    }
  };

  /**
   * Make a `DaysOfWeekBits`.
   *
   * @see module:nmodule/bajaScript/rc/baja/obj/DaysOfWeekBits.make
   */
  DaysOfWeekBits.prototype.make = function () {
    return DaysOfWeekBits.make.apply(DaysOfWeekBits, arguments);
  };

  /**
   * Decode a `DaysOfWeekBits` from a `String`.
   *
   * @param {String} str - an encoded `DaysOfWeekBits`.
   *
   * @returns {baja.DaysOfWeekBits}
   */
  DaysOfWeekBits.prototype.decodeFromString = cacheDecode(function (str) {
    return DaysOfWeekBits.make(parseInt(str, 16));
  });

  /**
   * Encode the `DaysOfWeekBits` (itself) to a `String`.
   *
   * @returns {String}
   */
  DaysOfWeekBits.prototype.encodeToString = cacheEncode(function () {
    return this.$bits.toString(16);
  });

  /**
   * `DaysOfWeekBits` Empty Value.
   * @type {baja.DaysOfWeekBits}
   */
  DaysOfWeekBits.EMPTY = new DaysOfWeekBits(0);

  /**
   * Default `DaysOfWeekBits` instance.
   * @type {baja.DaysOfWeekBits}
   */
  DaysOfWeekBits.DEFAULT = new DaysOfWeekBits(allBits);

  /**
   * Return the `Number` encapsulated in the `DaysOfWeekBits` (itself).
   *
   * @returns {Number}
   */
  DaysOfWeekBits.prototype.valueOf = function () {
    return this.$bits;
  };

  /**
   * Return whether or not the `baja:Weekday` enum is active.
   *
   * @param {Number|baja.FrozenEnum} weekday - the ordinal of the weekday, or a
   * `baja:Weekday` enum value.
   *
   * @returns {Boolean}
   */
  DaysOfWeekBits.prototype.includes = function (weekday) {
    // When a baja:Weekday is passed in, this automatically uses .valueOf() to
    // get the weekday ordinal.
    return !!(this.$bits & 0x1 << weekday);
  };

  /**
   * Return the `String` representation of the `DaysOfWeekBits` (itself).
   *
   * @returns {String}
   */
  DaysOfWeekBits.prototype.toString = function (cx) {
    var that = this,
      days = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'],
      ordinals = [0, 1, 2, 3, 4, 5, 6];

    // Filter out any inactive days.
    ordinals = ordinals.filter(function (index) {
      return that.includes(index);
    });
    if (cx) {
      return Promise.all([lexjs.module('baja'), baja.importTypes(['baja:Weekday'])]).then(function (results) {
        var lex = results[0],
          en = baja.$("baja:Weekday");
        return '{' + ordinals.map(function (index) {
          return lex.get(en.get(index).getTag() + '.short');
        }).join(' ') + '}';
      });
    } else {
      return '{' + ordinals.map(function (index) {
        return days[index];
      }).join(' ') + '}';
    }
  };
  return DaysOfWeekBits;
});
