/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.file.File}.
 * @module baja/file/File
 */
define(["bajaScript/bson", "bajaScript/baja/obj/Simple"], function (baja, Simple) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper;

  /**
   * Represents a `baja:File` in BajaScript.
   *
   * @class
   * @alias baja.file.File
   */
  var File = function File(fileData) {
    callSuper(File, this, arguments);
    this.$fileData = fileData;
  };
  subclass(File, Simple);

  /**
   * Default `File` instance.
   * @type {baja.file.File}
   */
  File.DEFAULT = new File({});

  /**
   * Make a `File`.
   * 
   * @private
   *
   * @param {Object} fileData
   * @returns {baja.file.File} the File.
   */
  File.make = function (fileData) {
    return fileData ? new File(fileData) : File.DEFAULT;
  };

  /**
   * Make a `File`.
   * 
   * @private
   *
   * @param {Object} fileData
   * @returns {baja.file.File} the File
   */
  File.prototype.make = function (fileData) {
    return File.make(fileData);
  };

  /**
   * Decode a `File` from a `String`.
   * 
   * @private
   * 
   * @param {String} str
   * @returns {baja.file.File}
   */
  File.prototype.decodeFromString = function (str) {
    return File.make(JSON.parse(str));
  };

  /**
   * Encode the `File` to a `String`.
   *
   * @private
   *
   * @returns {String}
   */
  File.prototype.encodeToString = function () {
    return JSON.stringify(this.$fileData);
  };

  /**
   * Return the File's Nav ORD.
   * 
   * @returns {baja.Ord}
   */
  File.prototype.getNavOrd = function () {
    return baja.Ord.make(this.$fileData.navOrd);
  };

  /**
   * Return the ORD in session.
   * 
   * @returns {baja.Ord}
   */
  File.prototype.getOrdInSession = function () {
    return baja.Ord.make(this.$fileData.ordInSession);
  };

  /**
   * Returns the file name of the file.
   * 
   * @returns {String}
   */
  File.prototype.getFileName = function () {
    return this.$fileData.fileName;
  };

  /**
   * Returns the extension of the file.
   * 
   * @returns {String}
   */
  File.prototype.getExtension = function () {
    return this.$fileData.ext;
  };

  /**
   * Return true if the file is a directory.
   * 
   * @returns {Boolean}
   */
  File.prototype.isDirectory = function () {
    return !!this.$fileData.dir;
  };

  /**
   * Return the file's MIME Type.
   * 
   * @returns {String}
   */
  File.prototype.getMimeType = function () {
    return this.$fileData.mime;
  };

  /**
   * Return the file's size.
   * 
   * @returns {Number}
   */
  File.prototype.getSize = function () {
    return this.$fileData.size;
  };

  /**
   * Return the file's last modified time.
   * 
   * @returns {baja.AbsTime}
   */
  File.prototype.getLastModified = function () {
    return baja.AbsTime.DEFAULT.decodeFromString(this.$fileData.lastModified);
  };

  /**
   * Return true if the file is readonly.
   * 
   * @returns {Boolean}
   */
  File.prototype.isReadonly = function () {
    return !!this.$fileData.readOnly;
  };

  /**
   * Return the file's URI to read file data via an HTTP GET. If 
   * the file can't be read then a blank string will be returned.
   *
   * @returns {String}
   */
  File.prototype.getReadUri = function () {
    return this.$fileData.read || "";
  };

  /**
   * Return the file's URI to write file data via an HTTP PUT. If 
   * the file can't be written too then a blank string will be returned.
   *
   * @returns {String}
   */
  File.prototype.getWriteUri = function () {
    return this.$fileData.write || "";
  };
  return File;
});
