/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/boxcs/SetOp
 * @private
 */
define(["bajaScript/bson", "bajaScript/baja/boxcs/SyncOp", "bajaPromises"], function (baja, SyncOp, bajaPromises) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper;
  var serverDecodeContext = baja.$serverDecodeContext;

  /**
   * @class Set SyncOp
   *
   * @name SetOp
   * @extends SyncOp
   * @inner
   * @private
   *
   * @param {baja.Component} comp the Component the set is happening upon.
   * @param {Array} propPath an array of Property names for the set.
   * @param val the value being used in the set.
   */
  function SetOp(comp, propPath, val) {
    callSuper(SetOp, this, arguments);

    // Encode argument to a data structure
    this.$arg = {
      nm: this.getId(),
      h: comp.getHandle(),
      n: propPath.reverse().join("/"),
      b: baja.bson.encodeValue(val) // Encode the value to BSON
    };
  }
  subclass(SetOp, SyncOp);
  SetOp.id = "s";

  /**
   * Decode and commit the SyncOp.
   *
   * @private
   *
   * @param {baja.Component} comp the Component the set is happening on.
   * @param {object} sp the syncOp data structure to be decoded.
   * @returns {Promise|*} if set is performed, return promise to be resolved after it is complete
   */
  SetOp.decodeAndCommit = function (comp, sp) {
    if (comp) {
      // Decode the value and call 'set'
      var slotPath = sp.n,
        displayName = sp.dn,
        timestamp = sp.ts,
        bson = sp.b;
      var display = bson.d;
      var names = slotPath.split('/');
      var target = comp;
      var slot = null;
      for (var i = 0; i < names.length; ++i) {
        if (slot !== null) {
          target = target.get(slot);
          if (!target) {
            return bajaPromises.reject(new Error('SetOp received invalid slot path: ' + slotPath));
          }
        }
        slot = names[i];
      }
      return baja.bson.decodeAsync(bson, serverDecodeContext).then(function (value) {
        // Set the desired target
        target.set({
          slot: slot,
          value: value,
          cx: {
            commit: true,
            serverDecode: true,
            syncStructVals: true,
            displayName: displayName,
            display: display,
            timestamp: timestamp
          }
        });
      });
    }
  };
  return SetOp;
});
