/**
 * @copyright 2024 Tridium, Inc. All Rights Reserved.
 */

define(['nmodule/gx/rc/baja/Point'], function (Point) {
  'use strict';

  /**
   * API Status: **Private**
   * @since Niagara 4.15
   * @exports module:nmodule/uxBuilder/rc/ux/wysiwyg/artisans/path/segmentArtisanUtil
   */
  var exports = {};

  /**
   * Returns a new point determining how much the current point (x, y) is 
   * offset from a reference point (previous point). Drawing a Path involves a 
   * starting point, adding a bunch of geometry along the point until the end point.
   * "nextPoint" helps determine these points along the path, to paint handles that are used 
   * as guide points to edit a Path.
   * All the points and x, y values in this method are from the original Path geometry which has 
   * not been translated to any other coordinate system.  
   * @param {module:nmodule/gx/rc/baja/Point} refPt a point on the original non-translated Path 
   * @param {boolean} isAbsolute 
   * @param {number} x the x value of a point on the original non-translated Path 
   * @param {number} y the y value of a point on the original non-translated Path
   * @returns {module:nmodule/gx/rc/baja/Point}
   */
  exports.nextPoint = function (refPt, isAbsolute, x, y) {
    return isAbsolute ? new Point(x, y) : new Point(refPt.x() + x, refPt.y() + y);
  };

  /**
   * This utility works in tandem with {@link module:nmodule/uxBuilder/rc/ux/wysiwyg/artisans/path/segmentArtisanUtil#nextPoint}
   * While nextPoint returns a new point determining how much it is offset by a reference point, this method returns a mirror image
   * of this nextPoint with regards to the same reference point. 
   * The "nextPoint" and their "mirror" is then used to draw a straight line which acts as a guideline to edit the Path.
   * @param {module:nmodule/gx/rc/baja/Point} a 
   * @param {module:nmodule/gx/rc/baja/Point} b 
   * @returns {module:nmodule/gx/rc/baja/Point}
   */
  exports.mirror = function (a, b) {
    var dx = b.x() - a.x();
    var dy = b.y() - a.y();
    return new Point(a.x() - dx, a.y() - dy);
  };
  return exports;
});
