/**
 * @copyright 2024 Tridium, Inc. All Rights Reserved.
 */

define(['baja!', 'baja!bajaui:Layout', 'bajaux/Widget', 'jquery', 'Promise', 'underscore', 'nmodule/bajaui/rc/ux/CanvasPane', 'nmodule/bajaui/rc/ux/LabelPane', 'nmodule/uxBuilder/rc/util/uxBuilderUtils', 'nmodule/webEditors/rc/util/htmlUtils', 'nmodule/bajaui/rc/ux/shape/Shape', 'nmodule/uxBuilder/rc/ux/model/UxModelTreeNode'], function (baja, types, Widget, $, Promise, _, CanvasPane, LabelPane, uxBuilderUtils, htmlUtils, Shape, UxModelTreeNode) {
  'use strict';

  var uniq = _.uniq;
  var getCssTransformScaling = htmlUtils.getCssTransformScaling;
  var getOriginatingNode = uxBuilderUtils.getOriginatingNode;
  var DRAG_THRESHOLD = 10;
  function round(num) {
    var p = Math.pow(10, 2);
    return Math.round(num * p) / p;
  }
  function distBetween(a, b) {
    return Math.sqrt(Math.pow(a.x - b.x, 2) + Math.pow(a.y - b.y, 2));
  }

  /**
   * @see GeomUtil.java#getFoot
  * Given a line L1 and a point P1, the 'foot' of 
  * point P1 in L1 is the point P2 that occurs at 
  * the intersection of lines L1 and L2, where L2 
  * is the line perpendicular to L1 and passing 
  * through P1.
  */
  function getFoot(point, pt1, pt2) {
    if (pt1.x === pt2.x && pt1.y === pt2.y) {
      // zero length segment
      throw new Error('Cannot determine foot for a zero length segment.');
    } else if (pt1.y === pt2.y) {
      // Horizontal segment
      return {
        x: point.x,
        y: pt1.y
      };
    } else if (pt1.x === pt2.x) {
      // Vertical segment
      return {
        x: pt1.x,
        y: point.y
      };
    } else {
      // regular 'tilted' segment
      // Derive the line equation y = mx + b for the line
      // that passes through points (x1,y1) and (x2, y2). 
      //
      // We can get the slope 'm' like so:
      //
      //         y2 - y1
      //     m = -------
      //         x2 - y1
      // 
      // And once we know m, then we get b like so:
      //
      //     b = y1 - m*x1
      //

      var m = (pt1.y - pt2.y) / (pt1.x - pt2.x);
      var b = pt1.y - m * pt1.x;

      //
      // The equations for coordinates of the 'foot' of 
      // point (xa, ya) in the line y = mx + b are:
      //
      //      m*ya + xa - m*b
      // fX = ---------------
      //         (m*m + 1)
      //
      //      m*m*ya + m*xa + b
      // fY = -----------------
      //         (m*m + 1)
      //

      var fX = (m * point.y + point.x - m * b) / (m * m + 1);
      var fY = (m * m * point.y + m * point.x + b) / (m * m + 1);
      return {
        x: fX,
        y: fY
      };
    }
  }

  /**
   * @see GeomUtil.java#dist
   * Calculate the distance of a point to a segment (made of two points pt1 and pt2) 
   * @param {{ x, y }} point 
   * @param {{ x, y }} pt1 
   * @param {{ x, y }} pt2 
   * @returns {number}
   */
  function dist(point, pt1, pt2) {
    var dx = pt1.x - pt2.x,
      dy = pt1.y - pt2.y;
    var l2 = Math.pow(dx, 2) + Math.pow(dy, 2); //length squared
    var distance;
    if (l2 === 0) {
      // the segment is a point, so just return the distance to the point
      distance = distBetween(point, pt1);
    } else if (pt1.y === pt2.y) {
      // Horizontal segment
      var minX = Math.min(pt1.x, pt2.x),
        maxX = Math.max(pt1.x, pt2.x);
      // if point.x is outside the range, then
      // return the distance to the nearest endpoint
      if (point.x < minX || point.x > maxX) {
        distance = Math.min(distBetween(point, pt1), distBetween(point, pt2));
      } else {
        distance = Math.abs(point.y - pt1.y);
      }
    } else if (pt1.x === pt2.x) {
      // vertical segment
      var minY = Math.min(pt1.y, pt2.y),
        maxY = Math.max(pt1.y, pt2.y);
      // if point.y is outside the range, then
      // return the distance to the nearest endpoint
      if (point.y < minY || point.y > maxY) {
        distance = Math.min(distBetween(point, pt1), distBetween(point, pt2));
      } else {
        distance = Math.abs(point.x - pt1.x);
      }
    } else {
      // regular 'tilted' segment
      // get the foot
      var foot = getFoot(point, pt1, pt2);

      // If the foot intersects our segment, return the distance  
      // from the foot to the point.

      var _minX = Math.min(pt1.x, pt2.x);
      var _maxX = Math.max(pt1.x, pt2.x);
      if (foot && foot.x >= _minX && foot.x <= _maxX) {
        distance = distBetween(point, foot);
      } else {
        // else return the distance to the nearest endpoint            
        distance = Math.min(distBetween(point, pt1), distBetween(point, pt2));
      }
    }
    return distance;
  }
  /**
   * API Status: **Private**
   * @since Niagara 4.15
   * @exports nmodule/uxBuilder/rc/util/wysiwygUtils
   */
  var exports = {};

  /**
   * Takes the given event and extracts the location info for mouse and touch
   * events.
   *
   * @param {JQuery.Event} event
   * @returns {Object}
   */
  exports.normalizeEventInfo = function (event) {
    if (event.touches && event.touches.length === 1) {
      return event.touches[0];
    }
    // `touchend` populates changedTouches and not touches
    if (event.changedTouches && event.changedTouches.length === 1) {
      return event.changedTouches[0];
    }
    return event;
  };

  /**
   * Getting page coordinates behaves a bit differently between mouse and touch.
   * Rather than referencing `event.offsetX` etc. in an event handler, this
   * normalizes the behaviors to return the right values regardless of
   * environment.
   *
   * @param {JQuery.Event} event
   * @param {Element|JQuery} offsetParent coordinates will be calculated
   * relative to the offset parent.
   * @param {boolean} [canScale]
   * @param {number} [zoom=1] zoom level
   * @returns {{ x: number, y: number }} 
   */
  exports.getNormalizedCoords = function (event, offsetParent, canScale) {
    var zoom = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 1;
    if (!offsetParent) {
      throw new Error('offsetParent required');
    }
    var actualEvent = exports.normalizeEventInfo(event);
    var pageX = actualEvent.pageX,
      pageY = actualEvent.pageY;
    var scaling = canScale ? getCssTransformScaling(offsetParent) : {
      x: 1,
      y: 1
    };
    var _ref = $(offsetParent).offset() || {},
      _ref$left = _ref.left,
      left = _ref$left === void 0 ? 0 : _ref$left,
      _ref$top = _ref.top,
      top = _ref$top === void 0 ? 0 : _ref$top;
    return {
      x: round((pageX - left) / zoom / scaling.x),
      y: round((pageY - top) / zoom / scaling.y)
    };
  };

  /**
   * Given a widget return its root px content widget. 
   * @param {module:bajaux/Widget} widget 
   * @returns {module:bajaux/Widget|null}
   */
  exports.getRootWidget = function (widget) {
    var jq = widget.jq(),
      rootJq = jq && jq.closest('.ux-root') || [],
      topmostRootJq = jq && jq.parents().closest('.ux-root.-t-PxWidget-content').last() || [];
    if (topmostRootJq.length > 0) {
      // If the top most parent is the px content root then return that widget
      return Widget["in"](topmostRootJq);
    } else if (rootJq.length > 0) {
      // If widget is the root with no parent return self
      return Widget["in"](rootJq);
    }
    return null;
  };

  /**
   * 
   * @param {module:bajaux/Widget} widget 
   * @returns {boolean}
   */
  exports.isShape = function (widget) {
    return widget.jq().hasClass('ux-Shape');
  };

  /**
   * @param {JQuery.Event} e
   * @returns {boolean} true if this is a multitouch event
   */
  exports.isMultitouch = function (e) {
    var touches = e.touches;
    return !!(touches && touches.length > 1);
  };

  /**
   * @param {JQuery.Event} event latest mousemove event
   * @param {JQuery.Event} originatingEvent event that started the op
   * @returns {boolean} true if the event is within 10px of the originating op -
   * indicating that the user hasn't really dragged like they mean it
   */
  exports.isWobble = function (event, originatingEvent) {
    if (!originatingEvent) {
      return true;
    }
    var _exports$normalizeEve = exports.normalizeEventInfo(originatingEvent),
      origX = _exports$normalizeEve.pageX,
      origY = _exports$normalizeEve.pageY;
    var _exports$normalizeEve2 = exports.normalizeEventInfo(event),
      pageX = _exports$normalizeEve2.pageX,
      pageY = _exports$normalizeEve2.pageY;
    var deltaX = pageX - origX;
    var deltaY = pageY - origY;
    return !(Math.abs(deltaX) > DRAG_THRESHOLD || Math.abs(deltaY) > DRAG_THRESHOLD);
  };

  /**
   * A widget can be moved or resized only if it is the child of a CanvasPane
   * 
   * @param {module:bajaux/Widget} widget 
   * @returns {boolean}
   */
  exports.isCanvasPaneChild = function (widget) {
    var widgetJq = widget.jq();
    return widgetJq && widgetJq.hasClass('-t-CanvasPane-child');
  };

  /**
   * // TODO When additional options (with useSnap) is implemented
   * @returns {boolean}
   */
  exports.getUseSnap = function () {
    return false;
  };

  /**
   * Converts a DOMRect to the data structure used by Layouts, rubber bands, and almost everywhere
   * in UxBuilder.
   * @param {DOMRect} [domRect]
   * @returns {module:nmodule/bajaui/rc/baja/Layout~Rectangle}
   */
  exports.toLayoutRectangle = function (domRect) {
    var _ref2 = domRect || {},
      _ref2$x = _ref2.x,
      x = _ref2$x === void 0 ? 0 : _ref2$x,
      _ref2$y = _ref2.y,
      y = _ref2$y === void 0 ? 0 : _ref2$y,
      _ref2$width = _ref2.width,
      width = _ref2$width === void 0 ? 0 : _ref2$width,
      _ref2$height = _ref2.height,
      height = _ref2$height === void 0 ? 0 : _ref2$height;
    return {
      x: x,
      y: y,
      w: width,
      h: height
    };
  };

  /**
   * @param {module:nmodule/bajaui/rc/baja/Layout~Rectangle} rect
   * @returns {module:nmodule/bajaui/rc/baja/Layout~Rectangle} input rectangle with attributes
   * rounded to integers
   */
  exports.roundLayoutRectangle = function (rect) {
    var x = rect.x,
      y = rect.y,
      w = rect.w,
      h = rect.h;
    return {
      x: Math.round(x),
      y: Math.round(y),
      w: Math.round(w),
      h: Math.round(h)
    };
  };

  /**
   * @param {module:nmodule/bajaui/rc/baja/Layout~Rectangle} ...rectangles
   * @returns {module:nmodule/bajaui/rc/baja/Layout~Rectangle|*} the rectangle that intersects all
   * the given rectangles, or falsy if no intersection is possible
   */
  exports.intersectionOfRectangles = function () {
    for (var _len = arguments.length, rectangles = new Array(_len), _key = 0; _key < _len; _key++) {
      rectangles[_key] = arguments[_key];
    }
    return rectangles.reduce(function (a, b) {
      if (!a) {
        return;
      }
      if (!b) {
        return a;
      }
      var _exports$calculateSid = exports.calculateSides(a),
        t1 = _exports$calculateSid.top,
        r1 = _exports$calculateSid.right,
        b1 = _exports$calculateSid.bottom,
        l1 = _exports$calculateSid.left;
      var _exports$calculateSid2 = exports.calculateSides(b),
        t2 = _exports$calculateSid2.top,
        r2 = _exports$calculateSid2.right,
        b2 = _exports$calculateSid2.bottom,
        l2 = _exports$calculateSid2.left;
      if (t1 >= b2 || t2 >= b1) {
        return;
      }
      if (l1 >= r2 || l2 >= r1) {
        return;
      }
      var x = Math.max(l1, l2);
      var y = Math.max(t1, t2);
      var w = Math.min(r1, r2) - x;
      var h = Math.min(b1, b2) - y;
      return {
        x: x,
        y: y,
        w: w,
        h: h
      };
    }, rectangles[0]);
  };

  /**
   * @param {module:nmodule/bajaui/rc/baja/Layout~Rectangle} rectangle
   * @returns {{top: number, left: number, bottom: number, right: number}}
   */
  exports.calculateSides = function (rectangle) {
    var x = rectangle.x,
      y = rectangle.y,
      w = rectangle.w,
      h = rectangle.h;
    var top, right, bottom, left;
    if (w < 0) {
      left = x + w;
      right = x;
    } else {
      left = x;
      right = x + w;
    }
    if (h < 0) {
      top = y + h;
      bottom = y;
    } else {
      top = y;
      bottom = y + h;
    }
    return {
      x: x,
      y: y,
      w: w,
      h: h,
      top: top,
      right: right,
      bottom: bottom,
      left: left
    };
  };

  /**
   * @param {module:bajaux/Widget} widget widget currently rendered in the PxWidget
   * @returns {module:bajaux/Widget|*} the parent widget, or falsy if no parent could be found
   */
  exports.getParentLastBuiltWidget = function (widget) {
    var node = widget && getOriginatingNode(widget);
    var parentNode = node && node.getParent();
    // special case: TabbedPane does not paint its configured LabelPanes, so go up one more level.
    if (parentNode && parentNode.represents(LabelPane)) {
      parentNode = parentNode.getParent();
    }
    return parentNode && parentNode.getLastBuiltWidget();
  };

  /**
   * @param {function} type the constructor of a Widget type
   * @param {module:bajaux/Widget} widget a widget instance, rendered to screen in the PxWidget
   * @returns {module:bajaux/Widget|*} the closest instance of the given type in the widget's
   * ancestry (including the widget itself), or falsy if none found
   */
  exports.closestWidgetOfType = function (type, widget) {
    while (widget && !(widget instanceof type)) {
      widget = exports.getParentLastBuiltWidget(widget);
    }
    return widget;
  };

  /**
   * @private
   * @param {module:bajaux/Widget} widget
   * @returns {boolean} true if the widget can be moved (right now, this means the same as "is
   * direct child of a CanvasPane")
   */
  exports.$isMovable = function (widget) {
    return exports.getParentLastBuiltWidget(widget) instanceof CanvasPane;
  };

  /**
   * Sometimes a widget is not directly movable, but should still support a "move" gesture. Like:
   * if a widget is the content of a BorderPane, it itself cannot be moved, but I still want to be
   * able to drag it to move the BorderPane itself around. I don't want to have to think about
   * selecting the BorderPane and moving *that* - I just want to click and drag.
   *
   * This will find the closest widget that *should* be moved, if the user is dragging on the
   * given widget.
   *
   * @param {module:bajaux/Widget} widget
   * @returns {module:bajaux/Widget|null}
   */
  exports.closestMovableWidget = function (widget) {
    if (!widget) {
      return null;
    }
    return exports.$isMovable(widget) ? widget : exports.closestMovableWidget(exports.getParentLastBuiltWidget(widget));
  };

  /**
   * Given an array of selected widgets, find the widgets that will *actually* move if the user
   * clicks and drags them as a whole.
   *
   * We expect all the widgets to belong to a single CanvasPane. We won't support simultaneously
   * moving widgets that belong to different CanvasPanes - it will get unmanageable if they have
   * different scaling settings etc.
   *
   * But: I still want to lasso a whole CanvasPane and its contents and move it around by dragging.
   * So even if the selected widgets belong to different CanvasPanes, it's ok to move them as a
   * group as long as *those CanvasPanes are selected too*. If I lasso a CanvasPane and the Label
   * it contains, then dragging that Label should move that CanvasPane - not the Label itself. If I
   * ctrl-click two individual Labels that belong to two different CanvasPanes, I cannot move them
   * as a group.
   *
   * @param {Array.<module:bajaux/Widget>} widgets
   * @returns {{ widgets: Array.<module:bajaux/Widget>, canvasPane: module:nmodule/bajaui/rc/ux/CanvasPane }}
   */
  exports.getWidgetsToMove = function (widgets) {
    var movables = [];
    var commonCanvasPane;
    function actualWidgetToMove(widget) {
      var movable = exports.closestMovableWidget(widget);
      var movableParent = exports.getParentLastBuiltWidget(movable);
      if (widgets.includes(movableParent)) {
        return actualWidgetToMove(movableParent);
      } else {
        return movable;
      }
    }
    for (var i = 0, len = widgets.length; i < len; ++i) {
      var w = widgets[i];
      var widgetToMove = actualWidgetToMove(w);
      if (!widgetToMove) {
        return {
          widgets: []
        };
      }
      var movableParent = exports.getParentLastBuiltWidget(widgetToMove);
      commonCanvasPane = commonCanvasPane || movableParent;
      if (movableParent !== commonCanvasPane) {
        return {
          widgets: []
        };
      } else {
        movables.push(widgetToMove);
      }
    }
    return {
      widgets: uniq(movables),
      canvasPane: commonCanvasPane
    };
  };

  /**
   * 
   * @param {module:nmodule/uxBuilder/rc/ux/wysiwyg/artisans/Handle} handle 
   * @returns {number|null}
   */
  exports.getIndexOfHandle = function (handle) {
    var handleRegex = /^[a-z](\d+)$/g;
    var result = handleRegex.exec(handle.getName());
    return result && parseInt(result[1], 10);
  };

  /**
   * Find the index of the segment (line segment made of two points) nearest to the point
   * @param {{ x, y }} newPoint the target point 
   * @param {Array.<{ x, y }>} points list of points 
   * @param {boolean} closed polygons are always closed (for example)
   * @returns {number}
   */
  exports.nearest = function (newPoint, points, closed) {
    var distance = dist(newPoint, points[0], points[1]);
    var idx = 0;
    for (var i = 1; i < points.length - 1; i++) {
      var d = dist(newPoint, points[i], points[i + 1]);
      if (d < distance) {
        distance = d;
        idx = i;
      }
    }
    if (closed) {
      var _d = dist(newPoint, points[points.length - 1], points[0]);
      if (_d < distance) {
        distance = _d;
        idx = points.length - 1;
      }
    }
    return idx;
  };

  /**
   * Applies the snap to the new layout rectangle values
   * @param {Number} snapSize
   * @param {module:nmodule/bajaui/rc/baja/Layout~Rectangle} newLayoutValues
   * @param {module:nmodule/bajaui/rc/baja/Layout} currLayout
   * @returns {module:nmodule/bajaui/rc/baja/Layout~Rectangle}
   */
  exports.applySnapToLayout = function (snapSize, newLayoutValues, currLayout) {
    var x = newLayoutValues.x;
    var y = newLayoutValues.y;
    var w = newLayoutValues.w;
    var h = newLayoutValues.h;
    snapSize = snapSize || 1; //make sure that we have a snap size that is not zero of undefined
    var currX = currLayout.getX();
    var currY = currLayout.getY();
    var currW = currLayout.getWidth();
    var currH = currLayout.getHeight();

    //on resize of the widget then snap the values that changed only
    //if the resize movement is to the left or top we need to round the new value up, otherwise down
    w = currW === w ? w : (x < currX ? Math.ceil(w / snapSize) : Math.floor(w / snapSize)) * snapSize;
    h = currH === h ? h : (y < currY ? Math.ceil(h / snapSize) : Math.floor(h / snapSize)) * snapSize;

    //on a move of the widget, then set both x & y using the snapSize
    if (currX !== x || currY !== y) {
      x = Math.floor(x / snapSize) * snapSize;
      y = Math.floor(y / snapSize) * snapSize;
    }
    return {
      x: x,
      y: y,
      w: w,
      h: h
    };
  };

  /**
   * Applies the snap to point values
   * @param {Number} snapSize
   * @param {{ x: number, y: number }} point
   * @param {Boolean} [xIncreasing] set this to true if while moving the mouse the x point is increasing
   * @param {Boolean} [yIncreasing] set this to true if while moving the mouse the y point is increasing
   * @returns {{ x: number, y: number }}
   */
  exports.applySnapToPoint = function (snapSize, point, xIncreasing, yIncreasing) {
    var x = point.x || point.pageX;
    var y = point.y || point.pageY;

    // just to be on the safe side, let's make sure x & y are never undefined.
    x = x !== undefined ? x : 0;
    y = y !== undefined ? y : 0;
    snapSize = snapSize || 1; //make sure that we have a snap size that is not zero of undefined
    x = (xIncreasing ? Math.ceil(x / snapSize) : Math.floor(x / snapSize)) * snapSize;
    y = (yIncreasing ? Math.ceil(y / snapSize) : Math.floor(y / snapSize)) * snapSize;
    return {
      x: x,
      y: y
    };
  };

  /**
   * Given a segment and the previous point in a Path, this method returns
   * the segment's point. 
   * @private
   * @param {module:nmodule/gx/rc/baja/PathGeom~Segment} segment 
   * @param {{ x: number, y: number }} prevPoint 
   * @returns {{ x: number, y: number }}
   */
  exports.$getPoint = function (segment, _ref3) {
    var x = _ref3.x,
      y = _ref3.y;
    if (typeof segment.getPoint === 'function') {
      var point = segment.getPoint().toJson();
      return segment.getAbsolute() ? point : {
        x: point.x + x,
        y: point.y + y
      };
    } else if (typeof segment.getX === 'function') {
      var xVal = segment.getX();
      return segment.getAbsolute() ? {
        x: xVal,
        y: y
      } : {
        x: xVal + x,
        y: y
      };
    } else if (typeof segment.getY === 'function') {
      var yVal = segment.getY();
      return segment.getAbsolute() ? {
        x: x,
        y: yVal
      } : {
        x: x,
        y: yVal + y
      };
    } else if (typeof segment.getEnd === 'function') {
      var _segment$getEnd$toJso = segment.getEnd().toJson(),
        _xVal = _segment$getEnd$toJso.x,
        _yVal = _segment$getEnd$toJso.y;
      return segment.getAbsolute() ? {
        x: _xVal,
        y: _yVal
      } : {
        x: _xVal + x,
        y: _yVal + y
      };
    }
  };

  /**
   * Returns the bounding rect in document scope for given uxModels.
   *
   * @private
   * @param {module:nmodule/uxBuilder/rc/ux/wysiwyg/PxArtisanStudio~PxOverlayController} controller
   * @param {Array.<{module:bajaux/model/UxModel}>} models
   * @returns {Array.<{uxModel: {module:bajaux/model/UxModel}, rect: {module:nmodule/bajaui/rc/baja/Layout~Rectangle}}>}
   */
  exports.$getBoundingClientRectsOfUxModels = function (controller, models) {
    return models.map(function (uxModel) {
      var node = UxModelTreeNode.getOriginatingNode(uxModel);
      var lastBuiltWidget = node.getLastBuiltWidget();
      var rect = controller.$getBoundingRectangle(lastBuiltWidget);
      // When uxModel represents a Shape, use 'x and y' from the geom instead of the domRect
      // This helps to skip the browser math and rely on the geom math.
      // I chose geom to translate instead of a BoundRect because,
      // the browser BoundRect coordinates were offset by half a pixel, especially with polygons.
      if (uxModel.$represents(Shape)) {
        var origGeom = lastBuiltWidget.properties().getValue('geom');
        var artisan = controller.$makeArtisan(lastBuiltWidget);
        var canvasPane = exports.getParentLastBuiltWidget(lastBuiltWidget);
        var _artisan$$getStarting = artisan.$getStartingPoint(origGeom).toJson(),
          x = _artisan$$getStarting.x,
          y = _artisan$$getStarting.y;
        var _controller$getTracke = controller.getTracker().translatePoint({
            point: {
              x: x,
              y: y
            },
            to: 'document',
            from: canvasPane
          }),
          geomX = _controller$getTracke.x,
          geomY = _controller$getTracke.y;
        var w = rect.w,
          h = rect.h;
        return {
          uxModel: uxModel,
          rect: {
            x: geomX,
            y: geomY,
            w: w,
            h: h
          }
        };
      }
      return {
        uxModel: uxModel,
        rect: rect
      };
    });
  };

  /**
   * Translates the given uxModel to the rect coordinates.
   * It expects that the rect coordinates are at the scope of 'document',
   * and they will get translated to the parent CanvasPane.
   *
   * This method will translate for the Shape and non-Shape widgets alike.
   *
   * Uses: I want to move my uxModel to a new X and Y.
   *
   * @private
   * @param {object} obj
   * @param {module:bajaux/model/UxModel} obj.uxModel
   * @param {module:nmodule/bajaui/rc/baja/Layout~Rectangle} obj.rect
   * @param {module:nmodule/uxBuilder/rc/ux/wysiwyg/PxArtisanStudio~PxOverlayController} obj.controller
   * @returns {Promise.<module:bajaux/model/UxModel>}
   */
  exports.$translateUxModelToNewRect = function (_ref4) {
    var uxModel = _ref4.uxModel,
      rect = _ref4.rect,
      controller = _ref4.controller;
    var node = UxModelTreeNode.getOriginatingNode(uxModel);
    var widget = node.getLastBuiltWidget();
    var canvasPane = exports.getParentLastBuiltWidget(widget);
    var propertiesToUpdate = {};
    if (widget instanceof Shape) {
      var artisan = controller.$makeArtisan(widget);
      var origGeom = widget.properties().getValue('geom');
      var startPt = artisan.$getStartingPoint(origGeom).toJson();
      var _controller$getTracke2 = controller.getTracker().translateRect({
          rect: rect,
          from: 'document',
          to: canvasPane
        }),
        x = _controller$getTracke2.x,
        y = _controller$getTracke2.y;
      propertiesToUpdate.geom = origGeom.translate(Math.round(x) - startPt.x, Math.round(y) - startPt.y);
    } else {
      var layoutValues = controller.getTracker().translateRect({
        rect: rect,
        from: 'document',
        to: canvasPane
      });
      layoutValues = exports.roundLayoutRectangle(layoutValues);
      propertiesToUpdate.layout = baja.$('bajaui:Layout', layoutValues);
    }
    return uxModel.clone({
      properties: propertiesToUpdate
    });
  };
  return exports;
});
