/*
 * Copyright 2005 Tridium, Inc. All Rights Reserved.
 */

/* eslint-env browser */
/* exported $ */
/* global ActiveXObject, hx */

/**
 * hx
 *
 * @author    Andy Frank
 * @creation  5 Jan 05
 * @version   $Revision$ $Date$
 * @since     Baja 1.0
 */

////////////////////////////////////////////////////////////////
// DOM Extensions
////////////////////////////////////////////////////////////////

/**
 * Convenience for <code>document.getElementById(id)</code>.
 * <p>
 * Please note, since jQuery also uses a global '$' function, this method is
 * being deprecated. It's recommended to use the scoped hx.$ method instead.
 *
 * @see hx#id
 *
 * @deprecated
 */
function $(id) {
  //jshint ignore:line
  "use strict";

  return document.getElementById(id);
}

////////////////////////////////////////////////////////////////
// Hx
////////////////////////////////////////////////////////////////

//NCCB-27183: ensure that hx is only constructed once
if (!window.hx) {
  window.hx = new Hx();
}
function Hx() {
  "use strict";

  ////////////////////////////////////////////////////////////////
  // Attributes
  ////////////////////////////////////////////////////////////////
  this.pollTimeout = 5000;
  this.ie = navigator.appName.toLowerCase() === "microsoft internet explorer";
  this.ff = navigator.userAgent.toLowerCase().indexOf('firefox') > -1; //firefox
  this.ios = navigator.userAgent.toLowerCase().indexOf('like mac os x') > -1; // ios
  this.failure = false;
  this.dynamic = false;
  this.$fullScreen = false;

  // Last mouse position
  this.mx = 0;
  this.my = 0;

  // Menu
  this.menuId = "_hxMenu";

  // Dialog Id
  this.dialogId = "_hxDialog";
  this.dialogCounter = 0;
  this.dialogOnload = null;
  this.screenWidth = null;
  this.screenHeight = null;

  // Error
  this.err_LostConnection = "Session disconnected";
  this.errorInvokeCode = null;
  this.lastException = null;
  this.lastExceptionDetails = null;
  this.lastExceptionMessage = null;
  this.$firstPollStarted = null;
  this.$firstPollComplete = null;

  // Poll element array for sending data back on each poll request
  this.pollElementArray = [];

  /*
   * Array of functions that will return a promise to wait for completing before hx.save and form submit are completed.
   */
  this.preSave = [];

  /*
   * Object that contains previous windows opened as modal pop-ups
   */
  this.$modalPopups = {};
  this.$allowFormSubmit = false;
  this.$profileInfo = {};

  ////////////////////////////////////////////////////////////////
  // DOM Extensions
  ////////////////////////////////////////////////////////////////

  /**
   * Convenience for <code>document.getElementById(id)</code>.
   */
  this.$ = function (id) {
    return document.getElementById(id);
  };

  /**
   * Starting in Niagara 4.10U7, `hx.hyperlink` will be more standard and delegate to `niagara.env.hyperlink`.
   * This hyperlink will also now be asynchronous as it needs to contact the station for the proper
   * normalization of any `baja:ISubstitutableOrdScheme`s in the href.
   * @param {String} href
   * @returns {Promise}
   */
  this.hyperlink = function (href) {
    return hx.makePromise(function (resolve, reject) {
      require(["Promise", "nmodule/js/rc/asyncUtils/asyncUtils"], function (Promise, asyncUtils) {
        return asyncUtils.doRequire("nmodule/hx/rc/container/hxContainer").then(function () {
          return window.niagara.env.hyperlink(href);
        })["catch"](reject);
      });
    });
  };

  /**
   * Callback for when registered bajaux editors are marked modified.
   * @param {HTMLElement} elem
   * @since Niagara 4.6
   * @see module:nmodule/hx/rc/container/hxContainer.updateFormOnModify
   */
  this.modified = function (elem) {};

  /**
   * hx normally just saves by submitting the form, this can be overridden if needed. Additional work to be completed
   * prior to the save can be registered by adding to the 'hx.preSave' array of functions that will be called.
   * Saving will wait for any promises returned from those functions to complete. Any rejection errors will be given
   * to the user in a dialog.
   * @param {Window} [topWindow] pass the Window into the function.
   * @param {Function} [saveComplete] An Optional Function that will be called once save has been completed.
   * @param {Function} [saveError] An Optional Function that will be called if there is an error posting the form
   */
  this.save = function (topWindow, saveComplete, saveError) {
    var that = this;
    if (hx.preSave.length > 0) {
      require(["Promise"], function (Promise) {
        var callbacks = [];
        hx.preSave.forEach(function (preSaveFunction) {
          callbacks.push(preSaveFunction());
        });
        Promise.all(callbacks).then(function () {
          that.$saveSubmit(topWindow, saveComplete, saveError);
        })["catch"](function (err) {
          if (saveError) {
            saveError(err);
          } else {
            require(["nmodule/webEditors/rc/fe/feDialogs"], function (feDialogs) {
              feDialogs.error(err);
            });
          }
        });
      });
    } else {
      that.$saveSubmit(topWindow, saveComplete, saveError);
    }
  };

  /**
   * Once the preSave items are complete, submit the form based on the profile and UserAgent.
   * @param {Window} [topWindow] pass the Window into the function.
   * @param {Function} [saveComplete] An Optional Function that will be called once save has been completed.
   * @param {Function} [saveError] An Optional Function that will be called if there is an error posting the form
   */
  this.$saveSubmit = function (topWindow, saveComplete, saveError) {
    if (hx.ff && topWindow && topWindow !== window) {
      //This is specifically for an outer frame saving an iframe in firefox,
      //we lose the ability to show any error messages
      this.$submitFormData(document.forms[0], saveComplete, saveError);
    } else {
      this.$submitDirect(document.forms[0], saveComplete);
    }
  };

  /**
   * Submit Form Data either by or clicking the 'hx_submit' input submit button or directly calling Form.submit().
   * This input click allows form submission for firefox when programmatic calls to form submissions are sometimes ignored by browsers like firefox.
   * @param {HTMLFormElement} form
   * @param {Function} [saveComplete] Is called when submission is successful.
   */
  this.$submitDirect = function (form, saveComplete) {
    var submit = hx.$('hx_submit');
    if (submit && submit.click) {
      hx.$allowFormSubmit = true;
      try {
        submit.click();
      } finally {
        hx.$allowFormSubmit = false;
      }
    } else {
      form.submit();
    }
    if (saveComplete) {
      saveComplete();
    }
  };

  /**
   * Submit Form Data via FormData Constructor. This allows form submission for firefox when form submissions
   * is initiated from an another iframe. Note This does not display the form results.
   * @param {HTMLFormElement} form
   * @param {Function} [saveComplete] Is called when submission is successful.
   * @param {Function} [saveError] Is called when submission has an error.
   * @private
   */
  this.$submitFormData = function (form, saveComplete, saveError) {
    var XHR = new XMLHttpRequest();
    if (saveComplete) {
      XHR.addEventListener("load", saveComplete);
    }
    if (saveError) {
      XHR.addEventListener("error", saveError);
    }
    XHR.open("POST", form.action, true);
    XHR.send(new FormData(form));
  };

  /**
   * Register a 'javax.baja.hx.Event' to be completed prior to completing the save. If the Event returns an empty string as
   * its response, the pre-save is considered successful. If a non-empty response is provided, we will attempt to
   * return any error encoded as a 'baja.comm.BoxError'.
   * @param {String} path
   * @param {String} eventId
   */
  this.preSaveEvent = function (path, eventId) {
    require(["Promise", "baja!"], function (Promise, baja) {
      var callback = function callback() {
        // eslint-disable-next-line promise/avoid-new
        return new Promise(function (resolve, reject) {
          hx.fireEvent(path, eventId, null, function (resp) {
            if (!resp.responseText) {
              //no error messages, all went well, allow save to proceed
              resolve();
            } else {
              try {
                var error = baja.comm.BoxError.decodeFromServer("", JSON.parse(resp.responseText));
                reject(error);
              } catch (err) {
                baja.error(err);
                reject(new Error(resp.responseText));
              }
            }
          });
        });
      };
      hx.preSave.push(callback);
    });
  };

  ////////////////////////////////////////////////////////////////
  // Lifecycle
  ////////////////////////////////////////////////////////////////

  this.started = function (dynamic, timeout) {
    this.pollTimeout = timeout;
    this.dynamic = dynamic;

    //firefox form caching fix
    if (document.forms[0] !== null) {
      document.forms[0].reset();
    }
    var width = this.getScreenWidth(),
      height = this.getScreenHeight();
    if (width) {
      this.screenWidth = width;
    }
    if (height) {
      this.screenHeight = height;
    }

    //reset $modalPopups on any restart of hx
    this.$modalPopups = {};

    // TODO - can hide needed error messages
    //window.onerror = this.doWindowError;

    document.body.onclick = function () {
      if (!hx.$ignoreMouseDown) {
        hx.closeMenu();
      }
    };
    if (dynamic) {
      setTimeout(function () {
        hx.poller();
      }, this.pollTimeout);
    }
  };

  /**
   * Start the activity monitor that monitors for user activity and periodically
   * lets the server know about this to account for a user's session expiry.
   */
  this.startActivityMonitor = function () {
    //start the activity monitor
    require(["nmodule/web/rc/util/activityMonitor"], function (ActivityMonitor) {
      ActivityMonitor.start();
    });
  };

  /**
   * Set the `profileInfo`.
   * This information can include the `viewId` for the current ord.
   * @param {Object} params
   * @param {String} [params.viewId]
   * @since Niagara 4.10
   */
  this.setProfileInfo = function (params) {
    if (!params) {
      throw new Error("Invalid argument");
    }
    this.$profileInfo = Object.assign({}, params);
  };

  /**
   * Get the `profileInfo`.
   * Note that it may be out-of-date if the
   * content of the page has changed without reloading the browser.
   * @returns {{ viewId: string }}
   * @since Niagara 4.10
   */
  this.getProfileInfo = function () {
    return Object.assign({}, this.$profileInfo);
  };
  this.stopped = function () {};

  /**
   * Return the encoded csrfToken if it exists. For custom POSTs, ensure to add a header called 'x-niagara-csrfToken'
   * with this csrfToken.
   * @returns {String}
   *
   * @example
   *   <caption>Here is an example of a custom post to an HxView which contains the csrf token in the headers.
   *   </caption>
   * .ajax(window.location.href, {
   *      type: "POST",
   *      headers: {
   *        'x-niagara-csrfToken': hx.getCsrfToken()
   *      },
   *      contentType: contentType,
   *      dataType: "json",
   *      data: JSON.stringify(json),
   *     processData: false
   *    });
   */
  this.getCsrfToken = function () {
    var csrfTokenElem = document.getElementsByName("csrfToken")[0];
    if (csrfTokenElem) {
      return hx.encodeString(csrfTokenElem.value);
    } else {
      return null;
    }
  };

  ////////////////////////////////////////////////////////////////
  // Polling
  ////////////////////////////////////////////////////////////////

  /**
   * Ensure the page has started its first poll if its not already in process.
   * @param {boolean} [benchmarkEnabled] If true, wait for first poll to complete before considering the page loaded.
   */
  this.ensureFirstPoll = function (benchmarkEnabled) {
    if (benchmarkEnabled) {
      hx.waitForLoadTime(function () {
        return hx.$firstPollComplete;
      });
    }
    if (!hx.$firstPollStarted) {
      hx.doPoll(false);
    }
  };
  this.poller = function () {
    if (this.dynamic) {
      this.doPoll(true);
    }
  };
  this.poll = function () {
    this.doPoll(false);
  };
  this.doPoll = function (repeat) {
    if (this.failure) {
      return;
    }
    if (!this.$firstPollStarted) {
      this.$firstPollStarted = new Date();
    }
    if (this.screenWidth === null || this.screenWidth === 0 || this.screenHeight === null || this.screenHeight === 0) {
      this.screenWidth = this.getScreenWidth();
      this.screenHeight = this.getScreenHeight();
    }
    var msg = new Message();
    msg.setHeader("Content-Type", "application/x-niagara-hx-update");
    msg.setHeader("Screen-Width", this.screenWidth);
    msg.setHeader("Screen-Height", this.screenHeight);
    var csrfToken = hx.getCsrfToken();
    if (csrfToken) {
      msg.setHeader("x-niagara-csrfToken", csrfToken);
    }
    var pollElementBody = "";
    if (this.pollElementArray.length > 0) {
      pollElementBody = hx.encodeForm(document.body, msg, this.pollElementArray);
    }
    msg.send(window.location, pollElementBody, repeat && this.dynamic ? this.pollHandlerRepeat : this.pollHandler);
  };
  this.pollHandler = function (resp) {
    var text = resp.responseText;
    try {
      Hx.$eval(text);
      if (!hx.$firstPollComplete) {
        hx.$firstPollComplete = new Date();
      }
    } catch (err) {
      window.console.log(text);
      window.console.log(err);
      if (text.substring(0, 6) !== "<html>") {
        text = text.replace(new RegExp("<", "gi"), "&lt;");
        text = text.replace(new RegExp(">", "gi"), "&gt;");
      }
      hx.doError(null, text, err);
    }
  };
  this.pollHandlerRepeat = function (resp) {
    var text = resp.responseText;
    try {
      Hx.$eval(text);
      if (!hx.$firstPollComplete) {
        hx.$firstPollComplete = new Date();
      }
      setTimeout(function () {
        hx.poller();
      }, hx.pollTimeout);
    } catch (err) {
      window.console.log(text);
      window.console.log(err);
      if (text.substring(0, 6) !== "<html>") {
        text = text.replace(new RegExp("<", "gi"), "&lt;");
        text = text.replace(new RegExp(">", "gi"), "&gt;");
      }
      hx.doError(null, text, err);
    }
  };

  /**
   * Get the containing body, or the element containing the current iframe if
   * hx is being hosted in an iframe.
   *
   * @since Niagara 4.6
   */
  this.getContainer = function () {
    // IOS logic for calculating height within iframes
    var top;
    if (this.ios && (top = hx.topWindow()) !== window) {
      var elements = top.document.getElementsByName(window.name);
      if (elements.length && elements[0].parentNode) {
        if (elements[0].parentNode.clientWidth || elements[0].parentNode.clientHeight) {
          return elements[0].parentNode;
        }
      }
    }
    if (document.body && (document.body.clientWidth || document.body.clientHeight)) {
      return document.body;
    } else if (document.documentElement && (document.documentElement.clientWidth || document.documentElement.clientHeight)) {
      return document.documentElement;
    }
  };

  /**
   * @since Niagara 3.5
   */
  this.getScreenHeight = function () {
    var container = this.getContainer();
    if (container) {
      return container.clientHeight;
    }
    return 0;
  };

  /**
   * @since Niagara 3.5
   */
  this.getScreenWidth = function () {
    var container = this.getContainer();
    if (container) {
      return container.clientWidth;
    }
    return 0;
  };

  /**
   * Returns an array containing the provided elements bounds [left,top]
   * Coordinates are relative to the document
   * @since Niagara 3.5
   */
  this.getElementBounds = function (obj) {
    var curleft = 0,
      curtop = 0;
    if (obj && obj.offsetParent) {
      do {
        curleft += obj.offsetLeft;
        curtop += obj.offsetTop;
      } while (obj = obj.offsetParent);
      return [curleft, curtop];
    } else {
      return [0, 0];
    }
  };

  /**
   * Returns an array containing the provided elements bounds [left, top, right, bottom, width, height]
   * Coordinates are relative to the viewport
   */
  this.getElementBoundsViewport = function (obj) {
    var rect = obj.getBoundingClientRect();
    var left = rect.left;
    var top = rect.top;
    var right = rect.right;
    var bottom = rect.bottom;
    var height = rect.height;
    var width = rect.width;
    return [left, top, right, bottom, width, height];
  };

  ////////////////////////////////////////////////////////////////
  // Events
  ////////////////////////////////////////////////////////////////

  /**
   * Fire the specified event.
   * @param {String} path
   * @param {String} eventId
   * @param {Object} [event] optional event for helping determine Mouse Position
   * @param {Function} [handler] optional handler for when event is complete, default is hx.doFireEvent
   */
  this.fireEvent = function (path, eventId, event, handler) {
    //hx.enterBusy();

    // If the value for a header is empty, the header will not
    // be set, so check if path is empty, and force it to be
    // something if it is.  HxView will look for this case, and
    // reset path to an empty string.
    if (path.length === 0) {
      path = "*";
    }
    if (event !== null) {
      var pos = this.getMousePosition(event);
      this.mx = pos[0];
      this.my = pos[1];
    }
    if (!handler) {
      handler = this.doFireEvent;
    }
    var form = hx.encodeForm(document.body, null, null);
    var msg = new Message();
    msg.setHeader("x-niagara-hx-path", path);
    msg.setHeader("x-niagara-hx-eventId", eventId);
    msg.setHeader("Content-Type", "application/x-niagara-hx-event");
    msg.send(window.location, form, handler);
  };

  /**
   * Retrieve current x and y position for mouse for the given event relative to the document (within this iframe)
   * @param {Object} [event]
   * @return {Array.<int>} posx, posy
   */
  this.getMousePosition = function (event) {
    var posx = 0;
    var posy = 0;
    if (!event) {
      event = window.event;
    }
    if (!event) {
      return [posx, posy];
    }
    if (event.pageX || event.pageY) {
      //console.log('hx pageY: ', event.pageY, ' clientY: ', event.clientY);
      posx = event.pageX;
      posy = event.pageY;
    } else if (event.touches && event.touches[0] && (event.touches[0].pageX || event.touches[0].pageY)) {
      posx = event.touches[0].pageX;
      posy = event.touches[0].pageY;
    } else if (event.changedTouches && event.changedTouches[0] && (event.changedTouches[0].pageX || event.changedTouches[0].pageY)) {
      //for touchEnd/touchCancel which doesn't have event.touches
      posx = event.changedTouches[0].pageX;
      posy = event.changedTouches[0].pageY;
    } else if (event.clientX || event.clientY) {
      // console.log('hx clientY: ', event.clientY, 'scrollTop: ', document.body.scrollTop, 'scrollTop: ', document.documentElement.scrollTop);
      posx = event.clientX + document.body.scrollLeft + document.documentElement.scrollLeft;
      posy = event.clientY + document.body.scrollTop + document.documentElement.scrollTop;
    }
    return [posx, posy];
  };

  /**
   * Retrieve current x and y position for mouse for the given event relative to the current viewport
   * @param {Object} [event]
   * @return {Array.<int>} posx, posy
   */
  this.getMousePositionViewport = function (event) {
    var posx = 0;
    var posy = 0;
    if (!event) {
      event = window.event;
    }
    if (!event) {
      return [posx, posy];
    }
    if (event.clientX || event.clientY) {
      posx = event.clientX;
      posy = event.clientY;
    } else if (event.touches && event.touches[0] && (event.touches[0].clientX || event.touches[0].clientY)) {
      posx = event.touches[0].clientX;
      posy = event.touches[0].clientY;
    }
    return [posx, posy];
  };

  /**
   * Handle response to fireEvent.
   */
  this.doFireEvent = function (resp) {
    //hx.exitBusy();
    var text = resp.responseText;
    try {
      if (text.length > 0) {
        Hx.$eval(text);
      }
    } catch (err) {
      window.console.log(text);
      window.console.log(err);
      text = text.replace(new RegExp("<", "gi"), "&lt;");
      text = text.replace(new RegExp(">", "gi"), "&gt;");
      hx.doError(null, text, err);
    }
  };

  ////////////////////////////////////////////////////////////////
  // HxGraphics
  ////////////////////////////////////////////////////////////////

  this.fixHxPathBar = function () {
    hx.fixNavChildren('hx-PathBar-path', 'hx-PathBar');
    hx.fixNavChildren('hx-PathBar');
    hx.fixNavChildren('hx-PathBar-children');
  };
  this.fixNavChildren = function (elemId, firstId) {
    var i;
    var elem = document.getElementById(elemId);
    if (this.screenWidth === null) {
      this.screenWidth = this.getScreenWidth();
    }
    if (elem.scrollWidth > this.screenWidth) {
      if (firstId) {
        var firstLinks = document.getElementById(firstId).getElementsByTagName('A');
        for (i = 0; i < firstLinks.length; i++) {
          firstLinks[i].title = firstLinks[i].childNodes[firstLinks[i].childNodes.length - 1].nodeValue;
          firstLinks[i].childNodes[firstLinks[i].childNodes.length - 1].nodeValue = "";
        }
      }
      var links = elem.getElementsByTagName('A');
      i = 0;
      while (elem.scrollWidth > this.screenWidth && i < links.length) {
        links[i].title = links[i].childNodes[links[i].childNodes.length - 1].nodeValue;
        links[i].childNodes[links[i].childNodes.length - 1].nodeValue = "";
        i++;
      }
      if (elem.scrollWidth > this.screenWidth) {
        elem.style.width = this.screenWidth + "px";
      }
    }
  };

  /**
   * Set the current brush as an Object.
   * @param {CanvasRenderingContext2D} g
   * @param {string|CanvasGradient|CanvasPattern} color
   */
  this.setColorObject = function (g, color) {
    g.strokeStyle = color;
    g.fillStyle = color;
  };

  /**
   * Set the current brush as a color.
   * @param {CanvasRenderingContext2D} g
   * @param {Number} red
   * @param {Number} green
   * @param {Number} blue
   * @param {Number} alpha
   */
  this.setColor = function (g, red, green, blue, alpha) {
    g.strokeStyle = "rgba(" + red + ", " + green + ", " + blue + ", " + alpha + ")";
    g.fillStyle = "rgba(" + red + ", " + green + ", " + blue + ", " + alpha + ")";
  };

  /**
   * Set the current brush as a color.
   * @param {d3} elem
   * @param {boolean} fill
   * @param {Number} red
   * @param {Number} green
   * @param {Number} blue
   * @param {Number} alpha
   */
  this.setSvgColor = function (elem, fill, red, green, blue, alpha) {
    if (fill) {
      elem.attr('fill', "rgba(" + red + ", " + green + ", " + blue + ", " + alpha + ")");
    } else {
      elem.attr('stroke', "rgba(" + red + ", " + green + ", " + blue + ", " + alpha + ")");
    }
  };

  /**
   * Set the current brush as a linear gradient.
   */
  this.setLinearGradient = function (g, x1, y1, x2, y2, stops) {
    try {
      var grad = g.createLinearGradient(x1, y1, x2, y2);
      for (var i = 0; i < stops.length; i++) {
        grad.addColorStop(stops[i][0] / 100, stops[i][1]);
      }
      g.strokeStyle = grad;
      g.fillStyle = grad;
    } catch (err) {
      window.console.log("hx.setLinearGradient: " + err);
    }
  };

  /**
   * Set the current brush as a radial gradient.
   */
  this.setRadialGradient = function (g, x1, y1, r1, x2, y2, r2, stops) {
    try {
      var grad = g.createRadialGradient(x1, y1, r1, x2, y2, r2);
      for (var i = 0; i < stops.length; i++) {
        grad.addColorStop(stops[i][0] / 100, stops[i][1]);
      }
      g.strokeStyle = grad;
      g.fillStyle = grad;
    } catch (err) {
      window.console.log("hx.setRadialGradient: " + err);
    }
  };

  /**
   * Clip this rectangle.
   */
  this.clipRect = function (g, x, y, w, h) {
    g.beginPath();
    g.moveTo(x, y);
    g.lineTo(x + w, y);
    g.lineTo(x + w, y + h);
    g.lineTo(x, y + h);
    g.closePath();
    g.clip();
  };

  /**
   * Draws a line between the two points using the
   * current pen and brush.
   */
  this.strokeLine = function (g, x1, y1, x2, y2) {
    g.beginPath();
    g.moveTo(x1, y1);
    g.lineTo(x2, y2);
    g.stroke();
    g.closePath();
  };

  /**
   * Draws the text given by the specified string
   * @deprecated
   */
  this.drawString = function (parentId, str, color, font, x, y) {
    var canvas = document.getElementById(parentId);
    var canvasParent = canvas.parentNode;
    if (hx.ie && canvasParent.style.position === "relative") {
      // sort of a hack since I usually depend on a
      // relative div above the canvas tag
      var s = canvasParent.style.paddingTop;
      if (s !== null && s.length > 2) {
        y += parseInt(s.substring(0, s.length - 2));
      }
    } else {
      // the normal rest of the world...
      x += canvas.offsetLeft;
      y += canvas.offsetTop;
    }
    var child = document.createElement("div");
    child.style.position = "absolute";
    child.style.left = x + "px";
    child.style.top = y + "px";
    child.style.color = color;
    child.style.font = font;
    child.innerHTML = str;
    child.name = parentId + ".text";
    canvasParent.appendChild(child);
  };

  /**
   * Clear the canvas and all text.
   */
  this.clearCanvas = function (g, width, height, id) {
    var canvas = document.getElementById(id);
    var canvasParent = canvas.parentNode;
    g.clearRect(0, 0, width, height);
    var childNodes = canvasParent.childNodes;
    for (var i = childNodes.length - 1; i > 0; i--) {
      if (childNodes[i].tagName === 'DIV') {
        canvasParent.removeChild(childNodes[i]);
      }
    }
  };
  this.svgClearAll = function (g) {
    g.selectAll("*").remove();
    g.append('defs');
  };

  ////////////////////////////////////////////////////////////////
  // Dialog
  ////////////////////////////////////////////////////////////////

  /**
   * Display a dialog with the given body.
   */
  this.showDialog = function (body) {
    this.closeMenu();
    var dlg = document.createElement("div");
    var i = hx.dialogCounter;
    while (document.getElementById(hx.dialogId + i) !== null) {
      i++;
    }
    dlg.id = hx.dialogId + i;
    hx.dialogCounter = i;
    if (hx.ie) {
      dlg.style.position = "absolute";
      dlg.style.left = document.body.scrollLeft + "px";
      dlg.style.top = document.body.scrollTop + "px";
      dlg.style.width = hx.getScreenWidth() + "px";
      dlg.style.height = hx.getScreenHeight() + "px";
    } else {
      dlg.style.position = "fixed";
      dlg.style.top = "0px";
      dlg.style.left = "0px";
      dlg.style.width = "100%";
      dlg.style.height = "100%";
    }
    dlg.style.zIndex = 99;
    var bg = document.createElement("div");
    bg.className = "dialog-background";
    dlg.appendChild(bg);

    // Create dialog
    var content = document.createElement("div");
    content.className = "dialog";
    if (typeof body === "string") {
      content.innerHTML = body;
    } else {
      content.appendChild(body);
    }
    dlg.appendChild(content);

    // Add content
    document.forms[0].appendChild(dlg);

    //fix MaxHeight
    hx.dialogMaxHeight();

    // Try to focus first control in dialog
    hx.focus(dlg.childNodes);
    if (hx.dialogOnload !== null) {
      Hx.$eval(hx.dialogOnload);
    }
    require(["jquery"], function ($) {
      var dom = $(dlg),
        ENTER_KEY = 13;

      //PropertySheet.js consumes keydown so we'll use keyup as well
      dom.on('keydown keyup', 'input', function (e) {
        if (e.keyCode === ENTER_KEY) {
          if (e.type === "keyup") {
            dom.find(".dialog-button-content-hx .button:first").click();
          }
          return false;
        }
      });
    });
  };
  this.dialogMaxHeight = function () {
    var elem = this.$('dialog-maxHeight');
    if (elem === null) {
      return;
    }
    //replaceId do there is no collision
    elem.id = 'dialog-maxHeight' + hx.dialogCounter;
    var height = parseInt(hx.getScreenHeight()) * 8 / 10 - 100;
    hx.maxHeight(elem, height);
    var width = parseInt(hx.getScreenWidth()) * 8 / 10 - 30;
    hx.maxWidth(elem, width);
  };
  this.resizeCurrentDialog = function () {
    var elem = this.$('dialog-maxHeight' + hx.dialogCounter);
    if (elem === null) {
      return;
    }
    elem.style.height = "";
    var height = parseInt(hx.getScreenHeight()) * 8 / 10 - 100;
    hx.maxHeight(elem, height);
    elem.style.width = "";
    var width = parseInt(hx.getScreenWidth()) * 8 / 10 - 30;
    hx.maxWidth(elem, width, true);
  };
  this.maxHeight = function (elem, height) {
    if (elem === null) {
      return;
    }
    if (elem.offsetHeight > parseInt(height)) {
      elem.style.height = height + "px";
      elem.style.overflow = "auto";
    }
  };
  this.maxWidth = function (elem, width, ieFix) {
    if (elem === null) {
      return;
    }
    if (elem.offsetWidth > parseInt(width)) {
      elem.style.width = width + "px";
      elem.style.overflow = "auto";
    } else if (hx.ie && ieFix) {
      elem.style.width = elem.offsetWidth + "px";
      elem.style.overflow = "auto";
    }
  };

  /**
   * Recurse through nodes to focus first input control.
   */
  this.focus = function (nodes) {
    for (var i = 0; i < nodes.length; i++) {
      var name = nodes[i].nodeName;
      try {
        if (nodes[i].offsetHeight !== 0 && (name === "SELECT" || name === "TEXTAREA" || name === "INPUT" && nodes[i].type !== "hidden") && nodes[i].disabled !== true && nodes[i].readOnly !== true) {
          nodes[i].focus();
          return true;
        }
      } catch (err) {}
      if (hx.focus(nodes[i].childNodes)) {
        return true;
      }
    }
    return false;
  };

  /**
   * Close any open dialog. If path and eventId is not
   * null, then fire an event with that id.
   *
   * @param {String} path
   * @param {String} eventId
   * @param {String} [event]
   */
  this.closeDialog = function (path, eventId, event) {
    var eventSrc = null;
    if (event) {
      eventSrc = !event.target ? window.event.srcElement : event.target;
    }
    var form = null;
    var dlg = document.getElementById(hx.dialogId + hx.dialogCounter);
    if (dlg !== null) {
      form = hx.encodeForm(document.body, eventSrc, null);
      dlg.parentNode.removeChild(dlg);
      hx.dialogCounter--;
      if (hx.dialogCounter < 0) {
        hx.dialogCounter = 0;
      }
    }
    if (path !== null && eventId !== null) {
      // Set path to something if empty to force the
      // header to be set.
      if (path.length === 0) {
        path = "*";
      }
      var msg = new Message();
      msg.setHeader("x-niagara-hx-path", path);
      msg.setHeader("x-niagara-hx-eventId", eventId);
      msg.setHeader("Content-Type", "application/x-niagara-hx-event");
      msg.send(window.location, form, hx.doFireEvent);
    }
  };

  ////////////////////////////////////////////////////////////////
  // Menu
  ////////////////////////////////////////////////////////////////

  this.doShowMenu = function (body) {
    this.closeMenu();
    var menu = document.createElement("ul");
    menu.className = "context-menu-list context-menu-root";
    menu.id = hx.menuId;
    menu.style.position = "absolute";
    menu.style.left = hx.mx + "px";
    menu.style.top = hx.my + "px";
    menu.style.zIndex = "99";
    menu.oncontextmenu = function () {
      return false;
    };

    // Create dialog
    menu.innerHTML = body;

    // Add content
    document.forms[0].appendChild(menu);

    // Make sure it doesn't hang below the bottom of the
    // visible display window
    var screenHeight = this.getScreenHeight();
    if (menu.offsetHeight + hx.my > screenHeight) {
      menu.style.top = hx.my - (menu.offsetHeight + hx.my - screenHeight) - 10 + "px";
    }

    // Try to focus first control in dialog
    hx.focus(menu.childNodes);
  };
  this.closeMenu = function () {
    var menu = document.getElementById(hx.menuId);
    if (menu !== null) {
      menu.parentNode.removeChild(menu);
    }
  };

  ////////////////////////////////////////////////////////////////
  // Forms
  ////////////////////////////////////////////////////////////////

  /**
   * Encode the form elements that exist under the given
   * element into a string.
   */
  this.encodeForm = function (elem, submit, elemNameArray) {
    var controls = [];

    // Recursively search for Elements
    var _find = function find(currentElem) {
      var i;
      // Is the current node of type Node.ELEMENT_TYPE?
      if (currentElem.nodeType === 1) {
        var n = currentElem.tagName.toLowerCase();
        if (n === "input" || n === "select" || n === "textarea") {
          if (elemNameArray === null) {
            controls.push(currentElem);
          } else {
            for (i = 0; i < elemNameArray.length; i++) {
              if (currentElem.name === elemNameArray[i]) {
                controls.push(currentElem);
                break;
              }
            }
          }
        }
        for (i = 0; i < currentElem.childNodes.length; i++) {
          _find(currentElem.childNodes[i]);
        }
      }
    };
    _find(elem);
    var encoding = "",
      i;
    for (i = 0; i < controls.length; i++) {
      var control = controls[i];
      var type = control.type.toLowerCase();

      // if the input is a submit and it's not the button
      // that was pressed, skip it
      if (submit !== null && control.type === "submit" && control !== submit) {
        continue;
      }

      // Determine if control is "successful"
      if (control.disabled) {
        continue;
      }
      if (control.name === null || control.name.length === 0) {
        continue;
      }
      if ((type === "radio" || type === "checkbox") && !control.checked) {
        continue;
      }

      // Escape illegal characters
      var value = this.encodeString(control.value);

      // Append name/value pair
      if (encoding.length > 0) {
        encoding += "&";
      }
      encoding += control.name + "=" + value;
    }
    return encoding;
  };

  /**
   * Encode illegal characters.
   */
  this.encodeString = function (s) {
    var e = "",
      low,
      mid,
      high;
    for (var i = 0; i < s.length; i++) {
      var ch = s.charAt(i);
      var code = s.charCodeAt(i);
      var isChar = false;
      if (code >= 48 && code <= 57) {
        isChar = true;
      } else if (code >= 65 && code <= 90) {
        isChar = true;
      } else if (code >= 97 && code <= 122) {
        isChar = true;
      }
      if (ch === " ") {
        e += "+";
      } else if (!isChar && code <= 127) {
        e += "%" + hx.toHex(code);
      } else if (code > 127 && code < 0x0800) {
        // utf-8 two bytes
        high = 0xc0 | code >> 6 & 0x001F;
        low = 0x80 | code & 0x003f;
        e += "%" + hx.toHex(high);
        e += "%" + hx.toHex(low);
      } else if (code >= 0x0800) {
        // utf-8 three bytes
        high = 0xe0 | code >> 12 & 0x000f;
        mid = 0x80 | code >> 6 & 0x003f;
        low = 0x80 | code & 0x003f;
        e += "%" + hx.toHex(high);
        e += "%" + hx.toHex(mid);
        e += "%" + hx.toHex(low);
      } else {
        // TODO: Handle four bytes
        e += ch;
      }
    }
    return e;
  };
  this.toHex = function (val) {
    var s = val.toString(16).toUpperCase();
    if (s.length === 1) {
      s = "0" + s;
    }
    return s;
  };

  /**
   * Update a form value with the given element id. If the elem does not exist, do nothing.
   * @param {String} key
   * @param {Object} value
   * @param [escapeNewLines] optionally escape newlines
   */
  this.updateValue = function (key, value, escapeNewLines) {
    var elem = document.getElementById(key);
    if (!elem) {
      return;
    }
    if (escapeNewLines) {
      var sub = String.fromCharCode(65533);
      var re = new RegExp(sub + 'n', 'gi');
      value = value.replace(re, "\n");
      re = new RegExp(sub + 'r', 'gi');
      value = value.replace(re, "\r");
      re = new RegExp(sub + 's', 'gi');
      value = value.replace(re, "\\");
    }
    if (elem.value !== value) {
      elem.value = value;
    }
  };

  /**
   * Set the value of the form element with this name. If this
   * element does not exist, add it as a hidden control type.
   * @param {String} key
   * @param {String} value
   */
  this.setFormValue = function (key, value) {
    var control = null,
      inputs = document.body.getElementsByTagName("input"),
      i;
    for (i = 0; i < inputs.length; i++) {
      if (inputs[i].name === key) {
        control = inputs[i];
        break;
      }
    }
    var selects = document.body.getElementsByTagName("select");
    for (i = 0; i < selects.length; i++) {
      if (selects[i].name === key) {
        control = selects[i];
        break;
      }
    }
    var textarea = document.body.getElementsByTagName("textarea");
    for (i = 0; i < textarea.length; i++) {
      if (textarea[i].name === key) {
        control = textarea[i];
        break;
      }
    }
    if (control === null) {
      control = document.createElement("input");
      control.type = "hidden";
      control.name = key;
      control.id = key;
      document.forms[0].appendChild(control);
    }
    control.value = value;
  };

  /**
   * Add a poll element to include in the update
   */
  this.addFormElementToPoll = function (name) {
    this.pollElementArray.push(name);
  };

  /**
   * Remove a poll element from the update
   */
  this.removeFormElementFromPoll = function (name) {
    for (var i = 0; i < this.pollElementArray.length; i++) {
      if (this.pollElementArray[i] === name) {
        this.pollElementArray.splice(i, 1);
        break;
      }
    }
  };

  /**
   * For the given element add a long touch event. When the user has pressed an element for 500ms, launch the callback
   * @param {object} elem Html Element
   * @param {Function} callback Function to call when long press is complete
   * @private
   */
  this.$addLongTouch = function (elem, callback) {
    if (!elem) {
      return;
    }
    var cancelLongTouch = false,
      longClickDelay = 500;
    elem.addEventListener("touchstart", function (e) {
      cancelLongTouch = false;
      if (e.touches && e.touches.length > 1) {
        return;
      }
      setTimeout(function () {
        if (cancelLongTouch) {
          return;
        }
        e.button = 2; //simulate right click
        hx.$ignoreMouseDown = true;
        callback(e);
      }, longClickDelay);
    }, false);
    function ignoreMouseDown() {
      cancelLongTouch = true;
      setTimeout(function () {
        hx.$ignoreMouseDown = false;
      }, 100);
    }
    elem.addEventListener("touchend", function () {
      ignoreMouseDown();
    }, false);
    elem.addEventListener("touchcancel", function () {
      ignoreMouseDown();
    }, false);
    elem.addEventListener("touchmove", function () {
      cancelLongTouch = true;
    }, false);
  };

  /**
   * Set modifiers for mouse event
   * @param {MouseEvent} event
   * @return {boolean} false when to ignore the MouseEvent
   */
  this.setMouseEvent = function (event) {
    if (event && event.type === "mousedown" && hx.$ignoreMouseDown) {
      return false;
    }
    if (event) {
      // the hidden input fields used here are also
      // defined statically in BHxPxView.  When used
      // elsewhere the fields will be created dynamically.

      // set mouse button
      hx.setFormValue("button", event.button === 2 ? "right" : "left");

      // set mouse position
      var pos = this.getMousePosition(event);
      hx.setFormValue("x", pos[0]);
      hx.setFormValue("y", pos[1]);

      // set the mouse event id and modifiers
      hx.setFormValue("id", event.type);
      hx.setFormValue("shiftModifier", event.shiftKey);
      hx.setFormValue("ctlModifier", event.ctrlKey);
      hx.setFormValue("altModifier", event.altKey);
      hx.setFormValue("metaModifier", event.metaKey);
    }
    return true;
  };
  this.stopEventPropagation = function (event) {
    if (!event) {
      event = window.event;
    }
    if (event) {
      if (event.stopPropagation) {
        event.stopPropagation();
      }
      event.cancelBubble = true;
    }
    return false;
  };

  ////////////////////////////////////////////////////////////////
  // Busy
  ////////////////////////////////////////////////////////////////

  /**
   * Enter the busy state.  This blocks input and displays
   * a busy indicator to the user.  You must call exitBusy()
   * to leave this state.
   */
  this.enterBusy = function () {
    // TODO - do something cooler
    var body = "<div class='busy'>Loading...</div>";
    hx.showDialog(body);
  };

  /**
   * Exit the busy state.
   */
  this.exitBusy = function () {
    hx.closeDialog(null, null, null);
  };

  ////////////////////////////////////////////////////////////////
  // Error
  ////////////////////////////////////////////////////////////////

  /**
   * Handle window error.
   */
  this.doWindowError = function (msg, url, line) {
    // Create an exception to pass to doError
    var ex = {};
    ex.name = "window.onerror";
    ex.message = msg;
    ex.fileName = url;
    ex.lineNumber = line;
    ex.stack = "";
    hx.doError("window.onerror", "", ex);
    return true;
  };

  /**
   * Handle error.
   */
  this.doError = function (msg, details, exception) {
    // Make sure exception is valid
    if (exception === null) {
      exception = {};
    }

    // ONLY EFFECTS MOZILLA BROWSERS
    //   If an ajax request is currently being processed and the user
    //   attempts to navigate else where, then this error is thrown
    //   (Mozilla).
    if (exception.name === 'NS_ERROR_NOT_AVAILABLE') {
      return;
    }

    // Default message if not set
    if (msg === null) {
      msg = exception.name + ": " + exception.message;
    }

    //Issue 13517 - hook for profile to try and handle the error more gracefully
    try {
      if (hx.errorInvokeCode !== null) {
        hx.lastException = exception;
        hx.lastExceptionDetails = details;
        hx.lastExceptionMessage = msg;
        if (Hx.$eval(hx.errorInvokeCode)) {
          return;
        }
      }
    } catch (err) {} finally {
      hx.lastException = null;
      hx.lastExceptionDetails = null;
      hx.lastExceptionMessage = null;
    }
    var fileName = "undefined";
    var lineNumber = "undefined";
    var stack = "undefined";
    try {
      fileName = exception.fileName;
      lineNumber = exception.lineNumber;
      stack = exception.stack;
    } catch (err) {
      // If this throws an exception, we got one of those
      // internal exception things in Mozilla
      stack = exception;
    }
    while (document.body.childNodes.length > 0) {
      document.body.removeChild(document.body.firstChild);
    }

    // Force style so we don't get anything weird
    var style = document.body.style;
    style.color = "black";
    style.background = "white";
    style.font = "normal 11px Tahoma";
    style.padding = "10px";
    var html = "<div style='font:18px Tahoma; padding-bottom:5px;'>";
    html += "Cannot display page</div>";
    html += msg + "<br/><br/>";
    html += "<div style='color:blue; cursor:pointer; text-decoration:underline;'";
    html += "onclick='document.getElementById(\"details\").";
    html += "style.visibility=\"visible\";'>";
    html += "Show Details</div>";
    html += "<div id='details' style='visibility:hidden; margin-top:10px;'>";

    // Exception information
    html += "<table width='100%' cellspacing='0' cellpadding='3'";
    html += " style='border:1px solid #666;'>";
    html += "<tr>";
    html += " <td colspan='2' style='background:#ccc;'>";
    html += "  <b>" + exception.name + ": " + exception.message + "</b>";
    html += " </td>";
    html += "</tr>";
    html += "<tr>";
    html += " <td style='background:#ddd;'><b>File:</b></td>";
    html += " <td width='100%' style='background:#eee;'>" + fileName + "</td>";
    html += "</tr>";
    html += "<tr>";
    html += " <td style='background:#ddd;'><b>Line:</b></td>";
    html += " <td style='background:#eee;'>" + lineNumber + "</td>";
    html += "</tr>";
    html += "<tr>";
    html += " <td valign='top' style='background:#ddd;'><b>Stack:</b></td>";
    html += " <td style='background:#eee;'>" + stack + "</td>";
    html += "</tr>";
    html += "</table>";

    // Details if they exist
    if (details !== null && details.length > 0) {
      html += "<div style='margin-top:10px; padding:5px; background:#eee; ";
      html += "border:1px solid #666;'>";
      html += details;
      html += "</div>";
    }
    html += "</div>";
    document.body.innerHTML = html;
    hx.failure = true;
  };

  /**
   * Add a Javascript resources after the page has loaded.
   * @param {String|Array.<String>} uris uri(s) for the resource
   * @param {Function} [complete] optional callback for when Javascript has completed loading
   */
  this.addJavaScript = function (uris, complete) {
    if (!(uris instanceof Array)) {
      hx.$addSingleJavaScript(uris, complete);
      return;
    }
    require(["Promise"], function (Promise) {
      var promises = [];
      uris.forEach(function (uri) {
        // eslint-disable-next-line promise/avoid-new
        promises.push(new Promise(function (resolve) {
          hx.$addSingleJavaScript(uri, function () {
            resolve();
          });
        }));
      });
      Promise.all(promises).then(function () {
        if (complete) {
          complete();
        }
      })["catch"](function (ignore) {});
    });
  };

  /**
   * Add a single Javascript resources after the page has loaded.
   * @param {String} uri(s) for the resource
   * @param {Function} [complete] optional callback for when Javascript has completed loading
   * @private
   */
  this.$addSingleJavaScript = function (uri, complete) {
    var head = document.getElementsByTagName("head")[0];
    var tag = "script";
    var type = "text/javascript";
    var links = head.getElementsByTagName(tag);
    var hasLink = false;
    for (var i = 0; i < links.length; i++) {
      if (links[i].getAttribute("src") === uri) {
        hasLink = true;
      }
    }
    if (!hasLink) {
      var domResource = document.createElement(tag);
      domResource.type = type;
      domResource.setAttribute("src", uri);
      if (complete) {
        domResource.onload = complete;
      }
      head.appendChild(domResource);
    } else if (complete) {
      complete(); //document is already loaded, run now
    }
  };

  /**
   * Add a Css StyleSheet resource after the page has loaded.
   * @param {String} uri for the resource
   */
  this.addStyleSheet = function (uri) {
    var head = document.getElementsByTagName('head')[0];
    var tag = 'link';
    var type = 'text/css';
    var links = head.getElementsByTagName(tag);
    var hasLink = false;
    for (var i = 0; i < links.length; i++) {
      if (links[i].getAttribute("href") === uri) {
        hasLink = true;
      }
    }
    if (!hasLink) {
      var domResource = document.createElement(tag);
      domResource.type = type;
      domResource.rel = 'stylesheet';
      domResource.setAttribute("href", uri);
      head.appendChild(domResource);
    }
  };

  /**
   * Used to add images and icons to an Hx page. This function adds support
   * for loading images from a sprite sheet.
   * This function requires that a unique element (like a span with an id)
   * exists in the DOM before calling this function. The element with the id
   * passed into this function will be replaced with a block of HTML that
   * correctly displays an image based on whether or not it was found in a
   * sprite sheet.
   *
   * @param {Array.<String>} ords array containing list of ord strings from a BImage ordList
   * @param {Array.<String>} ids array of ids of the dummy elements to replace with the HTML that iconUtils generates
   * @param {String} [alt] used as the alt and title text for the requested image
   * @param {String} [attrs] contains HTML attributes (excluding alt and title) to add to the generated image HTML
   * @returns {Promise}
   * @see module:bajaux/icon/iconUtils
   */
  this.makeImage = function (ords, ids, alt, attrs) {
    var that = this,
      Promise = that.getPromiseConstructor(),
      makeImagePromise = that.$makeImagePromise,
      changeImagePromise = that.$changeImagePromise;
    // NCCB-30122: ensure that makeImage() and changeImage() don't run at the same time
    that.$makeImagePromise = that.makePromise(function (resolve) {
      resolve(changeImagePromise);
    })
    // eslint-disable-next-line promise/no-return-in-finally
    ["finally"](function () {
      return Promise.all([doMakeImage(that, ords, ids, alt, attrs), makeImagePromise]);
    });
    return that.$makeImagePromise;
  };

  /**
   * Replace an image in Hx previously created with hx.makeImage() with another.
   * This function copies over the previous classes on the span and img tags onto
   * the new elements.
   *
   * @param {String} parentId id of parent element containing images generated with hx.makeImage()
   * @param {Array.<String>} ords array containing list of ord strings from a BImage ordList
   * @param {String} [attrs] contains HTML attributes (excluding alt and title) to add to the generated image HTML
   * @returns {Promise}
   * @see module:bajaux/icon/iconUtils
   */
  this.changeImage = function (parentId, ords, attrs) {
    var that = this,
      Promise = that.getPromiseConstructor(),
      makeImagePromise = that.$makeImagePromise,
      changeImagePromise = that.$changeImagePromise;
    // NCCB-30122: ensure that makeImage() and changeImage() don't run at the same time
    that.$changeImagePromise = that.makePromise(function (resolve) {
      resolve(makeImagePromise);
    })
    // eslint-disable-next-line promise/no-return-in-finally
    ["finally"](function () {
      return Promise.all([doChangeImage(that, parentId, ords, attrs), changeImagePromise]);
    });
    return that.$changeImagePromise;
  };
  function doMakeImage(hx, ords, ids, alt, attrs) {
    return hx.makePromise(function (resolve, reject) {
      require(['jquery', 'underscore', 'bajaux/icon/iconUtils'], function (jq, _, iconUtils) {
        return iconUtils.toHtml(ords).then(function (html) {
          _.each(jq(html), function (imageHtml, index) {
            var span = jq(imageHtml),
              id = ids[index];
            span.data('ord', ords[index]);
            span.attr('id', id);
            if (alt) {
              span.attr('title', alt);
              span.children('img').attr('alt', alt);
            }
            if (attrs) {
              var dummySpan = jq('<span ' + attrs + '/>');
              _.each(dummySpan[0].attributes, function (attr) {
                if (attr.name !== 'class') {
                  span.attr(attr.name, attr.value);
                }
              });
              _.each(dummySpan[0].className.split(/\s+/), function (cls) {
                span.addClass(cls);
              });
            }
            jq(hx.$(id)).replaceWith(span);
          });
        }).then(resolve, reject);
      }, reject);
    });
  }
  function doChangeImage(hx, parentId, ords, attrs) {
    return hx.makePromise(function (resolve, reject) {
      require(['jquery', 'underscore', 'bajaux/icon/iconUtils'], function (jq, _, iconUtils) {
        var parent = jq(hx.$(parentId));
        if (!parent.length) {
          return resolve();
        }
        var wrapper = parent.find('.hxImageWrapper');
        if (!wrapper.length) {
          return resolve();
        }
        var spans = wrapper.children('span');
        if (!spans.length) {
          return resolve();
        }
        var old = spans.first(),
          alt = old.children('img').prop('alt'),
          oldAttrs = old[0].attributes,
          classes = old[0].className.split(/\s+/);
        return iconUtils.toHtml(ords).then(function (html) {
          _.each(spans, function (image) {
            jq(image).remove();
          });
          _.each(jq(html), function (imageHtml, index) {
            var span = jq(imageHtml);
            span.data('ord', ords[index]);
            var img = span.children('img');
            if (alt) {
              img.prop('alt', alt);
            }

            //hide the image instead of showing broken image
            img.on("error", function () {
              img.css('display', 'none');
            });
            if (oldAttrs.length) {
              _.each(oldAttrs, function (attr) {
                if (attr.name !== 'class') {
                  span.attr(attr.name, attr.value);
                }
              });
            }
            if (classes.length) {
              _.each(classes, function (cls) {
                if (cls.indexOf('icon-') !== 0) {
                  span.addClass(cls);
                }
              });
            }
            if (attrs) {
              var dummySpan = jq('<span ' + attrs + '/>');
              _.each(dummySpan[0].attributes, function (attr) {
                if (attr.name !== 'class') {
                  span.attr(attr.name, attr.value);
                }
              });
              _.each(dummySpan[0].className.split(/\s+/), function (cls) {
                span.addClass(cls);
              });
            }
            wrapper.append(span);
          });
        }).then(resolve, reject);
      }, reject);
    });
  }
  this.setAlphaImageLoader = function (elem) {
    elem.style.backgroundColor = "transparent";
    elem.style.filter = "progid:DXImageTransform.Microsoft.AlphaImageLoader(src='/ord?" + elem.ord + "')";
  };

  //touch support
  this.touchCurrentX = null;
  this.touchCurrentY = null;
  this.touchStartElem = null;

  /**
   * @param {HTMLElement} elem
   */
  this.addTouchScroll = function (elem) {
    if (hx.ie) {
      elem.attachEvent("ontouchstart", hx.touchStart);
      elem.attachEvent("ontouchmove", hx.touchMove);
      elem.attachEvent("ontouchend", hx.touchEnd);
    } else {
      elem.addEventListener("touchstart", hx.touchStart, false);
      elem.addEventListener("touchmove", hx.touchMove, false);
      elem.addEventListener("touchend", hx.touchEnd, false);
    }
  };
  this.touchStart = function (event) {
    var elem = hx.currentElemFromEvent(event);
    if (elem === null) {
      return;
    }
    if (hx.needsScroll(elem) && (hx.touchStartElem === null || hx.touchStartElem === elem || hx.touchStartElem.id.length < elem.id.length)) {
      hx.touchCurrentX = hx.clientX(event);
      hx.touchCurrentY = hx.clientY(event);
      hx.touchStartElem = elem;
    }
  };
  this.needsScroll = function (elem) {
    return elem.scrollHeight > elem.offsetHeight || elem.scrollWidth > elem.offsetWidth;
  };
  this.touchMove = function (event) {
    var elem = hx.currentElemFromEvent(event);
    if (elem === null) {
      return;
    }
    if (hx.touchStartElem === elem) {
      hx.touchStartElem.scrollLeft += hx.touchCurrentX - hx.clientX(event);
      hx.touchStartElem.scrollTop += hx.touchCurrentY - hx.clientY(event);
      hx.touchCurrentX = hx.clientX(event);
      hx.touchCurrentY = hx.clientY(event);
      hx.endEvent(event);
    }
  };
  this.touchEnd = function (event) {
    var elem = hx.currentElemFromEvent(event);
    if (elem === null) {
      return;
    }
    if (hx.touchStartElem === elem) {
      hx.touchStartElem = null;
    }
  };
  this.endEvent = function (event) {
    if (event === null) {
      return false;
    }
    if (event.stopPropagation) {
      event.stopPropagation();
    } else {
      event.cancelBubble = true;
    }
    event.returnValue = false;
    return false;
  };
  this.clientX = function (event) {
    if (event.touches !== null && event.touches.length === 1) {
      return event.touches[0].clientX;
    } else {
      return event.clientX;
    }
  };
  this.clientY = function (event) {
    if (event.touches !== null && event.touches.length === 1) {
      return event.touches[0].clientY;
    } else {
      return event.clientY;
    }
  };

  //find the element that attached the listener for the event
  this.currentElemFromEvent = function (event) {
    if (hx.ie) {
      return null; //not supported
    } else {
      return event.currentTarget;
    }
  };

  //find the most specific element for the event
  this.elemFromEvent = function (event) {
    if (hx.ie) {
      return event.srcElement;
    } else {
      return event.target;
    }
  };
  function isValidLocation(popupWindow) {
    try {
      return popupWindow.location && popupWindow.location.href;
    } catch (err) {}
    return true;
  }

  /**
   * Open a popup to the specified dimensions and position. If dimension information is not provided, will default
   * to 90% of screen size. Make sure to set external=true for links to non-niagara websites like https://www.google.com.
   * If the link is not external, fullScreen=true will be added to the view parameters to ensure that the HxProfile chrome
   * is not present in the popup.
   * @param {String} url Starting in Niagara 4.10U7 url will be converted via `niagara.env.toHyperlink`.
   * @param {Number} [popupWidth]
   * @param {Number} [popupHeight]
   * @param {Number} [popupX]
   * @param {Number} [popupY]
   * @param {Boolean} [external] If true, will not add the fullScreen=true view parameter to the url.
   * @param {Boolean} [modal] If true, the current pop-up can only be opened one at a time.
   * @param {String} [title] As of Niagara 4.10, the initial title on the pop-up can be set.
   * @return {Promise.<Boolean>} resolve to true if a new window was opened.
   */
  this.popup = function (url, popupX, popupY, popupWidth, popupHeight, external, modal, title) {
    return hx.makePromise(function (resolve, reject) {
      require(["Promise", "nmodule/js/rc/asyncUtils/asyncUtils"], function (Promise, asyncUtils) {
        return asyncUtils.doRequire("nmodule/hx/rc/container/hxContainer").then(function () {
          return window.niagara.env.toHyperlink(url);
        }).then(function (url) {
          return resolve(hx.$popup(url, popupX, popupY, popupWidth, popupHeight, external, modal, title));
        })["catch"](reject);
      });
    });
  };

  /**
   * Open a popup to the specified dimensions and position. If dimension information is not provided, will default
   * to 90% of screen size. Make sure to set external=true for links to non-niagara websites like https://www.google.com.
   * If the link is not external, fullScreen=true will be added to the view parameters to ensure that the HxProfile chrome
   * is not present in the popup.
   * @private
   * @param {String} url
   * @param {Number} [popupWidth]
   * @param {Number} [popupHeight]
   * @param {Number} [popupX]
   * @param {Number} [popupY]
   * @param {Boolean} [external] If true, will not add the fullScreen=true view parameter to the url.
   * @param {Boolean} [modal] If true, the current pop-up can only be opened one at a time.
   * @param {String} [title] As of Niagara 4.10, the initial title on the pop-up can be set.
   * @return {Boolean} return true if a new window was opened.
   */
  this.$popup = function (url, popupX, popupY, popupWidth, popupHeight, external, modal, title) {
    if (popupWidth === undefined) {
      var factor = 0.9;
      popupWidth = Math.floor(screen.width * factor);
      popupHeight = Math.floor(screen.height * factor);
      popupX = (screen.width - popupWidth) / 2;
      popupY = (screen.height - popupHeight) / 8; //allow extra space for toolbar and status bar
    }
    var fullUrl = url + (external ? "" : "|view:?fullScreen=true"),
      that = this;
    var win,
      specs = "resizable=yes, " + "location=no, " + "scrollbars=yes, " + "status=no, " + "toolbar=no, " + "left=" + popupX + ", " + "top=" + popupY + ", " + "width=" + popupWidth + ", " + "height=" + popupHeight;
    var pop = that.$modalPopups[fullUrl];
    try {
      if (pop && !pop.closed && isValidLocation(pop)) {
        //if pop-up is modal then only one mobile dialog at a time.
        pop.focus();
        return false;
      }
    } catch (err) {
      //if an error is provided, its likely that the page is still open and the site
      //prevents a cross-origin frame. When the page is closed, there should be no error
      window.console.log(err);
      return false;
    }
    win = window.open(fullUrl, 'fullScreen' + new Date().getTime(), specs, false);
    if (win) {
      if (title) {
        try {
          win.addEventListener('load', function () {
            win.document.title = title;
          });
        } catch (err) {}
      }
    } else {
      window.console.log("Pop-ups must be enabled...");
      return false;
    }

    //cleanup old modal dialogs
    Object.keys(that.$modalPopups).forEach(function (name) {
      var oldPop = that.$modalPopups[name];
      if (oldPop && oldPop.closed) {
        //references to closed modal dialogs are no longer required
        delete that.$modalPopups[name];
      }
    });
    if (modal) {
      that.$modalPopups[fullUrl] = win;
    }
    return true;
  };

  /**
   * Tell the window to be fullScreen.
   * @param {boolean} fullScreen
   */
  this.setFullScreen = function (fullScreen) {
    this.$fullScreen = fullScreen;
  };

  /**
   * Return true if window is fullScreen.
   * @returns {boolean}
   */
  this.isFullScreen = function () {
    return this.$fullScreen;
  };

  /**
   * JQuery selectors need escaping for the following characters if used with a className or id:
   * !"#$%&'()*+,./:;<=>?@[\]^`{|}~
   * @see {@link http://api.jquery.com/category/selectors/}
   */
  this.escapeSelector = function (str) {
    if (str) {
      return str.replace(/([ !"#%&$'()*+,./:;<=>?@[\]^`{|}~])/g, '\\$1');
    }
    return str;
  };

  /**
   * Reverse any escaping done from 'hx.escapeSelector' by removing the extra '\' in front all escape characters:
   * !"#$%&'()*+,./:;<=>?@[\]^`{|}~
   * @see hx.escapeSelector
   */
  this.unescapeSelector = function (str) {
    if (str) {
      return str.replace(/(\\)([ !"#%&$'()*+,./:;<=>?@[\]^`{|}~])/g, '$2');
    }
    return str;
  };

  /*
   * Array of functions that will return a promise to wait for completing before hx.loadTime returns the total time of the page load
   */
  this.loadTimeArray = [];

  /**
   * Wait 2 seconds before attempting to collect data for load time
   */
  this.$loadTimeCollectionDelay = 2000;

  /**
   * When log is enabled, let the server know how long the page took to load. HxViews and HxProfiles can add to the
   * pageLoad Array to extend the amount of time it takes to consider the page fully loaded by adding to the
   * 'hx.loadTimeArray' or calling 'hx.waitForLoadTime'.
   * @param {String} eventId
   */
  this.loadTime = function (eventId) {
    if (!window.performance || !window.performance.timing) {
      return; //browser does not support window.performance.timing
    }
    require(["Promise"], function (Promise) {
      setTimeout(function () {
        if (hx.loadTimeArray.length > 0) {
          var _waitForPromises = function waitForPromises() {
            var promises = [];
            hx.loadTimeArray.forEach(function (pageLoadFunction) {
              promises.push(pageLoadFunction());
            });
            Promise.all(promises).then(function (times) {
              //wait for any new promises too
              if (promises.length !== hx.loadTimeArray.length) {
                _waitForPromises();
                return;
              }
              var maxDate = new Date(Math.max.apply(null, times));
              var total = maxDate - window.performance.timing.navigationStart;
              window.console.log("loadTime: " + total + " ms");
              hx.setFormValue('loadTime', total);
              hx.fireEvent("", eventId);
            })["catch"](function (err) {
              window.console.log(err);
            });
          };
          _waitForPromises();
        } else {
          var total = window.performance.timing.loadEventEnd - window.performance.timing.navigationStart;
          window.console.log("loadTime: " + total + " ms");
          hx.setFormValue('loadTime', total);
          hx.fireEvent("", eventId);
        }
      }, hx.$loadTimeCollectionDelay);
    });
  };

  /**
   *
   * Get the top window that doesn't have cross-origin frame problems.
   * @param {boolean} requiresHx if true, also ensure hx is available in that window.
   * @returns {object} the top available window
   */
  this.topWindow = function (requiresHx) {
    var bestWindow = window;
    try {
      var wnd = window;
      while (wnd !== wnd.parent && (wnd = wnd.parent)) {
        if (wnd.location.href && (!requiresHx || wnd.hx)) {
          bestWindow = wnd;
        }
      }
    } catch (ignore) {
      //don't use the window when there are cross-origin frame problems
    }
    return bestWindow;
  };

  /**
   * @returns {object} the top available window with hx.
   */
  this.topHx = function () {
    return this.topWindow(true).hx;
  };

  /**
   * For benchmarking pageLoad, wait for something to be ready before allowing the page to be deemed complete.
   * The ready function is checked every 1 second for readiness, but the time reported will be the Date
   * resolved from the endTimeFunction.
   * @param {Function} readyFunction When a truthy value is returned the benchmark is ready
   * @param {Function} [endTimeFunction] If no endTimeFunction is provide, use the readyFunction
   */
  this.waitForLoadTime = function (readyFunction, endTimeFunction) {
    //report to the top window in the hierarchy if this is an iframe
    var bestHx = this.topHx();
    require(["Promise"], function (Promise) {
      bestHx.loadTimeArray.push(function () {
        // eslint-disable-next-line promise/avoid-new
        return new Promise(function (resolve) {
          function checkLoadTime() {
            var result = readyFunction();
            if (!result) {
              setTimeout(function () {
                checkLoadTime();
              }, 1000);
              return;
            }
            if (endTimeFunction) {
              resolve(endTimeFunction()); //resolves with a specific endTime
            } else {
              resolve(result); //resolves without end time (stop waiting)
            }
          }
          checkLoadTime();
        });
      });
    });
  };

  /**
   * @param {Function} func function to run when Promise is available
   * @returns {Promise}
   */
  this.makePromise = function (func) {
    var Promise = this.getPromiseConstructor();
    // eslint-disable-next-line promise/avoid-new
    return new Promise(func);
  };

  /**
   * @returns {Function} promise constructor
   */
  this.getPromiseConstructor = function () {
    if (require.defined('Promise')) {
      return require('Promise');
    } else {
      throw new Error('Cannot find Promise implementation');
    }
  };
}
Hx.$eval = function (text) {
  'use strict';

  // eslint-disable-next-line no-eval
  return window.eval(text);
};
////////////////////////////////////////////////////////////////
// XmlHttp
////////////////////////////////////////////////////////////////

function Message() {
  "use strict";

  var headers = [];
  var responseHandler = null;
  var xmlhttp = hx.ie ? new ActiveXObject("Msxml2.XMLHTTP") : new XMLHttpRequest();

  /**
   * Set the specified HTTP header.
   */
  this.setHeader = function (name, value) {
    headers.push({
      name: name,
      value: value
    });
  };

  /**
   * Send a message.
   */
  this.send = function (url, body, handler) {
    try {
      responseHandler = handler;
      xmlhttp.open("post", url, true);
      for (var i = 0; i < headers.length; i++) {
        var header = headers[i];
        xmlhttp.setRequestHeader(header.name, header.value);
      }

      // 24 Aug 07 - AndyF: Added for IE cache issues (when using a proxy server).
      if (hx.ie) {
        xmlhttp.setRequestHeader("If-Modified-Since", "Sat, 1 Jan 2000 00:00:00 GMT");
      }
      xmlhttp.onreadystatechange = this.handleResponse;
      xmlhttp.send(body);
    } catch (err) {
      hx.doError("Message error", null, err);
    }
  };

  /**
   * Handle request response.
   */
  this.handleResponse = function () {
    try {
      if (xmlhttp.readyState === 4) {
        if (xmlhttp.status !== 200) {
          var text = xmlhttp.responseText;
          if (text.length === 0) {
            text = hx.err_LostConnection;
          }
          hx.doError(text, null, null);
        } else if (responseHandler !== null) {
          responseHandler(xmlhttp);
        }

        // 23 Oct 07 - AndyF: make sure we remove circular
        // dependencies to prevent memory leaks in IE
        xmlhttp.onreadystatechange = function () {};
        xmlhttp = null;
      }
    } catch (err) {
      // 23 Oct 07 - AndyF: make sure we remove circular
      // dependencies to prevent memory leaks in IE
      xmlhttp.onreadystatechange = function () {};
      xmlhttp = null;
      hx.doError(hx.err_LostConnection, null, err);
    }
  };
}

////////////////////////////////////////////////////////////////
// StringUtil
////////////////////////////////////////////////////////////////

var StringUtil = {};
/**
 * Return true if str starts with prefix.
 */
StringUtil.startsWith = function (str, prefix) {
  "use strict";

  // If prefix longer than str, not possible
  if (str.length < prefix.length) {
    return false;
  }

  // Cut off and compare
  var sub = str.substring(0, prefix.length);
  return sub === prefix;
};

/**
 * Return true if str ends with suffix.
 */
StringUtil.endsWith = function (str, suffix) {
  "use strict";

  // If suffix is longer, not possible
  if (str.length < suffix.length) {
    return false;
  }

  // Cut off and compare
  var sub = str.substring(str.length - suffix.length);
  return sub === suffix;
};

/**
 * Trim leading and trailing whitespace from the given
 * string.  Return a new string as result. The original
 * string is not modified.
 */
StringUtil.trim = function (str) {
  "use strict";

  return str;
};
if (typeof define === "function" && define.amd) {
  define("hx", [], function () {
    "use strict";

    return hx;
  });
}
