/*
 * Copyright 2007 Tridium, Inc. All Rights Reserved.
 */

/* eslint-env browser */
/* globals jQuery: false, hx: true, d3: false */

////////////////////////////////////////////////////////////////
// Px

////////////////////////////////////////////////////////////////
// global instance
var px = new Px();
function Px() {
  "use strict";

  function logError(err) {
    if (typeof console !== 'undefined' && console && typeof console.error === 'function') {
      console.error(err);
    }
  }

  /*
   * Sets the given styles on the dom element with the given id.
   *
   * @param {String} id - id of the dom to set styles on
   * @param {Array.<{key: String, value: String, decode: Boolean}>} styles - An
   * array of object literals, each containing a key/value pair for the style to
   * be set, as well as an optional decode property to state whether or not the
   * value needs to be URIDecoded. If decode is undefined, no decoding will take
   * place.
   */
  this.setStyle = function (id, styles) {
    var elem = document.getElementById(id);
    if (!elem) {
      return;
    }
    var elemStyle = elem.style;
    styles.forEach(function (style) {
      var key = style.key,
        value = style.value,
        decode = style.decode;
      try {
        if (key in elemStyle) {
          elemStyle[key] = decode ? decodeURIComponent(value) : value;
        }
      } catch (err) {
        err.message = "Invalid value '" + value + "' for style " + id + "." + key + ". " + err.message;
        logError(err);
      }
    });
  };

  /*
   * Sets the given properties on the dom element with the given id.
   *
   * @param {String} id - id of the dom to set properties on
   * @param {Array.<{key: String, value: String, decode: Boolean}>} properties -
   * An array of object literals, each containing a key/value pair for the
   * property to be set, as well as an optional decode property to state whether
   * or not the value needs to be URIDecoded. If decode is undefined, no
   * decoding will take place.
   */
  this.setProperties = function (id, properties) {
    var elem = document.getElementById(id);
    if (!elem) {
      return;
    }
    properties.forEach(function (property) {
      var key = property.key,
        value = property.value,
        decode = property.decode;
      try {
        elem[key] = decode ? decodeURIComponent(value) : value;
      } catch (err) {
        err.message = "Invalid value '" + value + "' for property " + id + "." + key + ". " + err.message;
        logError(err);
      }
    });
  };

  /*
   * Sets the given events on the dom element with the given id.
   *
   * @param {String} id - id of the dom to set properties on
   * @param {Array.<{key: String, value: String, decode: Boolean}>} events - An
   * array of object literals, each containing a key/value pair for the event to
   * be set, as well as an optional decode property to state whether or not the
   * value needs to be URIDecoded. If decode is undefined, no decoding will take
   * place.
   */
  this.setEvents = function (id, events) {
    var elem = document.getElementById(id);
    if (!elem) {
      return;
    }
    events.forEach(function (event) {
      var key = event.key,
        value = event.value,
        decode = event.decode;
      try {
        value = decode ? decodeURIComponent(value) : value;
        // eslint-disable-next-line no-eval
        eval("elem." + key + "=function(event){if (!event) {event = window.event;}" + value + "};");
      } catch (err) {
        err.message = "Invalid value for event " + id + "." + key;
        logError(err);
      }
    });
  };

  // Move an object from on location in the DOM to another
  this.move = function (id, dest) {
    try {
      document.getElementById(dest).appendChild(document.getElementById(id));
    } catch (err) {
      err.message = "Error moving " + id + " to " + dest + ". " + err.message;
      logError(err);
    }
  };

  // Get the screen height
  // @deprecated Use hx.getScreenHeight() instead
  this.getHeight = function () {
    // eslint-disable-next-line no-unused-vars
    var myWidth = 0,
      myHeight = 0;
    if (typeof window.innerWidth === 'number') {
      //Non-IE
      myWidth = window.innerWidth;
      myHeight = window.innerHeight;
    } else if (document.documentElement && (document.documentElement.clientWidth || document.documentElement.clientHeight)) {
      //IE 6+ in 'standards compliant mode'
      myWidth = document.documentElement.clientWidth;
      myHeight = document.documentElement.clientHeight;
    } else if (document.body && (document.body.clientWidth || document.body.clientHeight)) {
      //IE 4 compatible
      // eslint-disable-next-line no-unused-vars
      myWidth = document.body.clientWidth;
      myHeight = document.body.clientHeight;
    }
    return myHeight;
  };
  var blinkFunction = function blinkFunction() {
    var px = window.px;

    // visible for 600 ms, hidden for 200 ms
    px.blinkCounter = ++px.blinkCounter % 8;
    px.elementVisibility = px.blinkCounter >= 2 ? 'inherit' : 'hidden';
    for (var i = 0; i < px.blinkingElements.length; i++) {
      if (document.body.contains(px.blinkingElements[i])) {
        px.blinkingElements[i].style.visibility = px.elementVisibility;
      } else {
        // remove element from array if no longer found in DOM
        px.blinkingElements.splice(i, 1);
        if (px.blinkingElements.length === 0) {
          clearInterval(px.blinkInterval);
          px.blinkInterval = null;
          px.blinkCounter = 0;
          px.elementVisibility = 'hidden';
          return;
        }
        i--;
      }
    }
  };

  // Make DOM elements start or stop blinking
  this.toggleBlink = function (elements, blink) {
    if (!blink && this.blinkingElements.length === 0) {
      return;
    }
    for (var i = 0; i < elements.length; i++) {
      if (!blink && elements[i]) {
        var index = this.blinkingElements.indexOf(elements[i]);
        if (index > -1) {
          this.blinkingElements.splice(index, 1);
          if (this.blinkingElements.length === 0) {
            clearInterval(this.blinkInterval);
            this.blinkInterval = null;
            this.blinkCounter = 0;
            this.elementVisibility = 'hidden';
          }
          elements[i].style.visibility = 'inherit';
        }
      } else {
        if (!elements[i] || this.blinkingElements.indexOf(elements[i]) > -1) {
          continue;
        }

        // sync newly blinking element up with the rest of the blinking elements
        elements[i].style.visibility = this.elementVisibility;
        this.blinkingElements.push(elements[i]);
        if (!this.blinkInterval && this.blinkingElements.length === 1) {
          this.blinkInterval = setInterval(blinkFunction, 100);
        }
      }
    }
  };
  this.blinkingElements = [];
  this.blinkInterval = null;
  this.blinkCounter = 0;
  this.elementVisibility = 'hidden';

  ////////////////////////////////////////////////////////////////
  // HxPxSplitPane
  ////////////////////////////////////////////////////////////////
  this.splitPane = new SplitPane();
  function SplitPane() {
    this.dragObj = {};
    this.valueObj = {};
    // Init dragging the split divider
    this.doStartDrag = function (event, id, valueId, splitPane) {
      var x, y;
      px.splitPane.dragObj = {};
      px.splitPane.valueObj = {};
      px.splitPane.dragObj.splitPane = !!splitPane; //need different vertical positioning behavior on splitPane vs slider
      px.splitPane.dragObj.elNode = document.getElementById(id);
      if (valueId !== null) {
        px.splitPane.valueObj.elNode = document.getElementById(valueId);
      } else {
        px.splitPane.valueObj = null;
      }
      px.splitPane.dragObj.scale = px.getScalingDimensions(px.splitPane.dragObj.elNode);
      var cursorPosition = hx.getMousePosition(event);
      x = cursorPosition[0] / px.splitPane.dragObj.scale[0];
      y = cursorPosition[1] / px.splitPane.dragObj.scale[1];

      // Save starting positions of cursor and element.
      px.splitPane.dragObj.cursorStartX = x;
      px.splitPane.dragObj.cursorStartY = y;
      px.splitPane.dragObj.elStartLeft = parseInt(px.splitPane.dragObj.elNode.style.left, 10);
      px.splitPane.dragObj.elStartTop = parseInt(px.splitPane.dragObj.elNode.style.top, 10);
      if (isNaN(px.splitPane.dragObj.elStartLeft)) {
        px.splitPane.dragObj.elStartLeft = 0;
      }
      if (isNaN(px.splitPane.dragObj.elStartTop)) {
        px.splitPane.dragObj.elStartTop = 0;
      }
      document.addEventListener("mousemove", px.splitPane.doDrag, true);
      document.addEventListener("mouseup", px.splitPane.doStopDrag, true);
      document.addEventListener("touchmove", px.splitPane.doDrag, true);
      document.addEventListener("touchend", px.splitPane.doStopDrag, true);
      document.addEventListener("touchcancel", px.splitPane.doStopDrag, true);
      event.preventDefault();
    };

    // Drag the split divider
    this.doDrag = function (event) {
      var cursorPosition = hx.getMousePosition(event),
        x = cursorPosition[0] / px.splitPane.dragObj.scale[0],
        y = cursorPosition[1] / px.splitPane.dragObj.scale[1],
        position;

      // Move drag element by the same amount the cursor has moved.
      var maxPosition = parseInt(px.splitPane.dragObj.elNode.dividerMax);
      var orient = px.splitPane.dragObj.elNode.dividerOrient;
      if (orient === "horiz") {
        position = px.splitPane.dragObj.elStartLeft + x - px.splitPane.dragObj.cursorStartX;
        position = Math.min(maxPosition, position);
        position = Math.max(0, position);
        px.splitPane.dragObj.elNode.style.left = position + "px";
      } else {
        position = px.splitPane.dragObj.elStartTop + y - px.splitPane.dragObj.cursorStartY;
        position = Math.min(maxPosition, position);
        position = Math.max(0, position);
        px.splitPane.dragObj.elNode.style.top = position + "px";
      }
      event.preventDefault();
    };

    // Stop dragging the split divider
    this.doStopDrag = function (event) {
      var mousePosition = hx.getMousePosition(event),
        x = mousePosition[0] / px.splitPane.dragObj.scale[0],
        y = mousePosition[1] / px.splitPane.dragObj.scale[1],
        position,
        formPosition;
      var maxPosition = parseInt(px.splitPane.dragObj.elNode.dividerMax);
      var orient = px.splitPane.dragObj.elNode.dividerOrient;
      if (orient === "horiz") {
        position = px.splitPane.dragObj.elStartLeft + x - px.splitPane.dragObj.cursorStartX;
        position = Math.min(maxPosition, position);
        position = Math.max(0, position);
        px.splitPane.dragObj.elNode.style.left = position + "px";
        if (px.splitPane.valueObj !== null) {
          px.splitPane.valueObj.elNode.style.left = position + "px";
        }
        formPosition = parseInt(position / maxPosition * 100);
      } else {
        position = px.splitPane.dragObj.elStartTop + y - px.splitPane.dragObj.cursorStartY;
        position = Math.min(maxPosition, position);
        position = Math.max(0, position);
        px.splitPane.dragObj.elNode.style.top = position + "px";
        if (px.splitPane.valueObj !== null) {
          px.splitPane.valueObj.elNode.style.top = position + "px";
        }
        if (px.splitPane.dragObj.splitPane) {
          formPosition = parseInt(position / maxPosition * 100);
        } else {
          formPosition = 100 - parseInt(position / maxPosition * 100);
        }
      }
      hx.setFormValue(px.splitPane.dragObj.elNode.id + '.position', formPosition);
      px.splitPane.dragObj.elNode = null;
      document.removeEventListener("mousemove", px.splitPane.doDrag, true);
      document.removeEventListener("mouseup", px.splitPane.doStopDrag, true);
      document.removeEventListener("touchmove", px.splitPane.doDrag, true);
      document.removeEventListener("touchend", px.splitPane.doStopDrag, true);
      document.removeEventListener("touchcancel", px.splitPane.doStopDrag, true);
      hx.poll();
    };
  }

  ////////////////////////////////////////////////////////////////
  // Resizing Utilities - @since Niagara 3.6   
  //////////////////////////////////////////////////////////////// 

  this.resizeTimeout = null;

  /**
   * Call 'doResize' in 500 ms. If multiple calls to 'resize' occur within the next 500 ms, this function will wait to
   * call 'doResize' until the calls to 'resize' have stopped. If available, provide the initial dimensions in which
   * the HxPxView write method used so we can sometimes prevent unneeded updates.
   * @param {String} scope
   * @param {Number} [initialScreenWidth]
   * @param {Number} [initialScreenHeight]
   * @since Niagara 3.6
   */
  this.resize = function (scope, initialScreenWidth, initialScreenHeight) {
    if (this.resizeTimeout !== null) {
      clearTimeout(this.resizeTimeout);
    }
    this.resizeTimeout = setTimeout(function () {
      px.doResize(scope, initialScreenWidth, initialScreenHeight);
    }, 500);
  };

  /**
   * Actually complete the resize.
   * @param {String} scope
   * @param {Number} [initialScreenWidth]
   * @param {Number} [initialScreenHeight]
   */
  this.doResize = function (scope, initialScreenWidth, initialScreenHeight) {
    px.resizeTimeout = null;
    document.body.style.overflow = 'hidden';
    var newWidth = hx.getScreenWidth(),
      newHeight = hx.getScreenHeight(),
      oldWidth = hx.screenWidth,
      oldHeight = hx.screenHeight,
      useInitial = false;
    if (!oldWidth) {
      oldWidth = initialScreenWidth;
      useInitial = true;
    }
    if (!oldHeight) {
      oldHeight = initialScreenHeight;
      useInitial = true;
    }
    var updateNow = newWidth !== oldWidth || newHeight !== oldHeight;
    document.body.style.overflow = 'auto';
    if (updateNow) {
      hx.setFormValue(scope, 0);
      hx.screenHeight = newHeight;
      hx.screenWidth = newWidth;
      hx.poll();
    } else if (useInitial) {
      //make sure to update screenHeight and width for next time.
      hx.screenHeight = newHeight;
      hx.screenWidth = newWidth;
    }
  };

  /**
   *  @since Niagara 3.6
   */
  this.fixResize = function () {
    var elem = document.body;
    if (elem.offsetHeight === elem.scrollHeight && elem.offsetWidth === elem.scrollWidth) {
      elem.style.overflow = 'hidden';
    } else {
      elem.style.overflow = 'auto';

      //NCCB-27068: sometimes this changes fixes scroll bars, no need to wait for the next update and cause 'shifting'
      if (elem.offsetHeight === elem.scrollHeight && elem.offsetWidth === elem.scrollWidth) {
        elem.style.overflow = 'hidden';
      }
    }
  };

  ////////////////////////////////////////////////////////////////
  //BPicture Scaling Utilities - @since Niagara 3.8
  ////////////////////////////////////////////////////////////////

  /**
   * Calculate the Image Dimensions
   * @param {String} imgSrc The Image Source
   * @param {Function} handleDims The function that gets called when the dimensions are known
   * @param {boolean} [extraCheck] When false, will attempt another dimension lookup if IE11 fails to get the correct
   * width and height.
   */
  function calculateImageDimensions(imgSrc, handleDims, extraCheck) {
    var img = document.createElement('IMG');
    img.style.visibility = false;
    img.onload = function () {
      if (this.width === 28 && this.height === 30 && !extraCheck) {
        //NCCB-16001: IE11 does not show svg images in BPicture in hxpx
        // 28 x 30 means that 'image might not yet be available' for IE11
        // https://github.com/davidjbradshaw/image-map-resizer/issues/10
        calculateImageDimensions(imgSrc, handleDims, true);
        return;
      } else {
        handleDims(this.width, this.height);
      }
      img.onload = null;
      if (img.parentNode === document.body) {
        document.body.removeChild(img);
      }
      img = null;
    };
    img.setAttribute('src', imgSrc);
    document.body.appendChild(img);
  }

  /**
   * Updates the src and layout of the Picture's <code>&lt;img&gt;</code> tag.
   * Called whenever there is an update to the Picture's <code>image</code> or
   * <code>scale</code> properties.
   * @since Niagara 3.8
   * @param {String} imgSrc
   * @param {String} imgId
   * @param {String} scale
   * @param {Number} picWidth
   * @param {Number} picHeight
   * @param {String} halign
   * @param {String} valign
   * @return {Promise}
   */
  this.updatePictureImage = function updatePictureImage(imgSrc, imgId, scale, picWidth, picHeight, halign, valign) {
    return hx.makePromise(function (resolve) {
      require(["jquery"], function ($) {
        var imgDom = document.getElementById(imgId),
          imgStyle = imgDom && imgDom.style,
          imgJq = $(imgDom);
        if (!imgStyle) {
          resolve();
          return;
        }
        if (!imgSrc) {
          /*
           * no image to show.
           */
          imgStyle.display = 'none';
          resolve();
        } else {
          /*
           * layout according to size and scale.
           */
          calculateImageDimensions(imgSrc, function (imgWidth, imgHeight) {
            var css = px.getScaledLayoutCSS(picWidth, picHeight, imgWidth, imgHeight, halign, valign, scale, imgSrc);

            //fix flashing on window resize which includes Html5HxProfile start when sidebar open
            if (imgSrc !== imgDom.srcRelative) {
              imgDom.srcRelative = imgSrc;
              imgDom.src = imgSrc;
            }
            imgJq.css(css);

            //NCCB-5908: webkit strikes again with poor SVG rescaling. the hide()
            //and offset() force a browser reflow as a workaround for the following
            //bug:
            //https://code.google.com/p/chromium/issues/detail?id=269446
            if (px.getLowerCaseFileExtension(imgSrc) === "svg") {
              imgStyle.display = 'none';
              // eslint-disable-next-line no-unused-vars
              var forceHeightCheck = imgDom.offsetHeight;
            }
            imgStyle.display = 'block';
            resolve();
          });
        }
      });
    });
  };

  /**
   * Get the lower case file extension for the URL. This handles query strings '?",  url fragments '#', and
   * view Queries that can include '|'.  If the URL is properly encoded, the URL will be decoded to find the
   * proper delimiters.
   *
   * @param {String|Object} url
   * @returns {String} The lowercase file extension. Empty String will be returned if no ext is found.
   * @since Niagara 4.9
   */
  this.getLowerCaseFileExtension = function (url) {
    url = String(url);
    try {
      url = decodeURI(url);
    } catch (err) {}
    var match = url.toLowerCase().match(/\.([^./?#|]+)($|\?|#|\|)/);
    if (match && match.length > 1 && match[1]) {
      return match[1];
    } else {
      return "";
    }
  };

  /**
   * IE10+ workaround for not-propagating the active pseudo class (add the active class instead to your css rules)
   * @param {String} id
   */
  this.activeFix = function (id) {
    var $ = jQuery;
    if ($) {
      $(document.getElementById(id)).on("mousedown mouseup mouseout", function (e) {
        $(this).toggleClass("active", e.type === "mousedown");
      });
    }
  };

  ////////////////////////////////////////////////////////////////
  // Loading dialog - @since Niagara 4.3
  ////////////////////////////////////////////////////////////////

  /**
   * In order to call this method from Java, you must call requireJs() on the HxOp first!
   * This function will show a Loading dialog and call POST to the current window location at the
   * given interval with the given header information until it gets a non SC_NO_CONTENT HTTP
   * response, such as OK. It will also supply a proper CSRF token header in the POSTs for the
   * server to check if necessary.
   *
   * @param headerName
   * @param headerVal
   * @param interval
   *
   * @since Niagara 4.3
   */
  this.showLoadingUntilPostOk = function (headerName, headerVal, interval) {
    require(["dialogs", "jquery", "Promise"], function (dialogs, $, Promise) {
      var header = {},
        responseStatus = 0;
      header[headerName] = headerVal;
      header['x-niagara-csrfToken'] = hx.getCsrfToken();

      // eslint-disable-next-line promise/avoid-new
      dialogs.showLoading(0, new Promise(function (resolve, reject) {
        function checkFinishedLoading() {
          Promise.resolve($.ajax({
            type: "POST",
            url: window.location,
            headers: header,
            complete: function complete(response) {
              responseStatus = response.status;
            }
          }))["finally"](function () {
            // Expecting to get either a 204 or 200 response from the server in normal circumstances
            // (see BHxPxView's doPost() method). If we ever get back anything unexpected, then go
            // ahead and clear the interval since we don't want it to linger around
            if (responseStatus !== 204) {
              resolve();
              if (responseStatus === 200) {
                window.location.reload(); // OK to reload the view since it finished loading
              }
            } else {
              // We got a 204, so set a timeout to check again later
              setTimeout(checkFinishedLoading, interval);
            }
          });
        }
        checkFinishedLoading();
      }));
    });
  };

  /**
   * This function applies to widgets that support dynamic scaling of an
   * internal element - currently just CanvasPanes in hxpx. It calculates
   * the CSS necessary to scale and align the internal element within the
   * absolutely positioned outer frame.
   *
   * @param {Number} outerWidth width of absolutely positioned outer element
   * @param {Number} outerHeight height of absolutely positioned outer element
   * @param {Number} innerWidth width of scaled/aligned inner element
   * @param {Number} innerHeight height of scaled/aligned inner element
   * @param {String} halign the horizontal alignment - should correspond to
   * a BHalign tag (left, center, right, fill)
   * @param {String} valign the vertical alignment - should correspond to
   * a BValign tag (top, center, bottom, fill)
   * @param {String} scale the scale mode - should correspond to a BScaleMode
   * tag (none, fit, fitRatio, fitWidth, fitHeight, zoomRatio, zoomWidth, zoomHeight)
   * @param {String} [imgSrc] the imgSrc- will be used to determine if additional
   * svg transforms should be used (if imgSrc ends in ".svg").
   * @return {Object} an object with <code>display</code>, <code>width</code>,
   * <code>height</code>, <code>margin-top</code>, <code>margin-left</code> and
   * <code>transform</code> properties. This should be applied to the inner
   * element to be scaled.
   */
  this.getScaledLayoutCSS = function (outerWidth, outerHeight, innerWidth, innerHeight, halign, valign, scale, imgSrc) {
    var imgRatio,
      desiredWidth,
      desiredHeight,
      marginTop,
      marginLeft,
      overrideTop,
      overrideLeft,
      transform,
      cssObj,
      isSvg = px.getLowerCaseFileExtension(imgSrc) === "svg";
    switch (scale) {
      case 'none':
        desiredWidth = innerWidth;
        desiredHeight = innerHeight;
        break;
      case 'fit':
        desiredWidth = outerWidth;
        desiredHeight = outerHeight;
        if (isSvg) {
          desiredWidth = innerWidth;
          desiredHeight = innerHeight;
          var heightRatio = outerHeight / innerHeight;
          var widthRatio = outerWidth / innerWidth;
          overrideTop = (outerHeight - innerHeight) / 2;
          overrideLeft = (outerWidth - innerWidth) / 2;
          transform = "scale(" + widthRatio + "," + heightRatio + ")";
        }
        break;
      case 'fitHeight':
        desiredWidth = innerWidth;
        desiredHeight = outerHeight;
        if (isSvg) {
          imgRatio = outerHeight / innerHeight;

          //This part is tricky, we never want to make the height smaller
          if (imgRatio > 1) {
            //NCCB-29440 adding this transform is correct as long as the svg `viewBox` is present
            transform = "scale(1," + imgRatio + ")";
          } else {
            transform = "scale(" + 1 / imgRatio + ",1)";
          }
        }
        break;
      case 'fitWidth':
        desiredWidth = outerWidth;
        desiredHeight = innerHeight;
        if (isSvg) {
          imgRatio = outerWidth / innerWidth;
          //This part is tricky, we never want to make the width smaller
          if (imgRatio > 1) {
            //NCCB-29440 adding this transform is correct as long as the svg `viewBox` is present
            transform = "scale(" + imgRatio + ",1)";
          } else {
            transform = "scale(1," + 1 / imgRatio + ")";
          }
        }
        break;
      case 'fitRatio':
        //scale image down only enough to fit within picture borders
        imgRatio = Math.min(outerWidth / innerWidth, outerHeight / innerHeight);
        desiredWidth = innerWidth * imgRatio;
        desiredHeight = innerHeight * imgRatio;
        break;
      case 'zoomRatio':
        imgRatio = Math.max(outerWidth / innerWidth, outerHeight / innerHeight);
        desiredWidth = innerWidth * imgRatio;
        desiredHeight = innerHeight * imgRatio;
        break;
      case 'zoomWidth':
        imgRatio = outerWidth / innerWidth;
        desiredWidth = innerWidth * imgRatio;
        desiredHeight = innerHeight * imgRatio;
        break;
      case 'zoomHeight':
        imgRatio = outerHeight / innerHeight;
        desiredWidth = innerWidth * imgRatio;
        desiredHeight = innerHeight * imgRatio;
        break;
    }

    /*
     * align using margins, since text-align won't quite do it if the image is
     * larger than the Picture itself
     */
    switch (halign) {
      case 'left':
        marginLeft = 0;
        break;
      case 'center':
        marginLeft = (outerWidth - desiredWidth) / 2;
        break;
      case 'right':
      case 'fill':
        marginLeft = outerWidth - desiredWidth;
        break;
    }
    switch (valign) {
      case 'top':
        marginTop = 0;
        break;
      case 'center':
        marginTop = (outerHeight - desiredHeight) / 2;
        break;
      case 'bottom':
      case 'fill':
        marginTop = outerHeight - desiredHeight;
        break;
    }
    cssObj = {
      display: 'block',
      width: desiredWidth,
      height: desiredHeight,
      'margin-top': marginTop,
      'margin-left': marginLeft
    };
    if (overrideTop) {
      cssObj['margin-top'] = overrideTop;
    }
    if (overrideLeft) {
      cssObj['margin-left'] = overrideLeft;
    }
    if (transform) {
      cssObj.transform = transform;
    }
    return cssObj;
  };

  /**
   * Adds the CSS3 transforms to the CSS object
   *
   * @param {Object} cssObj the CSS object to add rules to
   * @param {Number} hScale horizontal scale factor
   * @param {Number} vScale vertical scale factor
   */
  this.addScaling = function (cssObj, hScale, vScale) {
    cssObj.transform = 'scale(' + hScale + ',' + vScale + ')';
    cssObj['transform-origin'] = 'left top';
  };

  /**
   * Return the scaled dimensions of an HtmlElement. This is different than 1 if css Transforms has
   * scaled the element. If offsetWidth or offsetHeight is 0, then that dimension will not be scaled
   * @param {HTMLElement} element
   * @returns {Array.<Number>}} scaling dimensions for [x, y]
   */
  this.getScalingDimensions = function (element) {
    var scaleX = 1;
    var scaleY = 1;
    if (element) {
      if (element.offsetWidth) {
        scaleX = element.getBoundingClientRect().width / element.offsetWidth;
      }
      if (element.offsetHeight) {
        scaleY = element.getBoundingClientRect().height / element.offsetHeight;
      }
    }
    return [scaleX, scaleY];
  };

  /**
   * Set the dimensions for an svg element.
   * @param {String} id the ID of the element
   * @param {Number} width
   * @param {Number} height
   */
  this.setSvgDimensions = function (id, width, height) {
    var elem = hx.$(id);
    if (elem) {
      elem.setAttribute('width', width);
      elem.setAttribute('height', height);
    }
  };

  /**
   * Set the style of an inner svg element.
   * @param {dom} elem
   * @param {Object} state
   * @param {boolean} fill
   * @param {String} [groupTransform]
   */
  this.styleSvg = function (elem, state, fill, groupTransform) {
    var d3Elem = d3.select(elem);
    if (groupTransform && !elem.parentNode.$transformAppled) {
      var parentTransform = elem.parentNode.getAttribute('transform') || '';
      if (parentTransform) {
        parentTransform += ' ';
      }
      elem.parentNode.setAttribute('transform', parentTransform + groupTransform);
      elem.parentNode.$transformAppled = true;
    }
    d3Elem.classed('hxpx-mouseHandler', true).attr('stroke-linejoin', state.lineJoin).attr('stroke-linecap', state.lineCap).attr('stroke-dasharray', state.lineDash).attr('stroke-width', state.lineWidth).attr(fill ? 'fill' : 'stroke', state.color);
  };
  var HIDE_LABEL_CLASS = "-t-ImageButton-no-label";
  function isNullImage(image) {
    return !image;
  }
  function getImageStyleString(image) {
    if (isNullImage(image)) {
      return "";
    }
    return image;
  }
  function setImage(dom, image) {
    dom.find("button > .-t-ImageButton-backgroundImage").css("background-image", getImageStyleString(image));
  }

  /**
   * Setup the event handling for an UxImageButton in HxPx.
   *
   * @param {JQuery} dom
   * @param {String} [disabledImage]
   * @param {String} [normalImage]
   * @param {String} [mouseOverImage]
   * @param {String} [pressedImage]
   * @private
   */
  this.$registerImageButtonEvents = function (dom, disabledImage, normalImage, mouseOverImage, pressedImage) {
    if (!dom || !dom.length || dom[0].$isImageButtonEventsRegistered) {
      return;
    }
    dom[0].$isImageButtonEventsRegistered = true;
    var isPressed = false,
      isMouseOver = false;
    var updateDisplay = function updateDisplay() {
      var button = dom.find("button");
      button.toggleClass(HIDE_LABEL_CLASS, true);
      if (isPressed && !isNullImage(pressedImage)) {
        setImage(dom, pressedImage);
      } else if (isMouseOver && !isNullImage(mouseOverImage)) {
        setImage(dom, mouseOverImage);
      } else if (!isNullImage(normalImage)) {
        setImage(dom, normalImage);
      } else {
        setImage(dom, "");
        button.toggleClass(HIDE_LABEL_CLASS, false);
      }
    };
    dom.on('touchstart mousedown', function () {
      isPressed = true;
      updateDisplay();
    });
    dom.on('touchend mouseup', function () {
      isPressed = false;
      updateDisplay();
    });
    if (this.$canHover()) {
      dom.hover(function () {
        isMouseOver = true;
        updateDisplay();
      }, function () {
        isMouseOver = false;
        isPressed = false;
        updateDisplay();
      });
    }
  };

  /**
   * @private
   * @returns {boolean}
   */
  this.$canHover = function () {
    return window.matchMedia('(hover: hover)').matches;
  };
}
