/**
 * @copyright 2023 Tridium, Inc. All Rights Reserved.
 */

define(['baja!', 'baja!bajaui:Orientation'], function (baja, types) {
  'use strict';

  /**
   * Utility api to support a sidebar's dimension.
   *
   * API Status: **Private**
   * @module nmodule/bajaui/rc/util/sideBarUtils
   * @since Niagara 4.15
   */
  var exports = {};
  var MAXIMIZED_PERCENT = 100;
  var RESTORED_PERCENT = 75;

  /**
   * Makes a default sidebar state.
   *
   * @private
   * @param {Object} params
   * @param {baja:Simple} [params.orientation=baja.$('bajaui:Orientation', 'vertical')]
   * @param {number} [params.offsetPercent=0]
   * @param {boolean} [params.visible=true]
   * @param {boolean} [params.isMaximized=false]
   * @returns {SidebarState}
   */
  exports.$makeSidebarState = function () {
    var _ref = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      _ref$orientation = _ref.orientation,
      orientation = _ref$orientation === void 0 ? baja.$('bajaui:Orientation', 'vertical') : _ref$orientation,
      _ref$offsetPercent = _ref.offsetPercent,
      offsetPercent = _ref$offsetPercent === void 0 ? 0 : _ref$offsetPercent,
      _ref$visible = _ref.visible,
      visible = _ref$visible === void 0 ? true : _ref$visible,
      _ref$isMaximized = _ref.isMaximized,
      isMaximized = _ref$isMaximized === void 0 ? false : _ref$isMaximized;
    return {
      visible: visible,
      orientation: orientation,
      offsetPercent: offsetPercent,
      isMaximized: isMaximized
    };
  };

  /**
   * @returns {number}
   */
  exports.getMaximizedPercent = function () {
    return MAXIMIZED_PERCENT;
  };

  /**
   * @returns {number}
   */
  exports.getRestorePercent = function () {
    return RESTORED_PERCENT;
  };

  /**
   * @private
   * @returns {Object<String, SidebarState>}
   */
  exports.$getDefaultStatesInAbsolute = function () {
    var sidebars = ['boundOrds', 'widgetTree', 'pxLayers', 'pxProperties', 'properties'];
    var states = {};
    sidebars.forEach(function (sidebar) {
      states[sidebar] = exports.$makeSidebarState({
        orientation: baja.$('bajaui:Orientation', 'vertical'),
        offsetPercent: 100 / sidebars.length
      });
    });
    states['pxWidget'] = exports.$makeSidebarState({
      orientation: baja.$('bajaui:Orientation', 'horizontal'),
      offsetPercent: RESTORED_PERCENT
    });
    return states;
  };

  /**
   * It will set the relative offset percentages for the input sidebars with absolute percentages
   *
   * @private
   * @param {Object.<String, SidebarState>} states
   * @returns {Object.<String, SidebarState>}
   */
  exports.$setOffsetPercentsForSidebars = function (states) {
    var defaultSidebarNames = ['boundOrds', 'widgetTree', 'pxProperties', 'pxLayers', 'properties'];
    var sideBarNames = defaultSidebarNames.filter(function (name) {
      return Object.keys(states).contains(name);
    });
    sideBarNames.forEach(function (sideBarName, index) {
      var sideBarState = states[sideBarName];
      if (index === sideBarNames.length - 1) {
        sideBarState.offsetPercent = MAXIMIZED_PERCENT;
      } else {
        var numerator = sideBarState.updatedPercent;
        var denominator = 0;
        for (var j = index; j < sideBarNames.length; j++) {
          denominator += states[sideBarNames[j]].updatedPercent;
        }
        sideBarState.offsetPercent = numerator / denominator * 100;
      }
      delete sideBarState.updatedPercent;
    });
    return states;
  };

  /**
   * Sets the absolute percentages in the input sidebar states.
   * This step is necessary to derive a relative percentage for a sidebar.
   *
   * @private
   * @param{Object.<String, SidebarState>} states
   * @param {Object.<String: number>} [updatedPercents] sidebarName -> newOffset
   * @returns {Object.<String, SidebarState>}
   */
  exports.$setAbsolutePercentsForSidebars = function (states) {
    var updatedPercents = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    var remainingPercent = 100;
    var setPercentagesCount = 0;
    Object.keys(updatedPercents).forEach(function (sidebarName) {
      states[sidebarName].updatedPercent = updatedPercents[sidebarName];
      remainingPercent -= updatedPercents[sidebarName];
      setPercentagesCount++;
    });
    var stateNames = Object.keys(states);
    var remPercent = remainingPercent / (stateNames.length - setPercentagesCount);
    stateNames.forEach(function (stateName) {
      var updatedPercent = states[stateName].updatedPercent;
      if (!updatedPercent) {
        states[stateName].updatedPercent = remPercent;
      }
    });
    return states;
  };

  /**
   * Returns the default state for the sidebars, assigns the relative offset height which can be used by the split-pane.
   *
   * @param {Array.<String>}sidebarNames
   * @param {Object} [mainSidebar]
   * @param {String} mainSidebar.name
   * @param {number} mainSidebar.offsetPercent
   * @returns {Object.<String, SidebarState>} sidebar name -> state
   */
  exports.getDefaultSidebarStates = function (sidebarNames) {
    var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      name = _ref2.name,
      offsetPercent = _ref2.offsetPercent;
    var mainSidebarState = {};
    var otherStates = {};
    sidebarNames.forEach(function (sidebarName) {
      if (sidebarName === name) {
        mainSidebarState[sidebarName] = exports.$makeSidebarState({
          orientation: baja.$('bajaui:Orientation', 'horizontal'),
          offsetPercent: offsetPercent
        });
      } else {
        otherStates[sidebarName] = exports.$makeSidebarState();
      }
    });
    otherStates = exports.setSidebarsOffset(otherStates, {
      mainSidebar: {
        name: name,
        offset: offsetPercent
      }
    });
    return Object.assign(mainSidebarState, otherStates);
  };

  /**
   * Sets sidebar(s) offset height and width in case of a vertical sidebar.
   * User can provide an updated absolute percentage for the sidebar(s), based on that, all sidebar's relative offset will the adjusted accordingly.
   *
   * Example: I want sidebar-2 to take up 60% of screen height, and other sidebars can take up the remaining space equally.
   * Example: I want sidebar-1 and sidebar-2 to take up 80% of screen height, and other sidebars can take up remaining space equally.
   * Example: I want all sidebars to take up equal heights.
    * @param {Object.<String, SidebarState>} states
   * @param {Object} [metadata]
   * @param {Object} [metadata.mainSidebar]
   * @param {String} metadata.mainSidebar.name
   * @param {number} [metadata.mainSidebar.offset]
   * @param {Object.<String, number>} [metadata.updatedPercents] {sidebarName -> newOffset}
   * @returns {Object.<String, SidebarState>}
   */
  exports.setSidebarsOffset = function (states) {
    var _ref3 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref3$mainSidebar = _ref3.mainSidebar,
      _ref3$mainSidebar2 = _ref3$mainSidebar === void 0 ? {} : _ref3$mainSidebar,
      name = _ref3$mainSidebar2.name,
      offset = _ref3$mainSidebar2.offset,
      _ref3$updatedPercents = _ref3.updatedPercents,
      updatedPercents = _ref3$updatedPercents === void 0 ? {} : _ref3$updatedPercents;
    var mainSidebarState = {};
    var otherStates = {};
    Object.keys(states).forEach(function (sidebarName) {
      if (sidebarName === name) {
        mainSidebarState[sidebarName] = states[sidebarName];
        mainSidebarState[sidebarName].offsetPercent = offset || mainSidebarState[sidebarName].offsetPercent;
      } else {
        otherStates[sidebarName] = states[sidebarName];
      }
    });
    otherStates = exports.$setAbsolutePercentsForSidebars(otherStates, updatedPercents);
    return Object.assign(mainSidebarState, exports.$setOffsetPercentsForSidebars(otherStates));
  };

  /**
   * Describes the current layout and visibility of a sidebar.
   *
   * @private
   * @typedef {Object} SidebarState
   * @property {boolean} visible
   * @property {String} orientation `horizontal` or `vertical` - main divider is vertical, left sidebars are horizontal
   * @property {number} offsetPercent
   * @property {boolean} isMaximized
   */

  return exports;
});
