/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author JJ Frankovich
 */

/**
 * API Status: **Private**
 * @module nmodule/seriesTransform/rc/SeriesTransformSeries
 */
define(['baja!', 'nmodule/webChart/rc/model/BaseSeries', 'nmodule/webChart/rc/model/modelUtil', 'nmodule/webChart/rc/chartEvents', 'nmodule/webChart/rc/webChartUtil'], function (baja, BaseSeries, modelUtil, chartEvents, webChartUtil) {
  "use strict";

  /**
   * A Series for a TransformGraph.
   *
   * @private
   * @class
   * @extends module:nmodule/webChart/rc/model/BaseSeries
   * @alias module:nmodule/seriesTransform/rc/SeriesTransformSeries
   */
  var SeriesTransformSeries = function SeriesTransformSeries() {
    BaseSeries.apply(this, arguments);
  };
  SeriesTransformSeries.prototype = new BaseSeries();
  SeriesTransformSeries.prototype.constructor = SeriesTransformSeries;

  /**
   * Load the info for the series.
   *
   * @returns {Promise}
   */
  SeriesTransformSeries.prototype.loadInfo = function () {
    var that = this,
      model = that.$model,
      ord = that.$ord;
    if (that.$loaded || that.$loading) {
      return modelUtil.resolvedPromise();
    }

    //ensure we start with a refresh set of points
    that.$points.length = 0;
    that.$focusPoints = null;
    that.$samplingPoints = null;
    return webChartUtil.rpc("type:seriesTransform:SeriesTransformWebChartQueryRpc", "getInfo", [String(ord)]).then(function (response) {
      if (response.errors && response.errors.length > 0) {
        that.error(response.errors[0]);
        model.removeSeries(that);
        var message = webChartUtil.lex.get("webChart.cannotLoadSeries", that.ord(), that.error());
        model.jq().trigger(chartEvents.DISPLAY_ERROR, message);
        return;
      }
      if (response.recordTypes && response.recordTypes.length > 0) {
        that.$recordType = response.recordTypes[0];
      }
      if (response.displayPaths && response.displayPaths.length > 0) {
        that.$displayPath = response.displayPaths[0];
      }
      if (response.timezoneIDs && response.timezoneIDs.length > 0) {
        that.setPreferredTimeZone(model.timeScale().getTimeZoneDatabase().getTimeZone(response.timezoneIDs[0]));
      }
      var valueFacets = response.valueFacets;
      return valueFacets && valueFacets.length && baja.Facets.DEFAULT.decodeAsync(valueFacets[0]);
    }).then(function (facets) {
      if (facets) {
        that.$facets = facets;
        that.$units = that.$facets.get("units", baja.Unit.DEFAULT);
      }
      //setup valueScale now that units are available
      return model.makeValueScale(that);
    }).then(function (valueScale) {
      that.$valueScale = valueScale;
    });
  };

  /**
   * Load the series data.
   *
   * @returns {Promise}
   */
  SeriesTransformSeries.prototype.loadData = function () {
    var that = this,
      model = that.$model,
      delta = model.isDelta ? model.isDelta() : false,
      timeRange = model.timeRange(),
      ord = that.$ord,
      fullOrd;
    if (that.$loaded || that.$loading) {
      return modelUtil.resolvedPromise(); //nothing to do
    }
    that.$loading = true;

    //ensure we start with a refresh set of points
    that.$points.length = 0;
    that.$focusPoints = null;
    that.$samplingPoints = null;
    fullOrd = modelUtil.getFullScheduleOrd({
      ord: ord,
      timeRange: timeRange,
      delta: delta
    });
    var dataUri = "/seriesTransform/seriesTransformQuery/data/" + baja.SlotPath.escape(fullOrd);
    return modelUtil.chunkData(model, that, dataUri).then(function () {
      that.$loaded = true;
      that.$loading = false;
    });
  };

  /**
   * Properly format a results from QueryServlet for SeriesTransform and append it to the current data.
   * @param {Object} raw the object passed over from the QueryServlet
   */
  SeriesTransformSeries.prototype.preparePoint = function (raw) {
    return modelUtil.prepareServletPoint(raw, this.$points);
  };

  /**
   * Return the Series display name that will appear in the Legend.
   *
   * @returns {String}  The display name.
   */
  SeriesTransformSeries.prototype.displayName = function () {
    return this.$value.getDisplayName();
  };
  return SeriesTransformSeries;
});
