/**
 * @copyright 2015, Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/bql/rc/fe/filter/FilterSetEditor
 */
define(['baja!', 'baja!bql:FilterSet', 'jquery', 'underscore', 'Promise', 'bajaux/events', 'bajaux/Widget', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/fe', 'bajaux/mixin/batchSaveMixin', 'bajaux/mixin/subscriberMixIn', 'nmodule/bql/rc/fe/filter/saveUtil', 'css!nmodule/bql/rc/bql'], function (baja, types, $, _, Promise, events, Widget, BaseEditor, fe, batchSaveMixin, subscriberMixin, saveUtil) {
  "use strict";

  var FILTER_ENTRY_CLASS = "filterEntry",
    FILTER_ENTRY_SELECTOR = '.' + FILTER_ENTRY_CLASS,
    ENABLE_EVENT = events.ENABLE_EVENT,
    DISABLE_EVENT = events.DISABLE_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    INITIALIZE_EVENT = events.INITIALIZE_EVENT,
    LOAD_EVENT = events.LOAD_EVENT,
    READONLY_EVENT = events.READONLY_EVENT,
    WRITABLE_EVENT = events.WRITABLE_EVENT,
    DESTROY_EVENT = events.DESTROY_EVENT;

  /**
   * FilterSetEditor is a field editor for a bql:FilterSet instance
   * @class
   * @alias module:nmodule/bql/rc/fe/filter/FilterSetEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */

  var FilterSetEditor = function FilterSetEditor() {
    BaseEditor.apply(this, arguments);
    batchSaveMixin(this);
    subscriberMixin(this);
  };
  FilterSetEditor.prototype = Object.create(BaseEditor.prototype);
  FilterSetEditor.prototype.constructor = FilterSetEditor;

  /**
   * Initialization adds event handlers for event bubbling
   * @param dom
   * @returns {Promise}
   */
  FilterSetEditor.prototype.doInitialize = function (dom) {
    var that = this;

    //prevent subeditors from bubbling up events
    dom.addClass("FilterSetEditor").on([ENABLE_EVENT, DISABLE_EVENT, READONLY_EVENT, WRITABLE_EVENT, INITIALIZE_EVENT, LOAD_EVENT, DESTROY_EVENT].join(' '), FILTER_ENTRY_SELECTOR, false).on(MODIFY_EVENT, FILTER_ENTRY_SELECTOR, function () {
      that.setModified(true);
      return false;
    });
  };

  /**
   * Loads the child filter entries of the filter set
   * @param filterSet
   * @param params
   * @returns {Promise}
   */
  FilterSetEditor.prototype.doLoad = function (filterSet) {
    if (!baja.hasType(filterSet, 'bql:FilterSet')) {
      return Promise.reject(new Error('editor must be loaded with bql:FilterSet'));
    }
    var that = this,
      editors = this.$getFilterEntryEditors(),
      promises = _.map(editors, function (editor) {
        return editor.destroy();
      });
    return Promise.all(promises).then(function () {
      var filterEntries = filterSet.getSlots().is('bql:FilterEntry').toValueMap();
      that.$slots = _.keys(filterEntries) || [];
      promises = _.map(filterEntries, function (filterEntry, slotName) {
        return fe.buildFor({
          dom: $("<div/>").addClass(FILTER_ENTRY_CLASS).appendTo(that.jq()),
          value: filterEntry,
          formFactor: 'mini'
        }).then(function (editor) {
          editor.jq().data('slot', slotName);
        });
      });
      return Promise.all(promises);
    });
  };

  /**
   * Reads the current state of filter entries and returns a newly initialized bql:FilterSet instance
   * @returns {Promise} Promise resolved with a new instance of bql:FilterSet
   */
  FilterSetEditor.prototype.doRead = function () {
    var that = this,
      filterSet = baja.$('bql:FilterSet'),
      promises = _.map(this.$getFilterEntryEditors(), function (editor) {
        return editor.read().then(function (filterEntry) {
          return filterSet.add({
            slot: editor.jq().data('slot'),
            value: filterEntry
          });
        });
      });
    return Promise.all(promises).then(function () {
      return that.$slots.length > 1 && filterSet.reorder(that.$slots);
    }).then(function () {
      return filterSet;
    });
  };

  /**
   * Saves the bql:FilterSet returned by read() to the currently loaded value
   * @param filterEntry
   * @param params
   * @returns {Promise}
   */
  FilterSetEditor.prototype.doSave = function (filterSet, params) {
    return saveUtil.save(this, filterSet, params);
  };

  /**
   * Sets the readonly state of the child editors
   * @param readonly
   * @returns {Promise}
   */
  FilterSetEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
  };

  /**
   * Sets the enabled state of the child editors
   * @param enabled
   * @returns {Promise}
   */
  FilterSetEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };

  /**
   * Destroys the child editors
   * @param params
   * @returns {Promise}
   */
  FilterSetEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };
  FilterSetEditor.prototype.$getFilterEntries = function () {
    return this.jq().find(FILTER_ENTRY_SELECTOR);
  };
  FilterSetEditor.prototype.$getFilterEntryEditors = function () {
    return this.$getFilterEntries().map(function () {
      return Widget["in"](this);
    }).get();
  };
  return FilterSetEditor;
});
