/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/webChart/rc/grid/GridEditor
 */
define(['jquery', 'Promise', 'baja!', 'nmodule/analytics/rc/util/Tab', 'bajaux/commands/Command', 'bajaux/Widget', 'hbs!nmodule/analytics/rc/util/gridtemplate'], function ($, Promise, baja, Tab, Command, Widget, gridtemplate) {
  "use strict";

  /**
   * GridEditor is a Widget that can group other Widgets and commands that are organized in tabs.
   *
   * @class
   * @alias module:nmodule/webChart/rc/grid/GridEditor
   * @extends module:bajaux/Widget
   */
  var GridEditor = function GridEditor() {
    Widget.apply(this, arguments);
    this.$tabs = [];
  };
  GridEditor.prototype = Object.create(Widget.prototype);
  GridEditor.prototype.constructor = GridEditor;

  /**
   * Use the structure provided to initialize the dom the current tabs.
   * @param {jquery} dom
   * @returns {Promise}
   */
  GridEditor.prototype.doInitialize = function (dom) {
    var that = this,
      inits;
    dom.html(that.getStructure()({
      tabs: that.$tabs
    }));
    inits = $.map(that.$tabs, function (tab, index) {
      var tabDom = that.getTabContentElement(tab);
      if (tabDom.length === 0) {
        // eslint-disable-next-line prefer-promise-reject-errors
        return Promise.reject("init cannot find dom for tab: " + tab.title + ", " + index);
      }
      if (tab.widget instanceof Widget) {
        tab.dom = tabDom;
        return that.initTab(tab, index);
      } else {
        // eslint-disable-next-line prefer-promise-reject-errors
        return Promise.reject("init needs a widget for tab: " + tab.title + ", " + index);
      }
    });
    return Promise.all(inits);
  };

  /**
   * Callback for subclasses to addOn to the initialization of a tab.
   * @param {webChart.Tab} tab
   * @returns {Promise}
   */
  GridEditor.prototype.initTab = function (tab) {
    return tab.initialize();
  };
  /**
   * Callback for subclasses to addOn to the loading of a tab.
   * @param {webChart.Tab} tab
   * @returns {Promise}
   */
  GridEditor.prototype.loadTab = function (tab) {
    return tab.load();
  };

  /**
   * Return the element for the Content.
   * @param {webChart.tab}
   * @returns {jquery}
   */
  GridEditor.prototype.getTabContentElement = function (tab) {
    return $('.js-tab-' + tab.title, this.jq());
  };

  /**
   * Subclasses should override this method to alter the structure of the tabs
   * @returns {template}
   */
  GridEditor.prototype.getStructure = function () {
    return gridtemplate;
  };

  /**
   * Add a Command, uses the commandID at the title.
   * @param {module:bajaux/commands/Command} command
   * @return {webChart.Tab}
   */
  GridEditor.prototype.addCommand = function (command) {
    var that = this;
    if (command instanceof Command) {
      var tab = new Tab({
        item: command,
        title: String(command.$id)
      });
      that.$tabs.push(tab);
      return tab;
    } else {
      throw new Error("Must be a Command");
    }
  };

  /**
   * Get the Command based on the tab number.
   * @param {Number} tabNumber
   * @returns {module:bajaux/commands/Command}
   */
  GridEditor.prototype.getTabByIndex = function (tabNumber) {
    var that = this;
    return that.$tabs[tabNumber];
  };

  /**
   * Get the Command based on the tab number.
   * @param {Number} tabNumber
   * @returns {module:bajaux/commands/Command}
   */
  GridEditor.prototype.getCommand = function (tabNumber) {
    var that = this,
      tab = that.getTabByIndex(tabNumber),
      value;
    if (tab && tab.widget && tab.widget.value) {
      value = tab.widget.value();
      if (value instanceof Command) {
        return value;
      }
    }
    return null;
  };

  /**
   * Get the Command based on the title.
   * @param tabTitle
   * @returns {webChart.Tab}
   */
  GridEditor.prototype.getTab = function (tabTitle) {
    var that = this,
      i,
      tab;
    for (i = 0; i < that.$tabs.length; i++) {
      tab = that.$tabs[i];
      if (tab.title === tabTitle) {
        return tab;
      }
    }
  };

  /**
   * Add a Tab.
   * @param {webChart.Tab} tab
   * @return {webChart.Tab}
   */
  GridEditor.prototype.addTab = function (tab) {
    this.$tabs.push(tab);
    return tab;
  };

  /**
   * Replace a Tab.
   * @param {webChart.Tab} tab
   * @param {Number} index
   */
  GridEditor.prototype.replaceTab = function (tab, index) {
    this.$tabs[index] = tab;
  };

  /**
   * Read all the values for all the tabs.
   * @returns {Promise} All resolutions are organized in a single resolve Array.
   */
  GridEditor.prototype.doRead = function () {
    var that = this,
      reads = [],
      i;
    for (i = 0; i < that.$tabs.length; i++) {
      reads.push(that.$tabs[i].read());
    }
    return Promise.all(reads);
  };

  /**
   * Save all the values for all the tabs.
   * @returns {Promise} All resolutions are organized in a single resolve Array.
   */
  GridEditor.prototype.doSave = function () {
    var that = this,
      saves = [],
      i;
    for (i = 0; i < that.$tabs.length; i++) {
      saves.push(that.$tabs[i].save());
    }
    return Promise.all(saves);
  };

  /**
   * Load all the tabs values.
   * @returns {Promise}
   */
  GridEditor.prototype.doLoad = function () {
    var that = this,
      loads = $.map(that.$tabs, function (tab, index) {
        var tabDom = that.getTabContentElement(tab);
        if (tabDom.length === 0) {
          // eslint-disable-next-line prefer-promise-reject-errors
          return Promise.reject("load cannot find dom for tab: " + tab.title + ", " + index);
        }
        if (tab.widget instanceof Widget) {
          tab.dom = tabDom;
          return that.loadTab(tab, index);
        } else {
          // eslint-disable-next-line prefer-promise-reject-errors
          return Promise.reject("load needs a widget for tab: " + tab.title + ", " + index);
        }
      });
    return Promise.all(loads);
  };
  GridEditor.prototype.doDestroy = function (params) {
    var that = this,
      destroys = $.map(that.$tabs, function (tab, index) {
        var tabDom = that.getTabContentElement(tab);
        if (tabDom.length === 0) {
          // eslint-disable-next-line prefer-promise-reject-errors
          return Promise.reject("destroy cannot find dom for tab: " + tab.title + ", " + index);
        }
        if (tab.widget instanceof Widget) {
          tab.dom = tabDom;
          return tab.destroy();
        } else {
          // eslint-disable-next-line prefer-promise-reject-errors
          return Promise.reject("destroy needs a widget for tab: " + tab.title + ", " + index);
        }
      });
    that.$tabs = [];
    return Promise.all(destroys);
  };
  return GridEditor;
});
