/* eslint-disable promise/no-return-wrap */
/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/*jshint devel: true */

define(['baja!', 'Promise', 'moment', 'underscore', 'd3', 'lex!analytics'], function (baja, Promise, moment, _, d3, lexs) {
  'use strict';

  var exports = {};
  var colorRange = d3.scaleOrdinal(d3.schemeCategory10);
  var lex = lexs[0];
  /**
   * Retruns the start date time and end date time based on date Time string passed
   * @param dateStr
   */
  exports.getDatesFromTimeRange = function (dateStr) {
    dateStr = decodeURIComponent(dateStr);
    return baja.Ord.make("station:|service:analytics:AnalyticService").get().then(function (service) {
      return service.invoke({
        slot: "getStartEndTimes",
        value: baja.$("baja:String").make(dateStr)
      }).then(function (ret) {
        return Promise.resolve(ret);
      });
    });
  };
  /**
   *
   * @param dateStr
   */
  exports.getReportTypeFromJS = function (js) {
    var repType = baja.$("analytics:ReportType").make(0);
    if (js === "view:analytics:SpectrumUxReport") {
      repType = baja.$("analytics:ReportType").make(0);
    } else if (js === "view:analytics:LoadDurationUxReport") {
      repType = baja.$("analytics:ReportType").make(1);
    } else if (js === "view:analytics:AverageProfileUxReport") {
      repType = baja.$("analytics:ReportType").make(2);
    } else if (js === "view:analytics:AggregationUxReport") {
      repType = baja.$("analytics:ReportType").make(3);
    } else if (js === "view:analytics:EquipmentOperationUxReport") {
      repType = baja.$("analytics:ReportType").make(4);
    } else if (js === "view:analytics:RankingUxReport") {
      repType = baja.$("analytics:ReportType").make(5);
    } else if (js === "view:analytics:RelativeContributionUxReport") {
      repType = baja.$("analytics:ReportType").make(6);
    }
    return repType;
  };

  /**
   * Get the list of report container folders as a dynamic enum
   */
  exports.getFolderListEnum = function (folderList, enumVal) {
    var tagList = [],
      ordinals = [];
    _.each(folderList, function (folder, index) {
      ordinals.push(index);
      tagList.push(folder.n);
    });
    return baja.DynamicEnum.make({
      ordinal: enumVal || 0,
      range: baja.EnumRange.make({
        ordinals: ordinals,
        tags: tagList
      })
    });
  };

  /**
   *
   * @param st MainChart StartTime
   * @param et MainChart EndTime
   * @param baselineTag BaselineTimeRangeEnum
   * @param baseline BaselineValue
   */
  exports.getChartBaselineDateTimes = function (st, et, baselineTag, baseline) {
    baselineTag = baselineTag || "customPeriod";
    if (baselineTag === "customPeriod") {
      var cpSt = baja.$("baja:AbsTime").decodeFromString(baseline.baselineStartTime),
        cpEt = baja.$("baja:AbsTime").decodeFromString(baseline.baselineEndTime);
      return {
        blSt: this.getHourMinuteOnly(cpSt),
        blEt: this.getHourMinuteOnly(cpEt)
      };
    } else if (baselineTag === "previousWeek") {
      return this.getOneWeekPriorDates(st, et);
    } else if (baselineTag === "previousMonth") {
      return this.getOneMonthPriorDates(st, et, baseline.baselineAlignDOW);
    } else if (baselineTag === "previousYear") {
      return this.getOneYearPriorDates(st, et, baseline.baselineAlignDOW);
    }
  };
  /**
   * Get the baseline start and end date times to be used for building the ORD
   */
  exports.getBaselineDateTimes = function (st, et, baselineTag, baseline) {
    baselineTag = baselineTag || "customPeriod";
    if (baselineTag === "customPeriod") {
      var cpSt = baja.$("baja:AbsTime").decodeFromString(baseline.startTime),
        cpEt = baja.$("baja:AbsTime").decodeFromString(baseline.endTime);
      return {
        blSt: this.getHourMinuteOnly(cpSt),
        blEt: this.getHourMinuteOnly(cpEt)
      };
    } else if (baselineTag === "previousWeek") {
      return this.getOneWeekPriorDates(st, et);
    } else if (baselineTag === "previousMonth") {
      return this.getOneMonthPriorDates(st, et, baseline.alignDOW);
    } else if (baselineTag === "previousYear") {
      return this.getOneYearPriorDates(st, et, baseline.alignDOW);
    }
  };
  /**
   * Get the start and end times one week prior to the current start and end times
   * @param startTime
   * @param endTime
   */
  exports.getOneWeekPriorDates = function (startTime, endTime) {
    var timestamps = this.getTimestamps(startTime, endTime),
      startMoment = timestamps.st,
      endMoment = timestamps.et;
    var diff = endMoment.valueOf() - startMoment.valueOf();
    startMoment.subtract(7, 'days');
    endMoment = startMoment.clone().add(diff, 'milliseconds');
    return {
      blSt: baja.AbsTime.make({
        jsDate: startMoment.toDate()
      }),
      blEt: baja.AbsTime.make({
        jsDate: endMoment.toDate()
      })
    };
  };

  /**
    * Get the start and end times one month prior to the current start and end times
    * @param startTime
    * @param endTime
    */
  exports.getOneMonthPriorDates = function (startTime, endTime, alignDOW) {
    var timestamps = this.getTimestamps(startTime, endTime),
      startMoment = timestamps.st,
      endMoment = timestamps.et;
    var diff = endMoment.valueOf() - startMoment.valueOf(),
      whichDay = startMoment.weekday(),
      date = startMoment.date();
    startMoment.subtract(1, 'months');
    if (alignDOW) {
      // In case we need to align day of week, we need to find equivalent week last month and figure out the
      // same day in that week. E.g If today is the time range and today is tjird wednesday of this month, we try to get the third wednesday last month.
      // If it happens that the equivalent week is absent last month (5th week), we end up getting last week last month and figure
      //out the same day
      var whichWeek = -1; // Get which particular day e.g 3rd friday
      do {
        whichWeek += 1;
        date = date - 7;
      } while (date > 0);
      startMoment.startOf('month');
      var ref = startMoment.clone().endOf('month');
      startMoment.add(whichWeek * 7, 'days');
      var startOfMonthDay = startMoment.weekday();
      if (whichDay > startOfMonthDay) {
        startMoment.add(whichDay - startOfMonthDay, 'days');
      } else {
        startMoment.add(7 - (startOfMonthDay - whichDay), 'days');
      }
      if (startMoment.isAfter(ref)) {
        // This corner case generally happens when we have 5 weeks in a month
        startMoment.subtract(7, 'days');
      }
    }
    endMoment = startMoment.clone().add(diff, 'milliseconds');
    return {
      blSt: baja.AbsTime.make({
        jsDate: startMoment.toDate()
      }),
      blEt: baja.AbsTime.make({
        jsDate: endMoment.toDate()
      })
    };
  };
  /**
    * Get the start and end times one year prior to the current start and end times
    * @param startTime
    * @param endTime
    */
  exports.getOneYearPriorDates = function (startTime, endTime, alignDOW) {
    var timestamps = this.getTimestamps(startTime, endTime),
      startMoment = timestamps.st,
      endMoment = timestamps.et;
    var diff = endMoment.valueOf() - startMoment.valueOf();
    var weekOfYear = startMoment.week(),
      dayOfWeek = startMoment.weekday();
    startMoment.subtract(1, 'year');
    if (alignDOW) {
      // If we need to align days of week we need to get the same week last year and then figure out the same
      //day of week.
      startMoment.week(weekOfYear).weekday(dayOfWeek);
    }
    endMoment = startMoment.clone().add(diff, 'milliseconds');
    return {
      blSt: baja.AbsTime.make({
        jsDate: startMoment.toDate()
      }),
      blEt: baja.AbsTime.make({
        jsDate: endMoment.toDate()
      })
    };
  };
  /**
   * Get the time stamps from start and end abs date times
   * @param startTime
   * @param endTime
   */
  exports.getTimestamps = function (startTime, endTime) {
    var startMoment = moment(startTime.getMillis()),
      endMoment = moment(endTime.getMillis());
    if (startMoment.isAfter()) {
      startMoment = moment();
    }
    if (endMoment.isAfter()) {
      endMoment = moment();
    }
    return {
      st: startMoment,
      et: endMoment
    };
  };
  /**
   * Calculate the end time for baseline by adding the (end time -start time) to it.
   * @param startTime
   * @param endTime
   * @param baselineStartTime
   */
  exports.getCalculatedEndTime = function (startTime, endTime, baselineStartTime) {
    var diff = endTime.getMillis() - startTime.getMillis();
    var blStartTime = moment(baselineStartTime.getMillis()),
      blEndTime = blStartTime.clone().add(diff);
    return baja.AbsTime.make({
      jsDate: blEndTime.toDate()
    });
  };
  /**
   * Calculate the new time stamp by removing seconds and millis.
   * @param startTime
   * @param endTime
   * @param baselineStartTime
   */
  exports.getHourMinuteOnly = function (timestamp) {
    var fullTime = moment(timestamp.getMillis());
    fullTime.millisecond(0).second(0);
    return baja.AbsTime.make({
      jsDate: fullTime.toDate()
    });
  };
  /**
   * Get the color editor for the group
   * @param colorMap
   */
  exports.getColorEditorForGroup = function (colorMap) {
    var returnColorRange = colorRange[0];
    var found = false;
    var minIndex = 0;
    var minCount = colorMap[colorRange[0]] || 0;
    for (var i = 0; i < colorRange.length; i++) {
      if (colorMap[colorRange[i]] === undefined || colorMap[colorRange[i]] && colorMap[colorRange[i]] === 0) {
        // group.setSeriesColor(colorRange[i]);
        returnColorRange = colorRange[i];
        colorMap[colorRange[i]] = 1;
        found = true;
        break;
      }
      if (minCount > (colorMap[colorRange[i]] || 0)) {
        minCount = colorMap[colorRange[i]] || 0;
        minIndex = i;
      }
    }
    if (!found) {
      returnColorRange = colorRange[minIndex];
      colorMap[colorRange[minIndex]] = colorMap[colorRange[minIndex]] + 1;
    }
    return returnColorRange;
  };
  /**
   * Get the complete display node path till the root
   * @param nodeName
   * @param nodeOrd
   */
  exports.getDisplayNodePath = function (nodeName, nodeOrd) {
    var that = this;
    var ordQueries = nodeOrd.parse();
    var hier = ordQueries.get('hierarchy');
    if (hier) {
      var fragments = hier.getBody().split('/').map(baja.SlotPath.unescape);
      return baja.Ord.make("local:|hierarchy:").get().then(function (hierarchySpace) {
        var navChildrenPromise = hierarchySpace.getNavChildren(),
          counter = 1;
        return navChildrenPromise.then(function (navChildren) {
          return that.getHierarchyNodeDisplayPath(fragments, counter, navChildren, "");
        });
      });
    } else {
      return this.getConfigNodeDisplayPath(nodeName, nodeOrd);
    }
  };

  /**
   * Get the display path from hierarchy
   * @param fragments
   * @param counter
   * @param navChildren
   */
  exports.getHierarchyNodeDisplayPath = function (fragments, counter, navChildren, nodeName) {
    var that = this;
    for (var i = 0; i < navChildren.length; i++) {
      if (counter < fragments.length && navChildren[i].getNavName() === fragments[counter]) {
        return updateDisplayPath(navChildren, i, that, fragments, counter, nodeName);
      }
    }
    if (nodeName) {
      return Promise.resolve(lex.get({
        key: "hierarchy.node.format",
        args: [nodeName]
      }));
    } else {
      return Promise.resolve(false);
    }
  };

  /**
   * Updae the display path
   * @param navChildren
   * @param i
   * @param that
   * @param fragments
   * @param counter
   * @param nodeName
   */
  function updateDisplayPath(navChildren, i, that, fragments, counter, nodeName) {
    return navChildren[i].getNavChildren().then(function (nodes) {
      return that.getHierarchyNodeDisplayPath(fragments, counter + 1, nodes, nodeName + "/" + navChildren[i].getNavDisplayName());
    });
  }

  /**
   * Get the node path (Internal method)
   * @param nodeName
   * @param nodeOrd
   */
  exports.getConfigNodeDisplayPath = function (nodeName, nodeOrd) {
    var that = this;
    var conf = {
      lease: true
    };
    return nodeOrd.get(conf).then(function (c) {
      var p = c.getParent();
      if (p !== null && p.getDisplayName()) {
        return that.getDisplayNodePath(baja.SlotPath.unescape(p.getDisplayName()) + "/" + nodeName, p.getNavOrd());
      } else {
        return Promise.resolve(nodeName);
      }
      // eslint-disable-next-line handle-callback-err
    })["catch"](function (err) {
      return Promise.resolve(false);
    });
  };

  /**
   * Private function to know if group components have any children.
   * @param grpComponents
   * @returns {boolean}
   */
  exports.hasAnyChildren = function (grpComponents) {
    var flag = false;
    _.each(grpComponents, function (component, index) {
      // If node has any children, set the flag to true and end the _.each
      if (component.getSlots().properties().is("analytics:AnalyticNode").getSize() > 1) {
        flag = true;
        return !flag;
      }
    });
    // Ungroup cmd can be applied on node(s) which have children.
    return flag && grpComponents.length > 1;
  };

  /**
   * Get a random UUID.
   */
  exports.getRandomUuid = function (offset) {
    var currentMillis = baja.AbsTime.now().getMillis() - offset;
    return currentMillis;
  };

  /**
   * @param {baja.Ord} ord
   * @returns {boolean}
   */
  exports.isHierarchyOrd = function (ord) {
    try {
      return !!ord.parse().get('hierarchy');
    } catch (e) {
      return false;
    }
  };

  /**
   * Get the baseline model.
   * @param nodeModel
   * @param group
   */
  exports.getBaselineGroupName = function (groups, groupName) {
    var grpName = '';
    _.each(groups, function (value, index) {
      if (groupName === value.tag + '') {
        grpName = value.display;
      }
    });
    return baja.SlotPath.unescape(grpName);
  };

  /**
   * Get the baseline model.
   * @param nodeModel
   * @param group
   */
  exports.getBaselineNode = function (nodeModel, guid) {
    var node = false;
    _.each(nodeModel, function (value, key) {
      if (guid === value.guid + '') {
        node = value;
      }
    });
    return node;
  };
  return exports;
});
