function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2024 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

define(['baja!', 'lex!tagdictionary', 'log!nmodule.tagdictionary.rc.util.relationUtil', 'Promise', 'underscore', 'nmodule/tagdictionary/rc/util/taggingUtil', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (baja, lex, log, Promise, _, taggingUtil, compUtils, typeUtils) {
  'use strict';

  var logFine = log.fine.bind(log),
    logError = log.severe.bind(log),
    _lex = _slicedToArray(lex, 1),
    tagdictionaryLex = _lex[0],
    constant = _.constant,
    toDisplaySlotPathString = compUtils.toDisplaySlotPathString,
    workOutName = taggingUtil.workOutName,
    isKnob = typeUtils.isKnob,
    isRelationKnob = typeUtils.isRelationKnob,
    RELATION_SLOT = 'r?',
    LINK_SLOT = 'Link?',
    FINE = 'FINE',
    LINK_NOT_FOUND = 'Not Found';
  var RELATION_MANAGER_MODEL_VIEW_STYLE_CLASS = '-t-RelationManager-modelView-container',
    RELATION_MANAGER_MODEL_VIEW_SELECTOR = '.' + RELATION_MANAGER_MODEL_VIEW_STYLE_CLASS,
    IMPLIED_ROW_STYLE_CLASS = '-t-RelationManager-implied',
    DATA_LINK_DISPLAY_STR = tagdictionaryLex.getSafe('RelationManager.dataLink.display'),
    RELATION_ICON = baja.Icon.make('module://icons/x16/arrowRight.png'),
    LINK_ICON = baja.Icon.make('module://icons/x16/link.png');
  var INBOUND_PATH = 'inboundPath',
    INBOUND_PATH_HOVER_TEXT = 'inboundPathHoverText',
    RELATION_IN = 'relationIn',
    THIS_COMP = 'thisComp',
    THIS_COMP_HOVER_TEXT = 'thisCompHoverText',
    RELATION_OUT = 'relationOut',
    OUTBOUND_PATH = 'outboundPath',
    OUTBOUND_PATH_HOVER_TEXT = 'outboundPathHoverText',
    RELATION_TAGS = 'tags',
    RELATION_TAGS_HOVER_TEXT = 'tagsHoverText',
    RELATION_TYPE = 'type';
  var RELATION_TYPE_ENUM = baja.DynamicEnum.make({
    ordinal: 0,
    range: baja.EnumRange.make({
      options: baja.Facets.make({
        lexicon: 'tagdictionary'
      }),
      tags: ['dr',
      // direct relation
      'ir',
      // implied relation
      'lk',
      // link
      'tg',
      // tag group
      'kb',
      // knob
      'rk' // relation know
      ],
      ordinals: [0, 1, 2, 3, 4, 5]
    })
  });
  var RELATION_TYPE_DIRECT = RELATION_TYPE_ENUM.get('dr'),
    RELATION_TYPE_IMPLIED = RELATION_TYPE_ENUM.get('ir'),
    RELATION_TYPE_LINK = RELATION_TYPE_ENUM.get('lk'),
    RELATION_TYPE_TAG_GROUP = RELATION_TYPE_ENUM.get('tg'),
    RELATION_TYPE_KNOB = RELATION_TYPE_ENUM.get('kb'),
    RELATION_TYPE_RELATION_KNOB = RELATION_TYPE_ENUM.get('rk');

  /**
   * find a source ord string
   *
   * @param  {baja.Relation|Object} relation a baja:Relation, Knob or RelationKnob,
   * @returns {String}
   */
  var findSourceOrdString = function findSourceOrdString(relation) {
    var sourceOrdStr;
    if (baja.hasType(relation)) {
      if (relation.getType().is('baja:Relation')) {
        // relations or links
        sourceOrdStr = relation.getSourceOrd().toString();
      }
    } else {
      if (isKnob(relation)) {
        sourceOrdStr = relation.getTargetOrd && relation.getTargetOrd().toString();
      } else if (isRelationKnob(relation)) {
        sourceOrdStr = sourceOrdStr || relation.getRelationOrd && relation.getRelationOrd().toString();
      }
    }
    return sourceOrdStr || '';
  };

  /**
   * formats a path string
   *
   * @param {baja.Component} component
   * @param  {baja.Relation|{*}} relation a baja:Relation, Knob or RelationKnob,
   * @param {String} slotStr
   * @returns {String}
   */
  var makePathStr = function makePathStr(component, relation, slotStr) {
    var pathStr;
    if (component) {
      if (component.getDisplayName()) {
        pathStr = component.getDisplayName() + (slotStr ? ' / ' + slotStr : '');
      }
    } else {
      pathStr = tagdictionaryLex.get({
        key: 'RelationManager.ord.unresolved',
        args: [findSourceOrdString(relation)]
      });
    }
    if (!pathStr) {
      pathStr = workOutName(component) + (slotStr ? ' / ' + slotStr : '');
    }
    return pathStr;
  };

  /**
   * formats a hovertext string
   *
   * @param {baja.Component} component
   * @param  {baja.Relation|{*}} relation a baja:Relation, Knob or RelationKnob,
   * @returns {String}
   */
  var makeHovertextStr = function makeHovertextStr(component, relation) {
    var hovertextStr;
    if (component) {
      hovertextStr = toDisplaySlotPathString(component);
    } else {
      hovertextStr = tagdictionaryLex.get({
        key: 'RelationManager.ord.unresolved',
        args: [findSourceOrdString(relation)]
      });
    }
    if (!hovertextStr) {
      hovertextStr = workOutName(component);
    }
    return hovertextStr;
  };

  /**
   * returns a display string for the component
   *
   * @param {baja.Component} component
   * @returns {String}
   */
  var makeDisplayName = function makeDisplayName(component) {
    var displayName = component.getDisplayName();
    if (!displayName) {
      displayName = workOutName(component);
    }
    return displayName;
  };

  /**
   * formats a string from a relation's tags
   *
   * @param  {Array<module:baja/tag/Tag>} relationTags,
   * @returns {String}
   */
  var relationTagsToString = function relationTagsToString(relationTags) {
    var tagsStr = '';
    var tags = relationTags.getAll();
    if (tags.length === 0) {
      return tagsStr;
    }
    var first = true;
    _.each(tags, function (tag) {
      var valueStr = tag.getValue().toString();
      var tagValueType = tag.getValue().getType();
      if (tagValueType.is('baja:Marker')) {
        valueStr = '';
      }
      tagsStr += (first ? '' : '\n') + tag.getId().toString() + (valueStr ? ' -> ' + valueStr : '');
      first = false;
    });
    return tagsStr;
  };

  /**
   * Formats a string from a relation's tags when they are Facets, from a Knob or a Relation Knob
   *
   * @param  {bajaScript/baja/obj/Facets} facets
   * @returns {String}
   */

  var relationTagsAsFacetsToString = function relationTagsAsFacetsToString(facets) {
    var tagsStr = '';
    var first = true;
    facets.getKeys().forEach(function (key) {
      if (first) {
        first = false;
      } else {
        tagsStr += '\n';
      }
      tagsStr += baja.SlotPath.unescape(key);
      var tagValue = facets.get(key);
      if (tagValue !== baja.Marker.DEFAULT) {
        tagsStr += ' -> ' + tagValue;
      }
    });
    return tagsStr;
  };

  /**
   * A utility class for shared functions related to relations
   *
   * API Status: **Private**
   * @exports nmodule/tagdictionary/rc/util/relationUtil
   * @since Niagara 4.15
   */
  var exports = {};
  exports.RELATION_MANAGER_MODEL_VIEW_STYLE_CLASS = RELATION_MANAGER_MODEL_VIEW_STYLE_CLASS;
  exports.RELATION_MANAGER_MODEL_VIEW_SELECTOR = RELATION_MANAGER_MODEL_VIEW_SELECTOR;
  exports.IMPLIED_ROW_STYLE_CLASS = IMPLIED_ROW_STYLE_CLASS;
  exports.RELATION_SLOT = RELATION_SLOT;
  exports.LINK_SLOT = LINK_SLOT;
  exports.INBOUND_PATH = INBOUND_PATH;
  exports.INBOUND_PATH_HOVER_TEXT = INBOUND_PATH_HOVER_TEXT;
  exports.RELATION_IN = RELATION_IN;
  exports.THIS_COMP = THIS_COMP;
  exports.THIS_COMP_HOVER_TEXT = THIS_COMP_HOVER_TEXT;
  exports.RELATION_OUT = RELATION_OUT;
  exports.OUTBOUND_PATH = OUTBOUND_PATH;
  exports.OUTBOUND_PATH_HOVER_TEXT = OUTBOUND_PATH_HOVER_TEXT;
  exports.RELATION_TAGS = RELATION_TAGS;
  exports.RELATION_TAGS_HOVER_TEXT = RELATION_TAGS_HOVER_TEXT;
  exports.RELATION_TYPE = RELATION_TYPE;
  exports.RELATION_TYPE_DIRECT = RELATION_TYPE_DIRECT;
  exports.RELATION_TYPE_IMPLIED = RELATION_TYPE_IMPLIED;
  exports.RELATION_TYPE_LINK = RELATION_TYPE_LINK;
  exports.RELATION_TYPE_TAG_GROUP = RELATION_TYPE_TAG_GROUP;
  exports.RELATION_TYPE_KNOB = RELATION_TYPE_KNOB;
  exports.RELATION_TYPE_RELATION_KNOB = RELATION_TYPE_RELATION_KNOB;
  exports.makePathStr = makePathStr;
  exports.makeHovertextStr = makeHovertextStr;
  exports.makeDisplayName = makeDisplayName;
  exports.findSourceOrdString = findSourceOrdString;
  exports.relationTagsToString = relationTagsToString;
  exports.relationTagsAsFacetsToString = relationTagsAsFacetsToString;

  /**
   * Make a row subject for a relation.
   *
   * @param {baja.Relation} relation,
   * @param {Boolean} implied
   * @param {baja.Component} comp
   * @returns {Promise.<Object>}
   */
  exports.makeRelationSubject = function (relation, implied, comp) {
    var endpoint;
    var isLink = relation.getType && relation.getType().is('baja:Link');
    return relation.getEndpointOrd().get({
      base: baja.station
    }).then(function (resolvedEndpoint) {
      endpoint = resolvedEndpoint;
    })["catch"](function (err) {
      // #getEndpointOrd()#get() can throw a BoxError/PermissionException if the user
      // does not have read permissions on the endpoint (eg a TagGroupInfo under the TagDictionary).
      // Currently the BoxError gets logged anyway, and here this isn't necessarily a problem,
      // therefore fine logging seems appropriate.
      if (log.isLoggable(FINE)) {
        logFine(err);
      }
    }).then(function () {
      return Promise.all([relation.tags(), isLink && relation.getType().getDisplayName({})]);
    }).then(function (_ref) {
      var _ref2 = _slicedToArray(_ref, 2),
        relationTags = _ref2[0],
        linkTypeDisplayName = _ref2[1];
      var inbound = relation.getInbound(),
        subject = {};
      subject.relation = constant(relation);
      subject.relationId = constant(relation.getId());
      subject.isImplied = constant(!!implied);
      subject.otherComponent = constant(endpoint);
      subject.getIcon = constant(isLink ? LINK_ICON : RELATION_ICON);
      subject.getInbound = constant(relation.getInbound());
      var outboundPath = '',
        outboundPathHoverText = '',
        relationOut = '',
        thisCompStr = makePathStr(comp, relation),
        relationIn = '',
        inboundPath = '',
        inboundPathHoverText = '';
      if (inbound) {
        relationIn = relation.getId().getQName();
        if (isLink) {
          thisCompStr = makePathStr(comp, relation, relation.getTargetSlotNameDisplay());
          inboundPath = makePathStr(endpoint, relation, relation.getSourceSlotNameDisplay());
          inboundPathHoverText = makeHovertextStr(endpoint, relation);
          relationIn = linkTypeDisplayName;
        } else {
          // not a link

          inboundPath = makePathStr(endpoint, relation);
          inboundPathHoverText = makeHovertextStr(endpoint, relation);
        }
      } else {
        // outbound

        relationOut = relation.getId().getQName();
        if (!isLink) {
          outboundPath = makePathStr(endpoint, relation);
          outboundPathHoverText = makeHovertextStr(endpoint, relation);
        }
        // not expecting an outbound link, would expect that to come in as a knob
      }
      subject[OUTBOUND_PATH] = outboundPath;
      subject[OUTBOUND_PATH_HOVER_TEXT] = outboundPathHoverText;
      subject[RELATION_OUT] = relationOut;
      subject[THIS_COMP] = thisCompStr;
      subject[THIS_COMP_HOVER_TEXT] = toDisplaySlotPathString(comp);
      subject[RELATION_IN] = relationIn;
      subject[INBOUND_PATH] = inboundPath;
      subject[INBOUND_PATH_HOVER_TEXT] = inboundPathHoverText;
      subject[RELATION_TAGS] = relationTags.getAll();
      subject[RELATION_TAGS_HOVER_TEXT] = relationTagsToString(relationTags);
      var relationType;
      if (isLink) {
        relationType = RELATION_TYPE_LINK;
      } else if (relation.getId().getQName() === 'n:tagGroup') {
        relationType = RELATION_TYPE_TAG_GROUP;
      } else {
        relationType = implied ? RELATION_TYPE_IMPLIED : RELATION_TYPE_DIRECT;
      }
      subject.isEditable = constant(relationType === RELATION_TYPE_LINK || relationType === RELATION_TYPE_DIRECT);
      subject[RELATION_TYPE] = relationType;
      return subject;
    });
  };

  /**
   * Make a row subject for a knob.
   *
   * @param  {Object} knob,
   * @param {baja.Component} comp
   * @returns {Promise.<Object>}
   */
  exports.makeKnobSubject = function (knob, comp) {
    var target,
      isEditable = true;

    // a knob object has the following functions available:
    //   getId (a number, not a tag Id)
    //   getSourceComponent
    //   getSourceSlotName
    //   getTargetOrd
    //   getTargetSlotName
    // see bson.js #decodeKnob

    // in this context the sourceComponent is the component we have performed #getKnobs on, ie the component the view is on.

    var link;
    return knob.getTargetOrd().get({
      base: baja.station,
      lease: true
    }).then(function (resolvedTarget) {
      target = resolvedTarget;
    })["catch"](function (err) {
      if (log.isLoggable(FINE)) {
        logFine(err);
      }
    }).then(function () {
      var linkName = knob.getLinkName();
      if (linkName) {
        link = target.get(linkName);
        if (!link) {
          if (log.isLoggable(FINE)) {
            logFine('makeKnobSubject: link slot: [{}] not found on target: [{}] - may not exist on the target any more', linkName, makeDisplayName(target));
          }
          isEditable = false;
          return LINK_NOT_FOUND;
        }
        var linkType = link.getType();
        return linkType.getDisplayName({});
      }
      isEditable = false; // cannot edit if we do not know here it is on the target component
      return DATA_LINK_DISPLAY_STR;
    }).then(function (linkTypeDisplayName) {
      if (linkTypeDisplayName === LINK_NOT_FOUND) {
        return;
      }
      var subject = {};
      subject.relation = constant(knob);
      subject.relationId = constant(new baja.Id('n:dataLink'));
      subject.otherComponent = constant(target);
      subject.isImplied = constant(!isEditable); // treat as implied if not editable
      subject.getIcon = constant(LINK_ICON);

      // a knob is the other side of an inbound link, so count this as outbound
      subject.getInbound = constant(false);
      var outboundPath = makePathStr(target, knob, knob.getTargetSlotName());
      var outboundPathHoverText = makeHovertextStr(target, knob);
      subject[OUTBOUND_PATH] = outboundPath;
      subject[OUTBOUND_PATH_HOVER_TEXT] = outboundPathHoverText;
      subject[RELATION_OUT] = linkTypeDisplayName;
      subject[THIS_COMP] = makePathStr(comp, knob, knob.getSourceSlotName());
      subject[THIS_COMP_HOVER_TEXT] = toDisplaySlotPathString(comp);
      subject[RELATION_IN] = '';
      subject[INBOUND_PATH] = '';
      var linkTags = link ? link.getRelationTags() : undefined,
        linkTagsHovertext = linkTags ? relationTagsAsFacetsToString(linkTags) : '';
      subject[RELATION_TAGS] = linkTags;
      subject[RELATION_TAGS_HOVER_TEXT] = linkTagsHovertext;
      subject[RELATION_TYPE] = RELATION_TYPE_KNOB;
      subject.isEditable = constant(isEditable);
      return subject;
    });
  };

  /**
   * Make a row subject for a relationKnob.
   *
   * @param  {Object} relationKnob
   * @param {baja.Component} comp
   * @returns {Promise.<Object>}
   */
  exports.makeRelationKnobSubject = function (relationKnob, comp) {
    var resolvedTarget;

    // a relationKnob object has the following functions available: see bson.js #decodeRelationKnob
    //   getEndpointComponent
    //   getId (a number, not a tag Id)
    //   getRelationId
    //   getRelationOrd
    //   getRelationTags
    // see bson.js #decodeRelationKnob

    // in this context the endpointComponent is the component we have performed #getKnobs on, ie the component the view is on.

    return relationKnob.getRelationOrd().get({
      base: baja.station,
      lease: true
    }).then(function (resolved) {
      resolvedTarget = resolved;
    })["catch"](function (err) {
      if (log.isLoggable(FINE)) {
        logFine(err);
      }
    }).then(function () {
      var relationName = relationKnob.getRelationName(),
        relation = resolvedTarget.get(relationName);
      if (!relation) {
        if (log.isLoggable(FINE)) {
          logFine('makeRelationKnobSubject: relation slot: [{}] not found on target: [{}] - may not exist on the target any more', relationName, makeDisplayName(resolvedTarget));
        }
        return;
      }
      var subject = {};
      subject.relation = constant(relationKnob);
      subject.relationId = constant(relationKnob.getRelationId());
      subject.otherComponent = constant(resolvedTarget);
      subject.isImplied = constant(false);
      subject.getIcon = constant(RELATION_ICON);

      // a relationKnob is the other side of a direct relation, they are outbound, so count this as inbound
      subject.getInbound = constant(true);
      subject[OUTBOUND_PATH] = '';
      subject[RELATION_OUT] = '';
      subject[THIS_COMP] = makePathStr(comp, relationKnob);
      subject[THIS_COMP_HOVER_TEXT] = toDisplaySlotPathString(comp);
      subject[RELATION_IN] = relationKnob.getRelationId();
      var inboundPath = makePathStr(resolvedTarget, relationKnob);
      var inboundPathHoverText = makeHovertextStr(resolvedTarget, relationKnob);
      subject[INBOUND_PATH] = inboundPath;
      subject[INBOUND_PATH_HOVER_TEXT] = inboundPathHoverText;
      var relationTags = relation.getRelationTags();
      subject[RELATION_TAGS] = relationTags;
      subject[RELATION_TAGS_HOVER_TEXT] = relationTagsAsFacetsToString(relationTags);
      subject[RELATION_TYPE] = RELATION_TYPE_RELATION_KNOB;
      subject.isEditable = constant(true);
      return subject;
    });
  };

  /**
   * Finds the actual relation, slot and parent for the relation passed in.
   * (For Knobs and RelationKnobs, it will return the actual relation, parent and slotName, rather than the knob info)
   *
   * @param {baja.Relation} relationIn a relation or a link
   * @param {baja.Component} parentIn the parent of the relation
   * @returns {Promise.<Object>} an object literal with the relation, its parent and its slot on the parent.
   */
  exports.findRelationAndComponentToEdit = function (relationIn, parentIn) {
    var isKb = isKnob(relationIn),
      isRelKb = isRelationKnob(relationIn);
    var relationToEdit, parentToEdit, relationOrd, slotOnParent;
    if (isKb) {
      relationOrd = relationIn.getTargetOrd();
    } else if (isRelKb) {
      relationOrd = relationIn.getRelationOrd();
    }
    var resolvedRelation;
    return Promise.resolve(relationOrd ? relationOrd.get({
      base: baja.station,
      lease: true
    }) : null).then(function (resolved) {
      resolvedRelation = resolved;
    }).then(function () {
      if (resolvedRelation) {
        if (isKb) {
          slotOnParent = relationIn.getLinkName();
        } else if (isRelKb) {
          slotOnParent = relationIn.getRelationName();
        }
        relationToEdit = resolvedRelation.get(slotOnParent);
        parentToEdit = resolvedRelation;
      } else {
        relationToEdit = relationIn;
        slotOnParent = relationIn.getName();
        parentToEdit = parentIn;
      }
    })["catch"](function (err) {
      logError('expected to resolve: {} but got error: {}', relationOrd, err);
    }).then(function () {
      return {
        relation: relationToEdit,
        parent: parentToEdit,
        slotOnParent: slotOnParent
      };
    });
  };
  return exports;
});
