/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/wiresheet/rc/wb/layout/ComponentGlyphLayoutStrategy
 */
define(['underscore'], function (_) {
  'use strict';

  var any = _.any,
    isEqual = _.isEqual;

  ////////////////////////////////////////////////////////////////
  // ComponentGlyphLayoutStrategy
  ////////////////////////////////////////////////////////////////

  /**
   * @class
   * @alias module:nmodule/wiresheet/rc/wb/layout/ComponentGlyphLayoutStrategy
   * @implements module:nmodule/wiresheet/rc/typedefs~VertexLayoutStrategy
   */
  var ComponentGlyphLayoutStrategy = function ComponentGlyphLayoutStrategy() {};

  /**
   * Perform the layout of a `ComponentGlyph` entity. Existing layout data (such
   * as derived from a `wsAnnotation`) will be respected first, otherwise a
   * default position will be chosen. Layout data will be stored in wixels.
   *
   * @private
   * @param {module:nmodule/wiresheet/rc/typedefs~WiresheetEntity} obj the vertex
   * containing the `ComponentGlyph`
   * @param {module:nmodule/wiresheet/rc/wb/layout/Mask} mask
   * @returns {boolean} true if the layout was updated
   */
  ComponentGlyphLayoutStrategy.prototype.update = function (obj, mask) {
    var glyphLayout = obj.glyph.layout;
    var updated;
    if (glyphLayout) {
      var newLayout = _.extend({}, glyphLayout, {
        height: defaultHeight(obj)
      });
      updated = !isEqual(newLayout, obj.layout);
      obj.layout = newLayout;
    } else if (!obj.layout) {
      updated = true;
      placeDefault(obj, mask);
    } else {
      var newHeight = defaultHeight(obj);
      updated = newHeight !== obj.layout.height;
      obj.layout.height = newHeight;
    }
    return updated;
  };

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  function defaultHeight(obj) {
    return obj.glyph.bars.length + 3;
  }
  function placeDefault(obj, mask) {
    var hasExisting = function hasExisting(wixels) {
      var tiles = mask.getTiles(wixels, wixels);
      return any(tiles, function (tile) {
        var entity = tile.entity;
        var glyph = entity && entity.glyph;
        if (!glyph || glyph.type !== 'ComponentGlyph') {
          return false;
        }
        var layout = entity.layout;
        return layout && layout.x === wixels && layout.y === wixels;
      });
    };
    var wixels = 2;
    while (hasExisting(wixels)) {
      wixels += 4;
    }
    obj.layout = {
      x: wixels,
      y: wixels,
      width: 8,
      height: defaultHeight(obj)
    };
  }
  return ComponentGlyphLayoutStrategy;
});
