/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/* jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/wiresheet/rc/wb/layout/AntRaceRoutingStrategy
 */
define(['underscore', 'nmodule/wiresheet/rc/wb/WbConstants', 'nmodule/wiresheet/rc/wb/util/wsUtils'], function (_, WbConstants, wsUtils) {
  'use strict';

  var boundingBox = wsUtils.boundingBox,
    getLinkStartAndEnd = wsUtils.getLinkStartAndEnd,
    getRoadTile = wsUtils.getRoadTile,
    isUsableWixel = wsUtils.isUsableWixel,
    toDirection = wsUtils.toDirection,
    translate = wsUtils.translate,
    LINK_H = WbConstants.LINK_H,
    LINK_V = WbConstants.LINK_V;

  ////////////////////////////////////////////////////////////////
  // AntRaceRoutingStrategy
  ////////////////////////////////////////////////////////////////

  /**
   * Strategy for laying out `SnakeGlyph`s using an ant-racing algorithm.
   * Replicates the Workbench logic in `LinkSnakeGlyph#routeAnts()`.
   *
   * @class
   * @alias module:nmodule/wiresheet/rc/wb/layout/AntRaceRoutingStrategy
   */
  var AntRaceRoutingStrategy = function AntRaceRoutingStrategy() {};

  /**
   * Route one link using an ant-racing algorithm.
   *
   * @param {module:nmodule/wiresheet/rc/typedefs~WiresheetTriple} triple the
   * triple containing the `SnakeGlyph`
   * @param {Array.<Array.<number>>} mask
   * @returns {{ layout: object, segments: Array.<{ x: number, y: number }>}|null}
   * an object with desired `layout` and `segments` properties, or `null` if
   * ant-race routing was not possible
   */
  AntRaceRoutingStrategy.prototype.route = function (triple, mask) {
    var _getLinkStartAndEnd = getLinkStartAndEnd(triple),
      start = _getLinkStartAndEnd.start,
      end = _getLinkStartAndEnd.end;
    if (!start || !end) {
      return null;
    }
    return routeAnts(start, end, mask, triple.predicate.glyph);
  };

  /**
   * Perform the ant-race algorithm to derive desired layout and routed
   * segments.
   *
   * Ant-racing is essentially a flood-fill algorithm. It begins one wixel to
   * the right of the start point, and on each tick, attempts to put an "ant" on
   * the unvisited wixels in all four directions, and then disappears (marking
   * its wixel as having been visited on the current tick).
   *
   * The next tick, each previously placed ant will attempt to place another ant
   * in all four directions, and so on. The flood of ants expands in a diamond
   * pattern, potentially touching every wixel in the wiresheet, until one of
   * them happens on the end point.
   *
   * When the end point is hit, it then walks backwards in tick-order to create
   * the final path from start to end.
   *
   * @param {{ x: number, y: number }} start start point
   * @param {{ x: number, y: number }} end end point
   * @param mask
   * @returns {{ layout: Object, segments: Array.<{ x: number, y: number }> }}
   */
  function routeAnts(_ref, _ref2, mask, glyph) {
    var startX = _ref.x,
      startY = _ref.y;
    var endX = _ref2.x,
      endY = _ref2.y;
    var board = makeBoard(mask.getWidth(), mask.getHeight());
    var time = 1,
      lastAnts = [{
        x: startX + 1,
        y: startY
      }],
      currentAnts = [];
    var canSpawn = function canSpawn(x, y, tile) {
      return x >= 0 && x < mask.getWidth() && y >= 0 && y < board.length && !board[y][x] && isUsableWixel(mask, glyph, x, y, tile);
    };
    var spawnAnt = function spawnAnt(x, y) {
      return currentAnts.push({
        x: x,
        y: y
      });
    };
    var spawnAnts = function spawnAnts(x, y) {
      if (canSpawn(x, y - 1, LINK_V)) {
        spawnAnt(x, y - 1);
      }
      if (canSpawn(x, y + 1, LINK_V)) {
        spawnAnt(x, y + 1);
      }
      if (canSpawn(x - 1, y, LINK_H)) {
        spawnAnt(x - 1, y);
      }
      if (canSpawn(x + 1, y, LINK_H)) {
        spawnAnt(x + 1, y);
      }
    };

    // we found the end point: now walk backwards to build the path
    var buildAntRoad = function buildAntRoad() {
      var lastX = endX,
        lastY = endY,
        thisX = endX - 1,
        thisY = endY,
        points = [{
          x: endX,
          y: endY
        }];
      var availableCell = function availableCell(nextX, nextY, time) {
        if (board[nextY][nextX] !== time) {
          return false;
        }
        var tile = getRoadTile(lastX, lastY, thisX, thisY, nextX, nextY);
        return isUsableWixel(mask, glyph, thisX, thisY, tile);
      };
      var finished = function finished() {
        return thisX === startX && thisY === startY;
      };
      while (!finished()) {
        var t = time - 1,
          nextX = thisX,
          nextY = thisY;
        if (availableCell(thisX, thisY + 1, t)) {
          nextY++;
        } else if (availableCell(thisX - 1, thisY, t)) {
          nextX--;
        } else if (availableCell(thisX + 1, thisY, t)) {
          nextX++;
        } else if (availableCell(thisX, thisY - 1, t)) {
          nextY--;
        } else {
          // even though there was a clear path, it could still be unusable due
          // to routing restrictions
          return;
        }
        points.push({
          x: thisX,
          y: thisY
        });
        lastX = thisX;
        lastY = thisY;
        thisX = nextX;
        thisY = nextY;
        time--;
      }
      points.push({
        x: startX,
        y: startY
      });
      points.reverse(); //put in start -> finish order

      var layout = boundingBox(points),
        segments = consolidateSegments(points.map(translate.bind(null, {
          x: -layout.x,
          y: -layout.y
        })));
      return {
        layout: layout,
        segments: segments
      };
    };

    // perform the logic: spawn ants until we reach the end point
    board[startY][startX] = time;
    while (lastAnts.length) {
      ++time;
      for (var i = 0; i < lastAnts.length; i++) {
        var _lastAnts$i = lastAnts[i],
          antX = _lastAnts$i.x,
          antY = _lastAnts$i.y;
        if (!board[antY][antX]) {
          board[antY][antX] = time;
          if (antX === endX - 1 && antY === endY) {
            return buildAntRoad();
          }
          spawnAnts(antX, antY);
        }
      }
      lastAnts = currentAnts;
      currentAnts = [];
    }
  }

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  function makeBoard(width, height) {
    var board = [];
    for (var i = 0, len = height; i < len; ++i) {
      board[i] = new Uint16Array(width);
    }
    return board;
  }

  /**
   * Given an array of all visited points, condense down to only the junction
   * points where the direction changes.
   * @param {Array.<{x: number, y: number}>} points
   * @returns {Array.<{x: number, y: number}>}
   */
  function consolidateSegments(points) {
    var segments = [points[0]],
      lastDir = toDirection(points[0], points[1]);
    _.each(_.initial(points), function (p, i) {
      var dir = toDirection(p, points[i + 1]);
      if (dir !== lastDir) {
        segments.push(p);
      }
      lastDir = dir;
    });
    segments.push(_.last(points));
    return segments;
  }
  return AntRaceRoutingStrategy;
});
