function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/wiresheet/rc/wb/baja/GlyphFactory
 */
define(['baja!', 'lex!baja', 'Promise', 'underscore', 'nmodule/gx/rc/baja/Color', 'nmodule/gx/rc/baja/Font', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/wb/mgr/componentStatusUtils', 'nmodule/wiresheet/rc/wb/WbConstants', 'nmodule/wiresheet/rc/wb/baja/bajaUtils'], function (baja, lexs, Promise, _, Color, Font, compUtils, typeUtils, componentStatusUtils, WbConstants, bajaUtils) {
  'use strict';

  var difference = _.difference,
    extend = _.extend,
    uniq = _.uniq;
  var getForeground = componentStatusUtils.getForeground,
    getBackground = componentStatusUtils.getBackground;
  var getTypeDisplayName = typeUtils.getTypeDisplayName,
    isKnob = typeUtils.isKnob,
    isRelationKnob = typeUtils.isRelationKnob;
  var componentToId = bajaUtils.componentToId,
    getLink = bajaUtils.getLink,
    isComponent = bajaUtils.isComponent,
    isLink = bajaUtils.isLink,
    isRelation = bajaUtils.isRelation,
    isTextBlock = bajaUtils.isTextBlock,
    ordToComponentId = bajaUtils.ordToComponentId;
  var toDisplayPathString = compUtils.toDisplayPathString;
  var LINK_COLOR_LON = WbConstants.LINK_COLOR_LON,
    LINK_COLOR_NORMAL = WbConstants.LINK_COLOR_NORMAL,
    LINK_COLOR_RELATION = WbConstants.LINK_COLOR_RELATION;
  var _lexs = _slicedToArray(lexs, 1),
    bajaLex = _lexs[0];

  ////////////////////////////////////////////////////////////////
  // GlyphFactory
  ////////////////////////////////////////////////////////////////

  /**
   * GlyphFactory converts Baja values into raw JSON display data suitable for
   * rendering.
   *
   * It will perform no ORD resolutions or leasing; methods for generating JSON
   * for links or relations will have their own requirements regarding
   * pre-retrieving the components on the other end of the link or relation.
   *
   * @class
   * @alias module:nmodule/wiresheet/rc/wb/baja/GlyphFactory
   * @param {Object} [params]
   * @param {Boolean} [params.showRelations] true if relations should be
   * shown along with slot data
   * @param {Boolean} [params.showLinks] true if links should be
   * shown along with slot data
   * @param {baja.Component} [params.base=baja.station] a base component to use
   * to resolve ORDs
   */
  var GlyphFactory = function GlyphFactory() {
    var _ref = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      _ref$showRelations = _ref.showRelations,
      showRelations = _ref$showRelations === void 0 ? true : _ref$showRelations,
      _ref$showLinks = _ref.showLinks,
      showLinks = _ref$showLinks === void 0 ? true : _ref$showLinks,
      base = _ref.base;
    this.$showRelations = showRelations;
    this.$showLinks = showLinks;
    this.$base = base || baja.station;
  };

  /**
   * @returns {boolean} true if relations should be shown along with slot data
   */
  GlyphFactory.prototype.isShowRelations = function () {
    return !!this.$showRelations;
  };

  /**
   * @returns {boolean} true if links should be shown along with slot data
   */
  GlyphFactory.prototype.isShowLinks = function () {
    return !!this.$showLinks;
  };

  /**
   * @param {Object} [options]
   * @param {boolean} options.showLinks
   * @param {boolean} options.showRelations
   */
  GlyphFactory.prototype.setOptions = function () {
    var _ref2 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      showLinks = _ref2.showLinks,
      showRelations = _ref2.showRelations;
    this.$showLinks = !!showLinks;
    this.$showRelations = !!showRelations;
  };

  /**
   * Given a Component child of the wiresheet container, create JSON data for a
   * `ComponentGlyph`.
   *
   * @see nmodule/wiresheet/rc/wb/schema/ComponentGlyph.json
   * @param {baja.Component} component
   * @param {Object} [cx] context that will be merged in with slot facets when
   * deriving the display values of properties
   * @returns {Promise.<Object>} promise to be resolved with `ComponentGlyph`
   * JSON
   */
  GlyphFactory.prototype.componentGlyph = function (component, cx) {
    var _this = this;
    if (!isComponent(component)) {
      return Promise.reject(new Error('Component required'));
    }
    var type = component.getType();
    var showRelations = this.isShowRelations();
    var showLinks = this.isShowLinks();
    var summarySlots = GlyphFactory.$getVisibleSlots(component, showRelations, showLinks);
    return Promise.all([Promise.all(summarySlots.map(slotToBar.bind(null, component, cx))), showRelations ? getRelationKnobIds(component) : [], getTypeDisplayName(type)]).then(function (_ref3) {
      var _ref4 = _slicedToArray(_ref3, 3),
        slotBars = _ref4[0],
        relationIds = _ref4[1],
        typeDisplay = _ref4[2];
      var glyph = {
          type: 'ComponentGlyph',
          name: component.getName(),
          header: {
            title: component.getDisplayName(),
            subtitle: typeDisplay,
            icon: component.getIcon().getImageUris(),
            tooltip: toDisplayPathString(component) + ' h:' + component.getHandle() + ' [' + type + ']'
          },
          bars: slotBars.concat(relationIds.map(relationIdToBar)),
          footer: {
            type: _this.isShowRelations() ? 'relation' : 'default'
          },
          uiStatus: {
            selected: false,
            pendingMove: false
          }
        },
        layout = annotationToLayout(component.get('wsAnnotation'));
      if (layout) {
        glyph.layout = layout;
      }
      return glyph;
    });
  };

  /**
   * Given a `baja:Link` child slot of a Component on the wiresheet, where the
   * link's source component is not a child of the wiresheet container, create
   * JSON data for a `StubGlyph`.
   *
   * @see nmodule/wiresheet/rc/wb/schema/StubGlyph.json
   * @param {baja.Struct} link
   * @param {baja.Component} [sourceComp] if the source component of the link
   * is known, pass it for additional tooltip information
   * @returns {Promise.<Object>} promise to be resolved with `StubGlyph` JSON
   */
  GlyphFactory.prototype.linkStubGlyph = function (link, sourceComp) {
    if (!isLink(link)) {
      return Promise.reject(new Error('Link required'));
    }

    //TODO: this might not work for offline? think we need a BajaScript analog
    //of ComponentSpace#handleToSlotPath.

    var targetComp = link.getParent();
    var sourceSlot = link.get('sourceSlotName');
    var targetSlot = link.get('targetSlotName');
    var linkColorCode = isLon(link) ? LINK_COLOR_LON : LINK_COLOR_NORMAL;
    var tooltip = getLinkTooltip(link, sourceComp, sourceSlot, targetComp, targetSlot);
    if (isComposite(link)) {
      tooltip += ' (C)';
    }
    return ordToComponentId(link.get('sourceOrd'), {
      base: targetComp
    }).then(function (sourceId) {
      return {
        type: 'StubGlyph',
        direction: 'in',
        connectorType: 'link',
        sourceConnector: 'slot:' + sourceSlot,
        sourceId: sourceId,
        targetConnector: 'slot:' + targetSlot,
        targetId: componentToId(targetComp),
        linkColorCode: linkColorCode,
        tooltip: tooltip,
        uiStatus: {
          selected: false,
          highlighted: false
        }
      };
    });
  };

  //TODO: BajaScript knobs don't fully implement the Knob interface. need
  //analysis: can getSourceComponent() always return the component on which it's
  //installed, or are there LocalKnob vs. ProxyKnob concerns?

  /**
   * Given a Knob of a Component on the wiresheet, where the knob's target
   * component is not a child of the wiresheet container, create JSON data for
   * a `StubGlyph`.
   *
   * @see nmodule/wiresheet/rc/wb/schema/StubGlyph.json
   * @param {Object} knob - a link knob
   * @param {baja.Component} [targetComp] - if the target component is known,
   * pass it for info on composite/coloring
   * @returns {Promise.<Object>} promise to be resolved with `StubGlyph` JSON
   */
  GlyphFactory.prototype.linkKnobGlyph = function (knob, targetComp) {
    if (!isKnob(knob)) {
      return Promise.reject(new Error('Knob required'));
    }
    var sourceComp = knob.getSourceComponent();
    return Promise.all([targetComp && getLink(targetComp, knob), ordToComponentId(knob.getTargetOrd(), {
      base: sourceComp
    })]).then(function (_ref5) {
      var _ref6 = _slicedToArray(_ref5, 2),
        link = _ref6[0],
        targetId = _ref6[1];
      var composite = link && isComposite(link);
      var tooltip = getKnobTooltip(knob, link) + (composite ? ' (C)' : '');
      var linkColorCode = isLon(link) ? LINK_COLOR_LON : LINK_COLOR_NORMAL;
      return {
        type: 'StubGlyph',
        direction: 'out',
        connectorType: 'link',
        sourceConnector: 'slot:' + knob.getSourceSlotName(),
        sourceId: componentToId(sourceComp),
        targetConnector: 'slot:' + knob.getTargetSlotName(),
        targetId: targetId,
        composite: composite,
        linkColorCode: linkColorCode,
        tooltip: tooltip,
        uiStatus: {
          selected: false,
          highlighted: false
        }
      };
    });
  };

  /**
   * Given a `baja:Link` child slot of a Component on the wiresheet, where both
   * the link's source and target Components are children of the wiresheet
   * container, create JSON data for a `SnakeGlyph`.
   *
   * @see nmodule/wiresheet/rc/wb/schema/SnakeGlyph.json
   * @param {baja.Struct} link a `baja:Link`
   * @param {baja.Component} sourceComp the component at the source of the link
   * @returns {Promise.<Object>} promise to be resolved with `SnakeGlyph` JSON
   */
  GlyphFactory.prototype.linkSnakeGlyph = function (link, sourceComp) {
    if (!isLink(link)) {
      return Promise.reject(new Error('Link required'));
    }
    if (!isComponent(sourceComp)) {
      return Promise.reject(new Error('source component required'));
    }
    var targetComp = link.getParent();
    var sourceSlot = link.get('sourceSlotName'),
      targetSlot = link.get('targetSlotName'),
      tooltip = getLinkTooltip(link, sourceComp, sourceSlot, targetComp, targetSlot);
    return Promise.resolve({
      type: 'SnakeGlyph',
      connectorType: 'link',
      sourceId: componentToId(sourceComp),
      sourceConnector: 'slot:' + sourceSlot,
      targetId: componentToId(targetComp),
      targetConnector: 'slot:' + targetSlot,
      linkColorCode: isLon(link) ? LINK_COLOR_LON : LINK_COLOR_NORMAL,
      tooltip: tooltip,
      uiStatus: {
        selected: false,
        highlighted: false
      }
    });
  };

  /**
   * Given a `baja:Relation` child slot of a Component on the wiresheet, where
   * the relation's target component is not a child of the wiresheet container,
   * create JSON data for a `StubGlyph`.
   *
   * @see nmodule/wiresheet/rc/wb/schema/StubGlyph.json
   * @param {baja.Struct} relation
   * @param {baja.Component} [targetComp] if the relation's target is known,
   * pass it for additional tooltip information
   * @returns {Promise.<Object>} promise to be resolved with `StubGlyph` JSON
   */
  GlyphFactory.prototype.relationStubGlyph = function (relation, targetComp) {
    if (!isRelation(relation)) {
      return Promise.reject(new Error('Relation required'));
    }
    var sourceComp = relation.getParent();
    var relationId = relation.getRelationId();
    var connectorId = 'relation:' + relationId;
    return ordToComponentId(relation.getSourceOrd(), {
      base: sourceComp
    }).then(function (targetId) {
      return {
        type: 'StubGlyph',
        direction: 'out',
        connectorType: 'relation',
        sourceConnector: connectorId,
        sourceId: componentToId(sourceComp),
        targetConnector: connectorId,
        targetId: targetId,
        linkColorCode: LINK_COLOR_RELATION,
        tooltip: getRelationTooltip(sourceComp, relation, targetComp),
        uiStatus: {
          selected: false,
          highlighted: false
        }
      };
    });
  };

  /**
   * Given a relation knob of a Component on the wiresheet, where the relation's
   * source Component is not a child of the wiresheet container, create JSON
   * data for a `StubGlyph`.
   *
   * @see nmodule/wiresheet/rc/wb/schema/StubGlyph.json
   * @param {Object} relationKnob
   * @param {baja.Component} [sourceComp] if the source of the relation is
   * known, pass it for additional tooltip information
   * @returns {Promise.<Object>} promise to be resolved with `StubGlyph` JSON
   */
  GlyphFactory.prototype.relationKnobGlyph = function (relationKnob, sourceComp) {
    if (!isRelationKnob(relationKnob)) {
      return Promise.reject(new Error('Relation knob required'));
    }
    var relationId = relationKnob.getRelationId();
    var targetComp = relationKnob.getEndpointComponent();
    var connectorId = 'relation:' + relationId;
    return ordToComponentId(relationKnob.getRelationOrd(), {
      base: targetComp
    }).then(function (sourceId) {
      return {
        type: 'StubGlyph',
        direction: 'in',
        connectorType: 'relation',
        sourceConnector: connectorId,
        sourceId: sourceId,
        targetConnector: connectorId,
        targetId: componentToId(targetComp),
        linkColorCode: LINK_COLOR_RELATION,
        tooltip: getRelationKnobTooltip(sourceComp, relationKnob, targetComp),
        uiStatus: {
          selected: false,
          highlighted: false
        }
      };
    });
  };

  /**
   * Given a `baja:Relation` child slot of a Component on the wiresheet, where
   * both the relations's source and target Components are children of the
   * wiresheet container, create JSON data for a `SnakeGlyph`.
   *
   * @see nmodule/wiresheet/rc/wb/schema/SnakeGlyph.json
   * @param {baja.Struct} relation
   * @param {baja.Component} targetComp the target of the relation
   * @returns {Promise.<Object>} promise to be resolved with `SnakeGlyph` JSON
   */
  GlyphFactory.prototype.relationSnakeGlyph = function (relation, targetComp) {
    if (!isRelation(relation)) {
      return Promise.reject(new Error('Relation required'));
    }
    if (!isComponent(targetComp)) {
      return Promise.reject(new Error('target component required'));
    }
    var sourceComp = relation.getParent(),
      relationId = relation.getRelationId();
    return Promise.resolve({
      type: 'SnakeGlyph',
      connectorType: 'relation',
      sourceConnector: 'relation:' + relationId,
      sourceId: componentToId(sourceComp),
      targetConnector: 'relation:' + relationId,
      targetId: componentToId(targetComp),
      linkColorCode: LINK_COLOR_RELATION,
      tooltip: getRelationTooltip(sourceComp, relation, targetComp),
      uiStatus: {
        selected: false,
        highlighted: false
      }
    });
  };

  /**
   * Given a `baja:WsTextBlock` child of the wiresheet container, create JSON
   * data for a `TextBlockGlyph`.
   *
   * @see nmodule/wiresheet/rc/wb/schema/TextBlockGlyph.json
   * @param {baja.Component} block a `baja:WsTextBlock`
   * @returns {Promise.<Object>} promise to be resolved with `TextBlockGlyph`
   * JSON
   */
  GlyphFactory.prototype.textBlockGlyph = function (block) {
    if (!isTextBlock(block)) {
      return Promise.reject(new Error('WsTextBlock required'));
    }
    var font = Font.make(block.get('font'));
    var glyph = {
      type: 'TextBlockGlyph',
      text: block.get('text'),
      foreground: Color.make(block.get('foreground')).toCssString(),
      background: Color.make(block.get('background')).toCssString(),
      font: font.toShorthand(),
      underline: font.isUnderline(),
      border: block.get('border'),
      layout: annotationToLayout(block.get('wsAnnotation')),
      uiStatus: {
        selectable: block.get('selectable'),
        selected: false,
        pendingMove: false
      }
    };
    return Promise.resolve(glyph);
  };

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Array of potentials slots to show in a Glyph.
   * Includes,
   * all non-hidden slots
   * all relations slots (with footer) if showRelations is true
   * all slots with links/knobs and showLinks is true
   * any other slot with the SUMMARY flag set.
   * @private
   * @param {baja.Component} comp
   * @param {boolean} showRelations
   * @param {boolean} showLinks
   * @returns {Array.<baja.Slot>}
   */
  GlyphFactory.$getVisibleSlots = function (comp, showRelations, showLinks) {
    var links = comp.getSlots().is('baja:Link').toValueArray();
    var knobs = comp.getKnobs();
    var relationIds = [];
    return comp.getSlots().filter(function (slot) {
      var flags = slot.getFlags();
      var slotName = slot.getName();
      var type = slot.isProperty() && slot.getType();
      if (flags & baja.Flags.HIDDEN) {
        return false;
      }
      if (type && type.is('baja:Component') && !type.is('baja:Vector')) {
        return false;
      }
      if (type && showRelations && type.is('baja:Relation')) {
        var relation = comp.get(slot);
        if (isRelation(relation)) {
          var relationId = toRelationId(relation);
          if (!relationIds.contains(relationId)) {
            // don't double up on relation ids
            relationIds.push(relationId);
            return true;
          }
        }
      }

      //determining if the knob is visible or not may not be worth the extra network call
      return showLinks && (hasVisibleLink(links, slotName) || hasKnob(knobs, slotName) || flags & baja.Flags.SUMMARY);
    }).toArray();
  };
  function hasVisibleLink(links, slotName) {
    for (var i = 0, len = links.length; i < len; ++i) {
      if (isHidden(links[i])) {
        return false;
      }
      if (links[i].get('targetSlotName') === slotName) {
        return true;
      }
      if (links[i].get('targetSlotName') === slotName) {
        return true;
      }
    }
  }
  function hasKnob(knobs, slotName) {
    for (var i = 0, len = knobs.length; i < len; ++i) {
      if (knobs[i].getSourceSlotName() === slotName) {
        return true;
      }
    }
  }

  /**
   * Convert a slot to JSON object representing one bar in a
   * `StdComponentGlyph`.
   *
   * @param {baja.Component} comp containing component
   * @param {Object} cx context to merge into `toString` calls, along with any
   * existing slot facets
   * @param {baja.Slot} slot
   * @returns {Promise.<Object>} promise to be resolved with a JSON object with the
   * data for one bar in the glyph
   */
  function slotToBar(comp, cx, slot) {
    var type,
      display,
      isProp,
      isRelation,
      id,
      value,
      mergedCx = extend({}, cx, comp.getFacets(slot).toObject()),
      tooltip;
    if (slot.isAction()) {
      type = 'action';
      tooltip = "".concat(slot, "(").concat(slot.getReturnType() || 'void', ")");
    } else if (slot.isTopic()) {
      type = 'topic';
      tooltip = "".concat(slot, "! ").concat(slot.getEventType());
    } else {
      isRelation = slot.getType().is('baja:Relation');
      value = comp.get(slot);
      if (isRelation) {
        var relationId = toRelationId(value);
        display = relationId;
        id = 'relation:' + relationId;
        type = 'relation';
        tooltip = relationId;
      } else {
        isProp = true;
        type = 'property';
        tooltip = "".concat(slot, ": ").concat(slot.getType());
      }
    }
    return Promise.resolve(isProp ? value.toString(mergedCx) : undefined).then(function (valueString) {
      var bar = {
        id: id || 'slot:' + slot.getName(),
        display: display || comp.getDisplayName(slot),
        tooltip: tooltip,
        type: type
      };
      var status = isProp && baja.Status.getStatusFromIStatus(value);
      var foreground = getForeground(status);
      var background = getBackground(status);
      if (valueString !== undefined) {
        bar.value = valueString;
      }
      if (foreground) {
        bar.foreground = foreground;
      }
      if (background) {
        bar.background = background;
      }
      return bar;
    });
  }

  /**
   * @param {string} relationId
   * @returns {Object}
   */
  function relationIdToBar(relationId) {
    return {
      id: 'relation:' + relationId,
      display: relationId,
      tooltip: relationId,
      type: 'relation'
    };
  }

  /**
   * Convert a `baja:WsAnnotation` to JSON object with layout info.
   * @param {baja.Simple} a a `baja:WsAnnotation`
   * @returns {{x: Number, y: Number, width: Number, height: Number}}
   */
  function annotationToLayout(a) {
    return a && {
      x: a.getX(),
      y: a.getY(),
      width: a.getWidth(),
      height: a.getHeight()
    };
  }

  /**
   * @param {baja.Struct} link
   * @returns {boolean} true if this link has the `COMPOSITE` flag on its slot
   * in its parent
   */
  function isComposite(link) {
    if (!link) {
      return false;
    }
    var flags = link.getParent().getFlags(link.getPropertyInParent());
    return !!(flags & baja.Flags.COMPOSITE);
  }

  /**
   * @param {baja.Struct} link
   * @returns {boolean} true if this link has the `HIDDEN` flag on its slot
   * in its parent
   */
  function isHidden(link) {
    if (!link) {
      return false;
    }
    var flags = link.getParent().getFlags(link.getPropertyInParent());
    return !!(flags & baja.Flags.HIDDEN);
  }

  /**
   * @param {baja.Struct} [link]
   * @returns {boolean} true if a LonLink
   */
  function isLon(link) {
    return link && link.getType().getTypeName() === 'LonLink';
  }
  function getKnobTooltip(knob, link) {
    var sourcePath = toDisplayPathString(knob.getSourceComponent());
    var sourceSlot = knob.getSourceSlotName();
    var targetPath = link ? toDisplayPathString(link.getParent()) : knob.getTargetOrd();
    var targetSlot = knob.getTargetSlotName();
    return bajaLex.get({
      key: 'linkcheck.linkDirection',
      args: ["".concat(sourcePath, ".").concat(sourceSlot), "".concat(targetPath, ".").concat(targetSlot)]
    });
  }

  /**
   * @param {baja.Complex} link
   * @param {baja.Component} [sourceComp]
   * @param {String|baja.Slot} sourceSlot
   * @param {baja.Component} targetComp
   * @param {String|baja.Slot} targetSlot
   * @returns {string} a `source.slot -> target.slot` string suitable for a
   * Link's tooltip
   */
  function getLinkTooltip(link, sourceComp, sourceSlot, targetComp, targetSlot) {
    var sourcePath = sourceComp ? sourceComp.getDisplayName() : link.getSourceOrd(),
      targetPath = targetComp.getDisplayName();
    return bajaLex.get({
      key: 'linkcheck.linkDirection',
      args: ["".concat(sourcePath, ".").concat(sourceSlot), "".concat(targetPath, ".").concat(targetSlot)]
    });
  }

  /**
   * @param {baja.Component} [sourceComp]
   * @param {object} relationKnob
   * @param {baja.Component} targetComp
   * @returns {string} a `source-<relationId>-target` string suitable for a
   * relation knob's tooltip
   */
  function getRelationKnobTooltip(sourceComp, relationKnob, targetComp) {
    var sourcePath = sourceComp ? sourceComp.toPathString() : relationKnob.getRelationOrd();
    var relationId = relationKnob.getRelationId();
    var targetPath = targetComp.toPathString();
    return getRelationDisplay(sourcePath, relationId, targetPath);
  }
  function getRelationTooltip(sourceComp, relation, targetComp) {
    var sourcePath = sourceComp.getDisplayName();
    var relationId = relation.getRelationId();
    var targetPath = targetComp ? targetComp.getDisplayName() : relation.get('sourceOrd');
    return getRelationDisplay(sourcePath, relationId, targetPath);
  }
  function getRelationDisplay(source, relationId, target) {
    return bajaLex.get({
      key: 'relationcheck.relationDirection',
      args: [source, relationId, target]
    });
  }
  function getRelationKnobIds(comp) {
    var relations = comp.getSlots().is('baja:Relation').toValueArray().filter(isRelation);
    var relationIds = uniq(relations.map(toRelationId));
    var relationKnobs = comp.getRelationKnobs();
    var relationKnobIds = uniq(relationKnobs.map(toRelationId));
    // relation ids that are already present in baja:Relations will be shown as
    // regular slots. we're only adding those relation ids that appear in knob
    // form only.
    return difference(relationKnobIds, relationIds);
  }
  function toRelationId(relation) {
    return relation.getRelationId();
  }
  return GlyphFactory;
});
