function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * API Status: **Private**
 * @module nmodule/wiresheet/rc/wb/baja/EntityFactory
 */
define(['baja!', 'log!nmodule.wiresheet.rc.wb.baja.EntityFactory', 'niagaraSystemProperties', 'Promise', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/wiresheet/rc/wb/baja/bajaUtils'], function (baja, log, niagaraSystemProperties, Promise, typeUtils, bajaUtils) {
  'use strict';

  var logFine = log.fine.bind(log);
  var LEASE_KNOBS_PROP = 'niagara.wiresheet.leaseForOutgoingStubColors';
  var LEASE_KNOBS = niagaraSystemProperties[LEASE_KNOBS_PROP] === 'true';
  var componentToId = bajaUtils.componentToId,
    getKnob = bajaUtils.getKnob,
    getLink = bajaUtils.getLink,
    getRelation = bajaUtils.getRelation,
    getRelationKnob = bajaUtils.getRelationKnob,
    isComponent = bajaUtils.isComponent,
    isLink = bajaUtils.isLink,
    isRelation = bajaUtils.isRelation,
    isTextBlock = bajaUtils.isTextBlock,
    isVector = bajaUtils.isVector,
    ordToComponentId = bajaUtils.ordToComponentId;
  var isKnob = typeUtils.isKnob,
    isRelationKnob = typeUtils.isRelationKnob;

  /**
   * This factory's job is to convert Baja values into entity objects suitable
   * for storing in a `ViewModel`.
   *
   * For each Baja value, it will decide on a suitable ID that will refer to the
   * `ViewModel` entity that that value maps to. For instance, a Knob and a Link
   * will map to the same view entity, providing that the source and target
   * components are both visible on the same wiresheet. Passing either the Knob
   * or the Link into this factory will resolve you the same ID.
   *
   * @class
   * @alias module:nmodule/wiresheet/rc/wb/baja/EntityFactory
   * @param {Object} [params]
   * @param {baja.Component} params.container wiresheet container component
   * @param {Function} params.containsId a function to decide whether a string ID
   * is already represented in the wiresheet. It will typically be up to the
   * ViewModel to decide - but this is passed in as a callback to avoid a
   * ViewModel/EntityFactory circular dependency. It should return a boolean or
   * a Promise that resolves to one.
   * @param {module:nmodule/wiresheet/rc/wb/baja/GlyphFactory} params.glyphFactory factory
   * for generating JSON glyph data
   */
  var EntityFactory = function EntityFactory() {
    var _ref = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      container = _ref.container,
      containsId = _ref.containsId,
      glyphFactory = _ref.glyphFactory;
    if (!container) {
      throw new Error('container required');
    }
    if (!containsId) {
      throw new Error('containsId required');
    }
    if (!glyphFactory) {
      throw new Error('glyphFactory required');
    }
    this.$container = container;
    this.$containsId = function (id) {
      return Promise.resolve(containsId(id));
    };
    this.$glyphFactory = glyphFactory;
  };

  /**
   * Given a Baja value, derive a suitable ID for finding its corresponding
   * entity in a `ViewModel`.
   *
   * @param {baja.Value|Object|string} value a Baja value (including knobs). A
   * string will just be resolved directly.
   * @param {baja.Component} [parent] if a Link or Relation has been removed
   * from its parent, you can still retrieve the correct ID if you still have
   * a reference to the parent and removed Slot. Otherwise, `getParent()` will
   * be used in deriving the ID.
   * @param {baja.Property} [prop] see `parent` param.
   * @returns {Promise.<String>}
   */
  EntityFactory.prototype.toId = function (value, parent, prop) {
    var containsId = this.$containsId;
    var id = value;
    if (isComponent(value)) {
      id = componentToId(value);
    } else if (isLink(value)) {
      id = linkToId(containsId, value, parent, prop);
    } else if (isKnob(value)) {
      id = knobToId(containsId, value);
    } else if (isRelation(value)) {
      id = relationToId(containsId, value, parent, prop);
    } else if (isRelationKnob(value)) {
      id = relationKnobToId(containsId, value);
    }
    return Promise.resolve(id);
  };

  /**
   * Given a Baja value, generate entity data suitable for inserting into a
   * `ViewModel`.
   *
   * @param {baja.Value|Object} object a Component, Link, Relation, or Knob
   * @returns {Promise.<{ id: string, glyph: Object}>} promise to be resolved
   * with entity data.
   */
  EntityFactory.prototype.toEntity = function (object) {
    var containsId = this.$containsId;
    var factory = this.$glyphFactory;
    var glyph;
    if (isComponent(object)) {
      glyph = toComponentGlyph(this.$container, factory, object);
    } else if (isLink(object)) {
      glyph = toLinkGlyph(containsId, factory, object);
    } else if (isKnob(object)) {
      glyph = toKnobGlyph(containsId, factory, object);
    } else if (isRelation(object)) {
      glyph = toRelationGlyph(containsId, factory, object);
    } else if (isRelationKnob(object)) {
      glyph = toRelationKnobGlyph(containsId, factory, object);
    } else {
      return Promise.reject(new Error('unsupported type'));
    }
    return Promise.all([this.toId(object), glyph]).then(function (_ref2) {
      var _ref3 = _slicedToArray(_ref2, 2),
        id = _ref3[0],
        glyph = _ref3[1];
      return {
        id: id,
        glyph: glyph
      };
    });
  };

  /**
   * @private
   * @returns {boolean} true if knobs should lease their targets for coloring
   * information
   */
  EntityFactory.$leaseKnobs = function () {
    return LEASE_KNOBS;
  };

  ////////////////////////////////////////////////////////////////
  // Derive IDs for Baja objects
  ////////////////////////////////////////////////////////////////

  /** simple, synchronous text id functions */
  var textId = {
    forComponent: function forComponent(comp) {
      return componentToId(comp);
    },
    forLink: function forLink(targetId, targetProp) {
      return "link:".concat(targetId, ".").concat(targetProp);
    },
    // TODO: if the target slot has fan-in, can you have dup links with the exact same source and target slots?
    forKnob: function forKnob(sourceId, sourceSlot, targetId, targetSlot) {
      return "knob:".concat(sourceId, ".").concat(sourceSlot, "->").concat(targetId, ".").concat(targetSlot);
    },
    forRelation: function forRelation(sourceId, sourceProp) {
      return "relation:".concat(sourceId, ".").concat(sourceProp);
    },
    forRelationKnob: function forRelationKnob(sourceId, relationId, targetId) {
      return "relationKnob:".concat(sourceId, "-<").concat(relationId, ">-").concat(targetId);
    }
  };
  function linkToId(containsId, link, parent, prop) {
    var sourceOrd = link.get('sourceOrd');
    var targetComp = parent || link.getParent();
    return containsId(componentToId(targetComp)).then(function (targetOnThisWiresheet) {
      if (targetOnThisWiresheet) {
        var linkSlot = prop || link.getPropertyInParent();
        return textId.forLink(componentToId(targetComp), linkSlot);
      } else {
        var sourceSlot = link.getSourceSlotName();
        var targetId = componentToId(targetComp);
        var targetSlot = link.getTargetSlotName();
        return ordToComponentId(sourceOrd, {
          base: targetComp
        }).then(function (sourceId) {
          return textId.forKnob(sourceId, sourceSlot, targetId, targetSlot);
        });
      }
    });
  }
  function knobToId(containsId, knob) {
    var sourceComp = knob.getSourceComponent();
    var targetOrd = knob.getTargetOrd();
    var sourceId = componentToId(sourceComp);
    var sourceSlot = knob.getSourceSlotName();
    var targetSlot = knob.getTargetSlotName();
    var base = sourceComp;
    return ordToComponentId(targetOrd, {
      base: sourceComp
    }).then(function (targetId) {
      return containsId(targetId).then(function (targetIsOnThisWiresheet) {
        if (targetIsOnThisWiresheet) {
          return targetOrd.get({
            base: base,
            lease: true
          }).then(function (targetComp) {
            return getLink(targetComp, knob, {
              base: base
            });
          }).then(function (link) {
            if (!link) {
              logFine('No corresponding link found for knob');
              return textId.forKnob(sourceId, sourceSlot, targetId, targetSlot);
            }
            return textId.forLink(targetId, link.getPropertyInParent());
          });
        } else {
          return textId.forKnob(sourceId, sourceSlot, targetId, targetSlot);
        }
      });
    });
  }
  function relationToId(containsId, relation, parent, prop) {
    var sourceComp = parent || relation.getParent();
    var sourceId = componentToId(sourceComp);
    return containsId(componentToId(sourceComp)).then(function (sourceOnThisWiresheet) {
      if (sourceOnThisWiresheet) {
        var relationProp = prop || relation.getPropertyInParent();
        return textId.forRelation(sourceId, relationProp);
      } else {
        var targetOrd = relation.getSourceOrd(); //not a typo
        var relationId = relation.getRelationId();
        return ordToComponentId(targetOrd, {
          base: sourceComp
        }).then(function (targetId) {
          return textId.forRelationKnob(sourceId, relationId, targetId);
        });
      }
    });
  }
  function relationKnobToId(containsId, relationKnob) {
    var sourceOrd = relationKnob.getRelationOrd();
    var targetComp = relationKnob.getEndpointComponent();
    var relationId = relationKnob.getRelationId();
    var targetId = componentToId(targetComp);
    var base = targetComp;
    return ordToComponentId(sourceOrd, {
      base: base
    }).then(function (sourceId) {
      return containsId(sourceId).then(function (sourceIsOnThisWiresheet) {
        if (sourceIsOnThisWiresheet) {
          return sourceOrd.get({
            base: targetComp,
            lease: true
          }).then(function (sourceComp) {
            return getRelation(sourceComp, relationKnob, {
              base: base
            });
          }).then(function (relation) {
            if (!relation) {
              logFine('No corresponding relation found for relation knob');
              return textId.forRelationKnob(sourceId, relationId, targetId);
            }
            // TODO is there a relation equivalent to leaseForKnobColors?
            var relationProp = relation.getPropertyInParent();
            return textId.forRelation(sourceId, relationProp);
          });
        } else {
          return textId.forRelationKnob(sourceId, relationId, targetId);
        }
      });
    });
  }

  ////////////////////////////////////////////////////////////////
  // Create Glyphs for Baja objects
  ////////////////////////////////////////////////////////////////

  /**
   * @param {baja.Component} container
   * @param {module:nmodule/wiresheet/rc/wb/baja/EntityFactory} factory
   * @param {baja.Component} component
   * @returns {Promise.<module:nmodule/wiresheet/rc/typedefs~Glyph>}
   */
  function toComponentGlyph(container, factory, component) {
    if (belongsOnThisWiresheet(container, component)) {
      return isTextBlock(component) ? factory.textBlockGlyph(component) : factory.componentGlyph(component);
    } else {
      return Promise.resolve({
        type: 'PlaceholderGlyph',
        uiStatus: {}
      });
    }
  }
  function belongsOnThisWiresheet(container, component) {
    // noinspection JSBitwiseOperatorUsage
    return component && component.getParent() === container && !isVector(component) && !(container.getFlags(component.getPropertyInParent()) & baja.Flags.HIDDEN);
  }

  /**
   * @param {function} containsId
   * @param {module:nmodule/wiresheet/rc/wb/baja/EntityFactory} factory
   * @param {baja.Struct} link
   * @returns {Promise.<module:nmodule/wiresheet/rc/typedefs~Glyph>}
   */
  function toLinkGlyph(containsId, factory, link) {
    var sourceOrd = link.getSourceOrd();
    var targetComp = link.getParent();
    return ordToComponentId(sourceOrd, {
      base: targetComp
    }).then(function (sourceId) {
      return Promise.all([containsId(sourceId), containsId(componentToId(targetComp))]);
    }).then(function (_ref4) {
      var _ref5 = _slicedToArray(_ref4, 2),
        sourceOnThisWiresheet = _ref5[0],
        targetOnThisWiresheet = _ref5[1];
      return Promise.resolve(sourceOnThisWiresheet && sourceOrd.get({
        base: targetComp
      })).then(function (sourceComp) {
        var sourceLinkable = sourceOnThisWiresheet && !isTextBlock(sourceComp);
        var targetLinkable = targetOnThisWiresheet && !isTextBlock(targetComp);
        if (sourceLinkable && targetLinkable) {
          return factory.linkSnakeGlyph(link, sourceComp);
        } else if (sourceLinkable) {
          var knob = getKnob(sourceComp, link);
          return factory.linkKnobGlyph(knob, targetComp);
        } else if (targetLinkable) {
          return factory.linkStubGlyph(link);
        }
      });
    });
  }

  /**
   * @param {function} containsId
   * @param {module:nmodule/wiresheet/rc/wb/baja/EntityFactory} factory
   * @param {object} knob
   * @returns {Promise.<module:nmodule/wiresheet/rc/typedefs~Glyph>}
   */
  function toKnobGlyph(containsId, factory, knob) {
    var sourceComp = knob.getSourceComponent();
    var sourceId = componentToId(sourceComp);
    var targetOrd = knob.getTargetOrd();
    var base = sourceComp;
    return Promise.all([containsId(sourceId), ordToComponentId(targetOrd, {
      base: sourceComp
    }).then(containsId)]).then(function (_ref6) {
      var _ref7 = _slicedToArray(_ref6, 2),
        sourceOnThisWiresheet = _ref7[0],
        targetOnThisWiresheet = _ref7[1];
      var leaseTarget = targetOnThisWiresheet || EntityFactory.$leaseKnobs();
      return Promise.resolve(leaseTarget && targetOrd.get({
        base: base,
        lease: true
      })).then(function (targetComp) {
        var sourceLinkable = sourceOnThisWiresheet && !isTextBlock(sourceComp);
        var targetLinkable = targetOnThisWiresheet && !isTextBlock(targetComp);
        if (targetLinkable) {
          return getLink(targetComp, knob, {
            base: base
          }).then(function (link) {
            if (link && sourceLinkable) {
              return factory.linkSnakeGlyph(link, sourceComp);
            } else if (link) {
              return factory.linkStubGlyph(link);
            } else if (sourceLinkable) {
              return factory.linkKnobGlyph(knob, targetComp);
            }
          });
        } else if (sourceLinkable) {
          return factory.linkKnobGlyph(knob, targetComp);
        }
      });
    });
  }

  /**
   * @param {function} containsId
   * @param {module:nmodule/wiresheet/rc/wb/baja/EntityFactory} factory
   * @param {baja.Struct} relation
   * @returns {Promise.<module:nmodule/wiresheet/rc/typedefs~Glyph>}
   */
  function toRelationGlyph(containsId, factory, relation) {
    var sourceComp = relation.getParent();
    var sourceId = componentToId(sourceComp);
    var targetOrd = relation.getSourceOrd();
    return Promise.all([containsId(sourceId), ordToComponentId(targetOrd, {
      base: sourceComp
    }).then(containsId)]).then(function (_ref8) {
      var _ref9 = _slicedToArray(_ref8, 2),
        sourceOnThisWiresheet = _ref9[0],
        targetOnThisWiresheet = _ref9[1];
      return Promise.resolve(targetOnThisWiresheet && targetOrd.get({
        base: sourceComp
      })).then(function (targetComp) {
        var sourceLinkable = sourceOnThisWiresheet && !isTextBlock(sourceComp);
        var targetLinkable = targetOnThisWiresheet && !isTextBlock(targetComp);
        if (sourceLinkable && targetLinkable) {
          return factory.relationSnakeGlyph(relation, targetComp);
        } else if (sourceLinkable) {
          return factory.relationStubGlyph(relation);
        } else if (targetLinkable) {
          var relationKnob = getRelationKnob(targetComp, relation);
          return factory.relationKnobGlyph(relationKnob);
        }
      });
    });
  }

  /**
   * @param {function} containsId
   * @param {module:nmodule/wiresheet/rc/wb/baja/EntityFactory} factory
   * @param {object} relationKnob
   * @returns {Promise.<module:nmodule/wiresheet/rc/typedefs~Glyph>}
   */
  function toRelationKnobGlyph(containsId, factory, relationKnob) {
    var targetComp = relationKnob.getEndpointComponent();
    var targetId = componentToId(targetComp);
    var sourceOrd = relationKnob.getRelationOrd();
    var base = targetComp;
    return Promise.all([ordToComponentId(sourceOrd, {
      base: targetComp
    }).then(containsId), containsId(targetId)]).then(function (_ref10) {
      var _ref11 = _slicedToArray(_ref10, 2),
        sourceOnThisWiresheet = _ref11[0],
        targetOnThisWiresheet = _ref11[1];
      return Promise.resolve(sourceOnThisWiresheet && sourceOrd.get({
        base: base
      })).then(function (sourceComp) {
        var sourceLinkable = sourceOnThisWiresheet && !isTextBlock(sourceComp);
        var targetLinkable = targetOnThisWiresheet && !isTextBlock(targetComp);
        if (sourceLinkable) {
          return getRelation(sourceComp, relationKnob, {
            base: base
          }).then(function (relation) {
            if (!relation) {
              logFine('No corresponding relation found for relation knob');
              return factory.relationKnobGlyph(relationKnob);
            }
            if (targetLinkable) {
              return factory.relationSnakeGlyph(relation, targetComp);
            } else {
              return factory.relationStubGlyph(relation);
            }
          });
        } else {
          return factory.relationKnobGlyph(relationKnob);
        }
      });
    });
  }
  return EntityFactory;
});
