/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/wiresheet/rc/core/controller/Selection
 */
define(['Promise', 'underscore', 'nmodule/js/rc/tinyevents/tinyevents'], function (Promise, _, tinyevents) {
  'use strict';

  /**
   * Object representing which set of entities in a view model are currently
   * selected by the user. It can hold both vertices and edge triples.
   *
   * @constructor
   * @param {module:nmodule/wiresheet/rc/core/ViewModel} viewModel
   */
  var Selection = function Selection(viewModel) {
    this.$viewModel = viewModel;
    this.$selection = {};
    tinyevents(this);
  };
  Selection.prototype.getSelectedEntities = function () {
    return _.values(this.$selection);
  };
  Selection.prototype.getSelectedEdges = function () {
    return this.getSelectedEntities().filter(isEdge);
  };
  Selection.prototype.getSelectedVertices = function () {
    return this.getSelectedEntities().filter(isVertex);
  };
  Selection.prototype.setSelectedEntities = function (entities) {
    var vm = this.$viewModel,
      selection = this.$selection,
      newSelection = {};
    entities.forEach(function (e) {
      newSelection[requireId(vm, e)] = e;
    });
    var added = _.filter(newSelection, function (e, key) {
        return !selection[key];
      }),
      removed = _.filter(selection, function (e, key) {
        return !newSelection[key];
      });
    this.$selection = newSelection;
    return Promise.resolve((added.length || removed.length) && this.emitAndWait('changed', {
      added: added,
      removed: removed
    }));
  };
  Selection.prototype.add = function (entity) {
    if (!this.contains(entity)) {
      var selection = this.$selection,
        id = requireId(this.$viewModel, entity);
      selection[id] = entity;
      return this.emitAndWait('changed', {
        added: [entity],
        removed: []
      });
    } else {
      return Promise.resolve();
    }
  };
  Selection.prototype.remove = function (entity) {
    if (this.contains(entity)) {
      var selection = this.$selection,
        id = getId(this.$viewModel, entity),
        existing = selection[id];
      delete selection[id];
      return this.emitAndWait('changed', {
        added: [],
        removed: [existing]
      });
    } else {
      return Promise.resolve();
    }
  };
  Selection.prototype.contains = function (entity) {
    var id = getId(this.$viewModel, entity);
    return !!(id && this.$selection[id]);
  };
  function getId(viewModel, e) {
    return viewModel.getId(e.predicate || e);
  }
  function requireId(viewModel, e) {
    var id = getId(viewModel, e);
    if (!id) {
      throw new Error('Entity does not exist in view model');
    }
    return id;
  }
  function isEdge(entity) {
    return entity.predicate;
  }
  function isVertex(entity) {
    return !isEdge(entity);
  }
  return Selection;
});
