/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * A module responsible for running maintenance operations on the
 * histories selected by the 'manager' module
 *
 * API Status: **Private**
 * @module nmodule/history/rc/maintenance
 */
define(['baja!', 'underscore', 'Promise', 'dialogs'], function (baja, _, Promise, dialogs) {
  "use strict";

  var spacePromise,
    HISTORY_SCHEME = 'history:',
    BATCH_SIZE = 5;

  /**
   *  Resolve and returns the history space
   * @returns {Promise} Promise with resolved space
   */
  function getSpace() {
    return spacePromise || (spacePromise = baja.Ord.make(HISTORY_SCHEME).get());
  }
  function spaceUnresolved(err) {
    dialogs.showOk(err);
  }

  /**
   * Clears records from the history database prior to a given baja.AbsTime
   * @param ordList Array of type baja.Ord
   * @param beforeTime Time (baja.AbsTime) prior to which records should be cleared
   * @returns {Promise} Promise
   */
  function clearOldRecords(ordList, beforeTime) {
    function spaceResolved(space) {
      return space.clearOldRecords(ordList, beforeTime);
    }
    return getSpace().then(spaceResolved, spaceUnresolved);
  }

  /**
   * Clears all records from the history database for the given list of ords
   * @param ordList Array of type baja.Ord
   * @returns {Promise} Promise
   */
  function clearAllRecords(ordList) {
    function spaceResolved(space) {
      return space.clearAllRecords(ordList);
    }
    return getSpace().then(spaceResolved, spaceUnresolved);
  }

  /**
   * Deletes histories from the history database for the given list of ords
   * @param ordList Array of type baja.Ord
   * @returns {Promise} Promise
   */
  function deleteHistories(ordList) {
    function spaceResolved(space) {
      //split the ord list into batches of size BATCH_SIZE
      var batches = [],
        i = 0;
      while (i < ordList.length) {
        if (i + BATCH_SIZE >= ordList.length) {
          batches.push(ordList.slice(i));
        } else {
          batches.push(ordList.slice(i, i + BATCH_SIZE));
        }
        i += BATCH_SIZE;
      }
      return Promise.all(_.map(batches, function (batch) {
        return space.deleteHistories(batch);
      }));
    }
    return getSpace().then(spaceResolved, spaceUnresolved);
  }

  //Public API
  return {
    getSpace: getSpace,
    clearOld: clearOldRecords,
    clearAll: clearAllRecords,
    deleteAll: deleteHistories
  };
});
