/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/fe/HistoryTablePaginationModel
 */
define(['baja!', 'Promise', 'nmodule/history/rc/historyUtil', 'nmodule/history/rc/fe/HistoryTableQueryAdapter', 'nmodule/history/rc/servlet/queryServletUtil', 'nmodule/js/rc/switchboard/switchboard', 'nmodule/webEditors/rc/servlets/QueryAdapter', 'nmodule/webEditors/rc/wb/table/model/TableModel', 'nmodule/webEditors/rc/wb/table/pagination/PaginationModel'], function (baja, Promise, historyUtil, HistoryTableQueryAdapter, queryServletUtil, switchboard, QueryAdapter, TableModel, PaginationModel) {
  'use strict';

  /**
   * @class
   * @alias module:nmodule/history/rc/fe/HistoryTablePaginationModel
   * @extends module:nmodule/webEditors/rc/wb/table/pagination/PaginationModel
   * @param {object} params
   * @param {baja.Subscriber} params.subscriber
   */
  var HistoryTablePaginationModel = function HistoryTablePaginationModel(params) {
    PaginationModel.apply(this, arguments);
    this.$subscriber = params.subscriber;
    this.$liveConfigured = false;
    this.$live = false;
    this.$delta = false;
    this.$subscribeContinuous = false;
    switchboard(this, {
      $setLive: {
        allow: 'oneAtATime',
        onRepeat: 'queue'
      },
      $resolveLiveQuery: {
        allow: 'oneAtATime',
        onRepeat: 'queue'
      },
      $efficientAdd: {
        allow: 'oneAtATime',
        onRepeat: 'queue'
      }
    });
  };
  HistoryTablePaginationModel.prototype = Object.create(PaginationModel.prototype);
  HistoryTablePaginationModel.prototype.constructor = HistoryTablePaginationModel;

  /**
   * Separate initialization, which will (re)-initialize the pagination model
   * without firing any events.
   *
   * @param {object} [params]
   * @param {number} [params.rowsPerPage=100] rows per page
   * @param {number} [params.rowCount=-1] row count. -1 indicates unlimited, or
   * unknown, number of total rows.
   * @param {number} [params.currentPage=1] current page
   * @param {boolean} [params.live=false]
   * @param {string} [params.sortColumn] column to sort by
   * @param {string} [params.sortDirection] `asc` or `desc`
   */
  HistoryTablePaginationModel.prototype.initialize = function (params) {
    PaginationModel.prototype.initialize.apply(this, arguments);
    this.$live = params.live === true;
    this.$sortColumn = params.sortColumn;
    this.$sortDescending = params.sortDirection === 'desc';
  };

  /**
   * Given a range of row numbers, the implementer should resolve a TableModel
   * containing the correct rows.
   *
   * This method is called by Table when Table#load() is called.
   *
   * @param {number} rowStart (inclusive)
   * @param {number} rowEnd (exclusive)
   * @param {object} [context] optional context
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/model/TableModel>}
   */

  HistoryTablePaginationModel.prototype.makeTableModel = function (rowStart, rowEnd, context) {
    var that = this;
    var model = new TableModel({
      columns: []
    });
    that.$model = model;
    var rowCount = rowEnd - rowStart;
    var queryAdapter = that.$makeQueryAdapter(model, context);
    queryAdapter.on(QueryAdapter.CHANGE_PAGE_EVENT, function (page) {
      if (page !== that.getCurrentPage()) {
        return that.setCurrentPage(page).then(function () {
          return that.setRowCount(page * rowCount);
        });
      }
    });
    var sortColumn = this.$sortColumn;
    var sortDescending = this.$sortDescending;
    return queryAdapter.doQuery({
      ord: this.$ord,
      filter: this.$filter,
      offset: rowStart,
      limit: rowCount,
      sortColumn: sortColumn,
      descending: sortDescending,
      timeRange: this.$timeRange,
      delta: this.$delta
    }).then(function () {
      that.setRowCount(PaginationModel.MAX_ROWS);
      if (sortColumn) {
        model.$setSortDirection(sortColumn, sortDescending ? 'desc' : 'asc');
      }
      return model;
    });
  };

  /**
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/table/model/TableModel}
   */
  HistoryTablePaginationModel.prototype.$getTableModel = function () {
    return this.$model;
  };

  /**
   * Attempt to change the page, if the number is positive and an error is found,
   * then the last page has already been found and someone is trying to go beyond the known limit
   * so instead of throwing an error switch the page back to the current page.
   * @param {Number}  currentPage
   * @returns {Promise}
   */
  HistoryTablePaginationModel.prototype.setCurrentPage = function (currentPage) {
    var that = this,
      args = arguments;
    return Promise["try"](function () {
      return PaginationModel.prototype.setCurrentPage.apply(that, args);
    })["catch"](function (error) {
      if (currentPage > 0) {
        //fallback to latestPage page
        var latestPage = that.getCurrentPage();
        return PaginationModel.prototype.setCurrentPage.call(that, latestPage).then(function () {
          return that.setRowCount(that.getRowsPerPage() * latestPage);
        });
      } else {
        throw error;
      }
    });
  };

  /**
   * Set the query parameters used by `HistoryTableQueryAdapter` and emit
   * a configuration change.
   *
   * @param {Object} params
   * @param {String} params.ordBody
   * @param {baja.Component} [params.config] a `history:HistoryConfig`
   * @param {baja.Component} [params.filter] `bql:FilterSet` component
   * @param {boolean} [params.delta]
   * @param {baja.Simple} [params.timeRange] `bql:DynamicTimeRange`
   * @param {boolean} [params.resetPage] true if the current page should
   * be reset to 1
   *
   * @returns {Promise}
   */
  HistoryTablePaginationModel.prototype.setQueryParams = function (params) {
    this.$ord = baja.Ord.make('history:' + params.ordBody);
    this.$filter = params.filter;
    this.$config = params.config;
    this.$delta = params.delta;
    this.$timeRange = params.timeRange;

    // If resetPage is true, set the current page to be the first page.
    if (params.resetPage) {
      // This is being done directly instead of calling this.setCurrentPage()
      //  because we don't want more than one event to fire.      
      //  The query is re-executed each time a changed:config or 
      //  changed:currentPage event fires.
      this.$currentPage = 1;
    }
    return this.emitAndWait('changed', 'config');
  };

  /**
   * Sort the model and refresh the page.
   * The current implementation refreshes the page by setting the current
   * page to 1.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Column} column
   * @param {boolean} desc
   * @returns {Promise}
   */
  HistoryTablePaginationModel.prototype.sort = function (column, desc) {
    this.$sortColumn = column && column.getName ? column.getName() : undefined;
    this.$sortDescending = desc;

    // This sets the page to 1, which has a side effect of calling `makeTableModel`,
    // which re-executes queryAdapter.doQuery().
    // If this changes, consider firing a 'config' event instead.
    return this.setCurrentPage(1);
  };

  /**
   * Attempt to add a record efficiently to the table. Return false if the configuration doesn't allow the efficient
   * add to work.
   *
   * @private
   * @param {baja.Component} record `history:HistoryRecord`
   * @returns {Promise} resolves to a truthy value to confirm the add has been taken care of because it was efficient
   */
  HistoryTablePaginationModel.prototype.$efficientAdd = function (record) {
    var that = this;
    if (that.$delta || that.$filter || !historyUtil.isLive(that.timeRange)) {
      return Promise.resolve(false);
    }
    var sortColumn = that.$sortColumn;
    var model = that.$getTableModel();
    if (!sortColumn || sortColumn === "timestamp") {
      if (this.$sortDescending) {
        var lastRow = model.getRows()[model.getRows().length - 1];
        return model.insertRows([record], 0).then(function () {
          if (that.getRowsPerPage() <= model.getRows().length) {
            return model.removeRows([lastRow]);
          }
        }).then(function () {
          return true;
        });
      } else {
        if (that.getRowsPerPage() <= model.getRows().length) {
          //record is too old for the page
          return Promise.resolve(true);
        }
        return model.insertRows([record]).then(function () {
          return true;
        });
      }
    }
    return Promise.resolve(false);
  };

  /**
   * @private
   * @param {module:nmodule/webEditors/rc/wb/table/model/TableModel} model
   * @param {object} context
   * @returns {module:nmodule/history/rc/fe/HistoryTableQueryAdapter}
   */
  HistoryTablePaginationModel.prototype.$makeQueryAdapter = function (model, context) {
    this.$queryAdapter = new HistoryTableQueryAdapter(model, this.$config, context);
    return this.$queryAdapter;
  };

  /**
   * @private
   * @param {boolean} live
   * @returns {Promise}
   */
  HistoryTablePaginationModel.prototype.$setLive = function (live) {
    var that = this;
    if (live === that.$liveConfigured || !that.$ord) {
      return Promise.resolve();
    }
    this.$liveConfigured = live;
    that.$live = live;
    if (live) {
      that.$changed = that.$changed || function (prop, cx) {
        if (prop.getName() !== "lastRecord" && prop.getName() !== "lastSuccess") {
          return;
        }
        var lastRecord = this.get('lastRecord');
        //non-niagara History Import
        if (lastRecord) {
          if (lastRecord.getTimestamp().getJsDate().getFullYear() < 1971) {
            return;
          }
          return that.$subscribeContinuous && that.$efficientAdd(lastRecord).then(function (wasAdded) {
            if (wasAdded) {
              that.setRowCount(PaginationModel.MAX_ROWS)["catch"](baja.error);
            } else {
              that.$resolveLiveQuery()["catch"](baja.error);
            }
          });
        }
        that.$resolveLiveQuery()["catch"](baja.error);
      };
      that.$subscriber.attach("changed", that.$changed);
    }
    return queryServletUtil.resolveHistorySourceOrd(that.$ord, live).then(function (ord) {
      if (ord) {
        return ord.get({
          subscriber: that.$subscriber
        });
      }
    }).then(function (value) {
      if (live) {
        return that.$resolveLiveQuery();
      } else {
        that.$subscribeContinuous = false;
        if (value !== undefined) {
          return that.$subscriber.unsubscribe(value);
        }
      }
    });
  };

  /**
   * Get the live flag
   *
   * @returns {Boolean} live flag
   */
  HistoryTablePaginationModel.prototype.getLive = function () {
    return this.$live;
  };

  /**
   * @private
   * @returns {Promise}
   */
  HistoryTablePaginationModel.prototype.$resolveLiveQuery = function () {
    var that = this;
    var queryAdapter = that.$queryAdapter;
    var tableModel = new TableModel({
      columns: queryAdapter.getColumns()
    });
    var liveQueryAdapter = new HistoryTableQueryAdapter(tableModel, that.$config);
    var lastTimestamp;
    if (that.$lastTimestamp) {
      lastTimestamp = baja.AbsTime.make({
        jsDate: new Date(that.$lastTimestamp.getJsDate().getTime() + 1)
      });
    }
    var oldestTimestamp = that.$getOldestTimestamp(lastTimestamp);
    return liveQueryAdapter.doQuery({
      ord: that.$ord,
      filter: that.$filter,
      offset: 0,
      limit: that.getRowsPerPage(),
      sortColumn: that.$sortColumn,
      descending: that.$sortDescending,
      lastTimestamp: oldestTimestamp,
      timeRange: that.$timeRange,
      delta: that.$delta
    }).then(function () {
      that.$subscribeContinuous = true;
      return that.$updateFromLive(liveQueryAdapter.$model);
    });
  };

  /**
   * Get the oldest Timestamp from the TableModel
   *
   * @private
   * @param {baja.AbsTime} [min]
   * @return {baja.AbsTime|undefined}
   */
  HistoryTablePaginationModel.prototype.$getOldestTimestamp = function (min) {
    var rows = this.$getTableModel().getRows();
    for (var i = 0; i < rows.length; i++) {
      var timestamp = rows[i].getSubject().get('timestamp');
      if (!min) {
        min = timestamp;
      } else {
        min = timestamp.valueOf() < min.valueOf() ? timestamp : min;
      }
    }
    return min;
  };

  /**
   * @private
   * @param {module:nmodule/webEditors/rc/wb/table/model/TableModel} liveModel
   * @return {Promise}
   */

  HistoryTablePaginationModel.prototype.$updateFromLive = function (liveModel) {
    var model = this.$getTableModel(),
      liveRows = liveModel.getRows(),
      rows = model.getRows(),
      different = false;
    for (var i = 0; i < liveRows.length; i++) {
      var row = rows[i],
        liveRow = liveRows[i],
        liveSubject = liveRow.getSubject();
      if (!row) {
        different = true;
        break;
      }
      var subject = row.getSubject();
      if (!liveSubject || !liveSubject.equivalent(subject)) {
        different = true;
        break;
      }
    }
    if (different) {
      return model.clearRows().then(function () {
        return model.insertRows(liveModel.getRows());
      });
    }
    return Promise.resolve();
  };

  /**
   * Get the lastTime for the history
   * @private
   * @returns {Promise}
   */
  HistoryTablePaginationModel.prototype.$resolveLastTime = function () {
    var that = this;
    return queryServletUtil.resolveLastTime(that.$ord).then(function (lastTimestamp) {
      that.$lastTimestamp = lastTimestamp;
      return lastTimestamp;
    });
  };
  return HistoryTablePaginationModel;
});
