/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/baja/TrendFlags
 */
define(['baja!', 'lex!history'], function (baja, lexs) {
  'use strict';

  var historyLex = lexs[0];
  var Simple = baja.Simple;
  var SUPPORT = {
    0x01: 'start',
    0x02: 'outOfOrder',
    0x04: 'hidden',
    0x08: 'modified',
    0x10: 'interpolated',
    0x20: 'reserved0',
    0x40: 'reserved1',
    0x80: 'reserved2'
  };

  /**
   * Type extension for `history:TrendFlags`.
   *
   * @class
   * @alias module:nmodule/history/rc/baja/TrendFlags
   * @extends baja.Simple
   * @param {number} bits
   */
  var TrendFlags = function TrendFlags(bits) {
    Simple.apply(this, arguments);
    this.$bits = bits & 0xff;
  };
  TrendFlags.prototype = Object.create(Simple.prototype);
  TrendFlags.prototype.constructor = TrendFlags;

  /**
   * @param {string} str
   * @returns {module:nmodule/history/rc/baja/TrendFlags}
   */
  TrendFlags.prototype.decodeFromString = function (str) {
    var bits = parseInt(str, 16);
    if (isNaN(bits)) {
      throw new Error('Invalid bits: ' + str);
    }
    return TrendFlags.make(bits);
  };

  /**
   * @returns {String}
   */
  TrendFlags.prototype.encodeToString = function () {
    return this.$bits.toString(16);
  };

  /**
   * @param {number} ordinal
   * @returns {boolean} true if this ordinal is set
   */
  TrendFlags.prototype.getBit = function (ordinal) {
    return !!(this.$bits & ordinal);
  };

  /**
   * @returns {number}
   */
  TrendFlags.prototype.getBits = function () {
    return this.$bits;
  };

  /**
   * @returns {number[]} all available ordinals for a `TrendFlags`
   */
  TrendFlags.prototype.getOrdinals = function () {
    return [TrendFlags.START, TrendFlags.OUT_OF_ORDER, TrendFlags.HIDDEN, TrendFlags.MODIFIED, TrendFlags.INTERPOLATED];
  };

  /**
   * @param {number} bits
   * @returns {module:nmodule/history/rc/baja/TrendFlags}
   */
  TrendFlags.make = function (bits) {
    return bits ? new TrendFlags(bits) : TrendFlags.DEFAULT;
  };

  /**
   * @param {number} bits
   * @returns {module:nmodule/history/rc/baja/TrendFlags}
   */
  TrendFlags.prototype.make = function (bits) {
    return TrendFlags.make(bits);
  };

  /**
   * @returns {baja.EnumSet} an `EnumSet` representing which bits are selected
   * out of the available bits for a `TrendFlags`
   */
  TrendFlags.prototype.toEnumSet = function () {
    var that = this,
      ordinals = that.getOrdinals(),
      range = baja.EnumRange.make({
        ordinals: ordinals,
        tags: ['start', 'outOfOrder', 'hidden', 'modified', 'interpolated']
      });
    return baja.EnumSet.make({
      ordinals: ordinals.filter(function (ordinal) {
        return that.getBit(ordinal);
      }),
      range: range
    });
  };
  TrendFlags.prototype.toString = function (cx) {
    var bits = this.getBits();
    if (!bits) {
      return '{ }';
    }
    var tags = [];
    for (var bit in SUPPORT) {
      if (bits & bit) {
        var tag = SUPPORT[bit],
          key = 'TrendFlags.' + tag;
        tags.push(historyLex.get({
          key: key,
          def: tag
        }));
      }
    }
    return '{' + tags.join(',') + '}';
  };
  TrendFlags.DEFAULT = new TrendFlags(0);
  TrendFlags.START = 0x01;
  TrendFlags.OUT_OF_ORDER = 0x02;
  TrendFlags.HIDDEN = 0x04;
  TrendFlags.MODIFIED = 0x08;
  TrendFlags.INTERPOLATED = 0x10;
  TrendFlags.RESERVED_0 = 0x20;
  TrendFlags.RESERVED_1 = 0x40;
  TrendFlags.RESERVED_2 = 0x80;
  return TrendFlags;
});
