/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/ndio/rc/baja/TabularThermistorConversion
 */
define(['baja!', 'nmodule/driver/rc/baja/XYPoint', 'underscore'], function (baja, XYPoint, _) {
  'use strict';

  var objUtil = require('bajaScript/baja/obj/objUtil');
  var Simple = baja.Simple,
    cacheDecode = objUtil.cacheDecode,
    cacheEncode = objUtil.cacheEncode;

  /**
   * BajaScript representation of a `ndio:TabularThermistorConversion` value.
   *
   * @class
   * @extends {baja.Simple}
   * @alias {module:nmodule/ndio/rc/baja/TabularThermistorConversion}
   */
  var TabularThermistorConversion = function TabularThermistorConversion() {
    Simple.apply(this, arguments);
  };
  TabularThermistorConversion.prototype = Object.create(Simple.prototype);
  TabularThermistorConversion.prototype.constructor = TabularThermistorConversion;

  /**
   * Create a new `ndio:TabularThermistorConversion` instance from the arguments.
   * @returns {module:nmodule/ndio/rc/baja/TabularThermistorConversion}
   */
  TabularThermistorConversion.make = function (desc, pnts) {
    if (!Array.isArray(pnts) || pnts.length < 2) {
      throw new Error('An array of at least two XYPoints is required.');
    }
    if (!desc) {
      desc = "";
    }
    var tc = new TabularThermistorConversion();
    tc.$desc = desc;
    tc.$pnts = pnts;
    return tc;
  };

  /**
   * @see TabularThermistorConversion.make
   * @returns {module:nmodule/ndio/rc/baja/TabularThermistorConversion}
   */
  TabularThermistorConversion.prototype.make = function () {
    return TabularThermistorConversion.make.apply(TabularThermistorConversion, arguments);
  };

  /**
   * Decode an `TabularThermistorConversion` from a string.
   *
   * @function
   * @param {String} str
   * @returns {module:nmodule/ndio/rc/baja/TabularThermistorConversion}
   */
  TabularThermistorConversion.prototype.decodeFromString = cacheDecode(function (str) {
    if (!str || str.split("|").length < 2) {
      return TabularThermistorConversion.DEFAULT;
    }
    var a = str.split("|");
    var pts = _.filter(a[1].split(";"), function (pt) {
      return pt !== "" && pt.split(',').length === 2;
    });
    return TabularThermistorConversion.make(baja.SlotPath.unescape(a[0]), _.map(pts, function (pt) {
      if (pt !== "") {
        return XYPoint.decodeFromString(pt);
      }
      return null;
    }));
  });

  /**
   * Encode this `TabularThermistorConversion` to a string.
   *
   * @function
   * @returns {String}
   */
  TabularThermistorConversion.prototype.encodeToString = cacheEncode(function () {
    return baja.SlotPath.escape(this.$desc) + "|" + _.map(this.$pnts, function (point) {
      return point.encodeToString();
    }).join(";");
  });

  /**
   * Get the XYPoint objects configured for this conversion
   * @returns {Array} Array of XYPoint
   */
  TabularThermistorConversion.prototype.getPoints = function () {
    return this.$pnts.slice();
  };
  /**
   * Get the description of this conversion object
   * @returns {*|string}
   */
  TabularThermistorConversion.prototype.getDescription = function () {
    return this.$desc;
  };
  TabularThermistorConversion.DEFAULT = TabularThermistorConversion.make("Thermistor Type 3", [new XYPoint({
    x: 0.0,
    y: 165.0
  }), new XYPoint({
    x: 610.0,
    y: 110.0
  }), new XYPoint({
    x: 1060.0,
    y: 90.0
  }), new XYPoint({
    x: 1690.0,
    y: 75.0
  }), new XYPoint({
    x: 2320.0,
    y: 65.0
  }), new XYPoint({
    x: 3250.0,
    y: 55.0
  }), new XYPoint({
    x: 4620.0,
    y: 45.0
  }), new XYPoint({
    x: 6240.0,
    y: 37.0
  }), new XYPoint({
    x: 8197.0,
    y: 30.0
  }), new XYPoint({
    x: 10000.0,
    y: 25.0
  }), new XYPoint({
    x: 12268.0,
    y: 20.0
  }), new XYPoint({
    x: 15136.0,
    y: 15.0
  }), new XYPoint({
    x: 18787.0,
    y: 10.0
  }), new XYPoint({
    x: 23462.0,
    y: 5.0
  }), new XYPoint({
    x: 29490.0,
    y: 0.0
  }), new XYPoint({
    x: 37316.0,
    y: -5.0
  }), new XYPoint({
    x: 47549.0,
    y: -10.0
  }), new XYPoint({
    x: 61030.0,
    y: -15.0
  }), new XYPoint({
    x: 78930.0,
    y: -20.0
  }), new XYPoint({
    x: 100000.0,
    y: -25.0
  })]);

  /**
   * Register this baja type as 'ndio:TabularThermistorConversion'
   */
  baja.registerType('ndio:TabularThermistorConversion', function () {
    return TabularThermistorConversion;
  });
  return TabularThermistorConversion;
});
