/**
 * @copyright  2019 Tridium, Inc. All Rights Reserved.
 * @author Ashutosh Chaturvedi
 */

/**
 * API Status: **Private**
 * @module nmodule/cloudLink/rc/history/util/CloudHistoryExportUnassignCommand
 */
define(['baja!', 'dialogs', 'lex!cloudLink', 'Promise', 'underscore', 'bajaux/commands/Command', 'nmodule/webEditors/rc/wb/mixin/mixinUtils', 'nmodule/webEditors/rc/wb/mgr/mgrUtils'], function (baja, dialogs, lexs, Promise, _, Command, mixinUtils, mgrUtils) {
  'use strict';

  var cloudLinkLex = lexs[0],
      getMainTableSelectedSubjects = mgrUtils.getMainTableSelectedSubjects,
      getLearnTableSelectedSubjects = mgrUtils.getLearnTableSelectedSubjects;
  /**
   * Remove history ids from their currently assigned history export configs.
   * @param {module:nmodule/cloudLink/rc/history/mgr/CloudHistoryExportUxManager} mgr - the
   * Cloud History Export Manager
   * @param {Object} configToHistoryIdsToRemoveMap - an object containing a mapping from history export config names to the configs
   * themselves and to the list of history ids to be removed from that config.
   * @param {Number} maxSimultaneousPromises - the maximum number of unresolved promises that may exist at one time
   * @returns {Promise}
   */

  function removeHistoryIdsFromPreviousCloudHistoryExportConfig(mgr, configToHistoryIdsToRemoveMap, maxSimultaneousPromises) {
    var chunkSize = maxSimultaneousPromises || 100;

    function processHistoryIdsRemoveChunk(component, historyIdSlots) {
      if (!historyIdSlots.length) {
        return Promise.resolve();
      }

      return Promise.all(historyIdSlots.slice(0, chunkSize).map(function (historyIdSlot) {
        return component.remove(historyIdSlot);
      })).then(function () {
        return processHistoryIdsRemoveChunk(component, historyIdSlots.slice(chunkSize));
      });
    }

    function processConfigMaps(configMaps) {
      if (!configMaps.length) {
        return Promise.resolve();
      }

      var configMap = configMaps[0];
      var assignedHistoryIdsComponent;
      var historyIdSlots;

      if (baja.hasType(configMap['config'], 'cloudLink:CloudHistoryAutoExportConfig')) {
        assignedHistoryIdsComponent = configMap['config'].getExcludedHistoryIds();
        historyIdSlots = assignedHistoryIdsComponent.getSlots().is('baja:Integer').toArray().filter(function (historyIdSlot) {
          return baja.SlotPath.unescape(historyIdSlot.getName()) in configMap['historyIds'];
        });
      } else {
        assignedHistoryIdsComponent = configMap['config'].getHistoryIds();
        historyIdSlots = assignedHistoryIdsComponent.getSlots().is('history:HistoryId').toArray().filter(function (historyIdSlot) {
          return assignedHistoryIdsComponent.get(historyIdSlot).encodeToString() in configMap['historyIds'];
        });
      }

      return processHistoryIdsRemoveChunk(assignedHistoryIdsComponent, historyIdSlots).then(function () {
        return processConfigMaps(configMaps.slice(1));
      });
    }

    return processConfigMaps(_.values(configToHistoryIdsToRemoveMap));
  }
  /**
   * For the selected discovery rows, if they contain a history id and are currently assigned
   * to a cloud history export config, unassign them and remove them from the assigned
   * cloud history export's history id list.
   * @param {module:nmodule/cloudLink/rc/history/mgr/CloudHistoryExportUxManager} mgr - the
   * Cloud History Export Manager
   * @returns {Promise}
   */


  function unassignCloudHistoryExportConfigFromSelectedHistories(mgr) {
    var learnTableSelection = mgrUtils.getLearnTableSelection(mgr),
        selectedDiscoveryRows = learnTableSelection.getSelectedElements(mgr.getLearnModel().getRows()),
        mainTableSelectionConfigNames = mgrUtils.getMainTableSelectedSubjects(mgr).map(function (row) {
      return row.getName();
    }),
        historyIdsToRemove = {};
    var maxSimultaneousPromises = 100;

    function handleRow(row) {
      if (!row.getSubject().getHistoryId) {
        return;
      }

      var historyId = row.getSubject().getHistoryId(),
          subjects = mgrUtils.getMainTableSelectedSubjects(mgr),
          prevCloudHistoryConfig;

      if (subjects.length === 1 && baja.hasType(subjects[0], "cloudLink:CloudHistoryAutoExportConfig")) {
        var autoExportConfig = mgr.getCloudHistoryAutoExportConfigFromHistoryId(historyId.toString());

        if (subjects[0] === autoExportConfig) {
          prevCloudHistoryConfig = autoExportConfig;
        }
      } else {
        prevCloudHistoryConfig = mgr.getCloudHistoryExportConfigFromHistoryId(historyId.toString());
      } // Remove from previously assigned cloud history export config only if it is one of the selected configs in
      // the main table


      if (prevCloudHistoryConfig && mainTableSelectionConfigNames.includes(prevCloudHistoryConfig.getName())) {
        if (!(prevCloudHistoryConfig.getName() in historyIdsToRemove)) {
          historyIdsToRemove[prevCloudHistoryConfig.getName()] = {
            config: prevCloudHistoryConfig,
            historyIds: {}
          };
        }

        historyIdsToRemove[prevCloudHistoryConfig.getName()]['historyIds'][historyId.encodeToString()] = true;
      }
    }

    selectedDiscoveryRows.forEach(function (row) {
      return handleRow(row);
    });
    return removeHistoryIdsFromPreviousCloudHistoryExportConfig(mgr, historyIdsToRemove, maxSimultaneousPromises);
  }
  /**
   * Command to unassign discovered history id(s) from their assigned cloud
   * history export config objects.
   * This requires at least one row selected from the main table and at least one row selected in the learn table.
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/cloudLink/rc/history/util/CloudHistoryExportUnassignCommand
   * @param {module:nmodule/cloudLink/rc/history/mgr/CloudHistoryExportUxManager} mgr
   */


  var CloudHistoryExportUnassignCommand = function CloudHistoryExportUnassignCommand(mgr) {
    Command.call(this, {
      module: 'cloudLink',
      lex: 'cloudHistoryExportLearn.unassign',
      enabled: false,
      flags: Command.flags.MENU_BAR,
      func: function func() {
        var _this = this;

        if (!mixinUtils.hasMixin(mgr, 'MGR_LEARN')) {
          return Promise.resolve();
        }

        var historiesSelectionSubjects = getLearnTableSelectedSubjects(mgr);

        if (historiesSelectionSubjects.length >= 1) {
          return dialogs.showLoading(500, unassignCloudHistoryExportConfigFromSelectedHistories(mgr).then(function () {
            _this.setEnabled(false);

            mgr.setAssignCommandEnabled(true);
          })).promise();
        } else {
          return Promise.reject(new Error(cloudLinkLex.get("cloudHistoryExportLearn.unassign.error")));
        }
      }
    });
    this.$mgr = mgr;
  };

  CloudHistoryExportUnassignCommand.prototype = Object.create(Command.prototype);
  CloudHistoryExportUnassignCommand.prototype.constructor = CloudHistoryExportUnassignCommand;
  /**
   * Function called when the selection in the main table or the learn
   * table changes. The unassign command should only be enabled when at least one item
   * is selected in the main table, at least one item is selected in the learn table, and an unassign is capable of
   * being performed (at least one entry in the learn table selection is currently assigned to an entry in
   * the main table).
   *
   * @param learnModeSelectionState The learn mode selection state
   */

  CloudHistoryExportUnassignCommand.prototype.tableSelectionChanged = function (learnModeSelectionState) {
    var mainTableSelectedSubjects = getMainTableSelectedSubjects(this.$mgr);

    if (mainTableSelectedSubjects.length === 1 && baja.hasType(mainTableSelectedSubjects[0], "cloudLink:CloudHistoryAutoExportConfig")) {
      this.$displayName = cloudLinkLex.get('cloudHistoryExportLearn.unassign.displayNameAuto');
      this.$description = cloudLinkLex.get('cloudHistoryExportLearn.unassign.descriptionAuto');
    } else {
      this.$displayName = cloudLinkLex.get('cloudHistoryExportLearn.unassign.displayName');
      this.$description = cloudLinkLex.get('cloudHistoryExportLearn.unassign.description');
    } // If we've selected the Cloud History Auto Export along with one or more other cloud history export configs, we don't know whether
    // to Unassign or Include, so set to disabled


    if (mainTableSelectedSubjects.length === 0 || mainTableSelectedSubjects.length > 1 && baja.hasType(mainTableSelectedSubjects[0], "cloudLink:CloudHistoryAutoExportConfig")) {
      return this.setEnabled(false);
    }

    var learnTableSelection = mgrUtils.getLearnTableSelection(this.$mgr),
        selectedDiscoveryRows = learnTableSelection.getSelectedElements(this.$mgr.getLearnModel().getRows()),
        selectedDiscoveryRowsWithoutHistoryId = selectedDiscoveryRows.filter(function (row) {
      return !row.getSubject().getHistoryId;
    }),
        isValidRowSelectedInDiscovery = selectedDiscoveryRows.length >= 1 && selectedDiscoveryRowsWithoutHistoryId.length === 0;

    if (!learnModeSelectionState && isValidRowSelectedInDiscovery) {
      var mainTableSelectionConfigNames = mainTableSelectedSubjects.map(function (row) {
        return row.getName();
      });

      for (var i = 0; i < selectedDiscoveryRows.length; i++) {
        if (selectedDiscoveryRows[i].getSubject().getHistoryId) {
          var historyId = selectedDiscoveryRows[i].getSubject().getHistoryId(),
              prevCloudHistoryConfig = this.$mgr.getCloudHistoryExportConfigFromHistoryId(historyId.toString()),
              prevAutoExportConfig = this.$mgr.getCloudHistoryAutoExportConfigFromHistoryId(historyId.toString());

          if (prevAutoExportConfig && baja.hasType(mainTableSelectedSubjects[0], "cloudLink:CloudHistoryAutoExportConfig") || prevCloudHistoryConfig && mainTableSelectionConfigNames.includes(prevCloudHistoryConfig.getName())) {
            this.setEnabled(true);
            return;
          }
        }
      }
    }

    this.setEnabled(false);
  };

  return CloudHistoryExportUnassignCommand;
});
