/**
 * @copyright  2019 Tridium, Inc. All Rights Reserved.
 * @author Ashutosh Chaturvedi
 */

/**
 * API Status: **Private**
 * @module nmodule/cloudLink/rc/history/util/CloudHistoryExportAssignCommand
 */
define(['baja!', 'dialogs', 'lex!cloudLink', 'Promise', 'underscore', 'bajaux/commands/Command', 'nmodule/webEditors/rc/wb/mixin/mixinUtils', 'nmodule/webEditors/rc/wb/mgr/mgrUtils'], function (baja, dialogs, lexs, Promise, _, Command, mixinUtils, mgrUtils) {
  'use strict';

  var cloudLinkLex = lexs[0],
      getMainTableSelectedSubjects = mgrUtils.getMainTableSelectedSubjects,
      getLearnTableSelectedSubjects = mgrUtils.getLearnTableSelectedSubjects;
  /**
   * Add history ids to a cloud history export config.
   * @param {module:nmodule/cloudLink/rc/history/mgr/CloudHistoryExportUxManager} mgr - the
   * Cloud History Export Manager
   * @param {baja.Component} newCloudHistoryConfig - the selected existing cloud
   * history export config database component
   * @param {Array<module:nmodule/history/rc/baja/HistoryId>} historyIds - the selected history ids
   * @param {Number} maxSimultaneousPromises - the maximum number of unresolved promises that may exist at one time
   * @returns {Promise}
   */

  function addHistoryIdsToSelectedCloudHistoryExportConfig(mgr, newCloudHistoryConfig, historyIds, maxSimultaneousPromises) {
    var chunkSize = maxSimultaneousPromises || 100;

    function processHistoryIdsAddChunk(historyIds) {
      if (!historyIds.length) {
        return Promise.resolve();
      }

      return Promise.all(historyIds.slice(0, chunkSize).map(function (historyId) {
        if (baja.hasType(newCloudHistoryConfig, 'cloudLink:CloudHistoryAutoExportConfig')) {
          // add histories to the auto export config's excluded historyIds set
          return newCloudHistoryConfig.getExcludedHistoryIds().add({
            slot: baja.SlotPath.escape(historyId.encodeToString()),
            value: baja.Integer.make(mgr.EXCLUDED_BY_MANAGER),
            flags: baja.Flags.READONLY
          });
        } else {
          // add histories to the new cloud history export config's historyIds set
          return newCloudHistoryConfig.getHistoryIds().add({
            slot: 'historyId?',
            value: historyId,
            flags: baja.Flags.READONLY
          });
        }
      })).then(function () {
        return processHistoryIdsAddChunk(historyIds.slice(chunkSize));
      });
    }

    return processHistoryIdsAddChunk(historyIds);
  }
  /**
   * Remove history ids from their currently assigned history export configs.
   * @param {module:nmodule/cloudLink/rc/history/mgr/CloudHistoryExportUxManager} mgr - the
   * Cloud History Export Manager
   * @param {Object} configToHistoryIdsToRemoveMap - an object containing a mapping from history export config names to the configs
   * themselves and to the list of history ids to be removed from that config.
   * @param {Number} maxSimultaneousPromises - the maximum number of unresolved promises that may exist at one time
   * @returns {Promise}
   */


  function removeHistoryIdsFromPreviousCloudHistoryExportConfigs(mgr, configToHistoryIdsToRemoveMap, maxSimultaneousPromises) {
    var chunkSize = maxSimultaneousPromises || 100;

    function processHistoryIdsRemoveChunk(component, historyIdSlots) {
      if (!historyIdSlots.length) {
        return Promise.resolve();
      }

      return Promise.all(historyIdSlots.slice(0, chunkSize).map(function (historyIdSlot) {
        return component.remove(historyIdSlot);
      })).then(function () {
        return processHistoryIdsRemoveChunk(component, historyIdSlots.slice(chunkSize));
      });
    }

    function processConfigMaps(configMaps) {
      if (!configMaps.length) {
        return Promise.resolve();
      }

      var configMap = configMaps[0];
      var assignedHistoryIdsComponent = configMap['config'].getHistoryIds();
      var historyIdSlots = assignedHistoryIdsComponent.getSlots().is('history:HistoryId').toArray().filter(function (historyIdSlot) {
        return assignedHistoryIdsComponent.get(historyIdSlot).encodeToString() in configMap['historyIds'];
      });
      return processHistoryIdsRemoveChunk(assignedHistoryIdsComponent, historyIdSlots).then(function () {
        return processConfigMaps(configMaps.slice(1));
      });
    }

    return processConfigMaps(_.values(configToHistoryIdsToRemoveMap));
  }
  /**
   * For the selected discovery rows, if they contain a history id and are being assigned
   * to a new or different cloud history export config, add them to the history id list
   * of the new cloud history export config and remove them from the existing ones.
   * @param {module:nmodule/cloudLink/rc/history/mgr/CloudHistoryExportUxManager} mgr - the
   * Cloud History Export Manager
   * @param {baja.Component} newCloudHistoryConfig the new cloud history export config
   * @returns {Promise}
   */


  function assignCloudHistoryExportConfigToSelectedHistories(mgr, newCloudHistoryConfig) {
    var learnTableSelection = mgrUtils.getLearnTableSelection(mgr),
        selectedDiscoveryRows = learnTableSelection.getSelectedElements(mgr.getLearnModel().getRows()),
        historyIdsToAdd = [],
        historyIdsToRemove = {};
    var maxSimultaneousPromises = 100;

    function handleRow(row) {
      if (!row.getSubject().getHistoryId) {
        return;
      }

      var assigningToAutoConfig = baja.hasType(newCloudHistoryConfig, "cloudLink:CloudHistoryAutoExportConfig"),
          historyId = row.getSubject().getHistoryId(),
          prevCloudHistoryConfig = mgr.getCloudHistoryExportConfigFromHistoryId(historyId.toString()),
          prevAutoExportConfig = mgr.getCloudHistoryAutoExportConfigFromHistoryId(historyId.toString());

      if (assigningToAutoConfig && !prevAutoExportConfig || !assigningToAutoConfig && prevCloudHistoryConfig !== newCloudHistoryConfig) {
        historyIdsToAdd.push(historyId); // Remove historyId from previous cloud history export config if there is one

        if (!assigningToAutoConfig && prevCloudHistoryConfig) {
          if (!(prevCloudHistoryConfig.getName() in historyIdsToRemove)) {
            historyIdsToRemove[prevCloudHistoryConfig.getName()] = {
              config: prevCloudHistoryConfig,
              historyIds: {}
            };
          }

          historyIdsToRemove[prevCloudHistoryConfig.getName()]['historyIds'][historyId.encodeToString()] = true;
        }
      }
    }

    selectedDiscoveryRows.forEach(function (row) {
      return handleRow(row);
    });
    return removeHistoryIdsFromPreviousCloudHistoryExportConfigs(mgr, historyIdsToRemove, maxSimultaneousPromises).then(function () {
      return addHistoryIdsToSelectedCloudHistoryExportConfig(mgr, newCloudHistoryConfig, historyIdsToAdd, maxSimultaneousPromises);
    });
  }
  /**
   * Command to assign a discovered history id/ids to an existing cloud history
   * export config object. This requires one subject to be selected in the main
   * database table and at least one from the discovered history ids table.
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/cloudLink/rc/history/util/CloudHistoryExportAssignCommand
   * @param {module:nmodule/cloudLink/rc/history/mgr/CloudHistoryExportUxManager} mgr
   */


  var CloudHistoryExportAssignCommand = function CloudHistoryExportAssignCommand(mgr) {
    Command.call(this, {
      module: 'cloudLink',
      lex: 'cloudHistoryExportLearn.assign',
      enabled: false,
      flags: Command.flags.MENU_BAR,

      /**
       * Match the selected object(s) in the discovery table to the selected object
       * in the main database table.
       *
       * @alias module:nmodule/cloudLink/rc/history/util/CloudHistoryExportAssignCommand#invoke
       * @returns {Promise}
       */
      func: function func() {
        var _this = this;

        var exportConfigSelectionSubjects, historiesSelectionSubjects;

        if (!mixinUtils.hasMixin(mgr, 'MGR_LEARN')) {
          return Promise.resolve();
        }

        exportConfigSelectionSubjects = getMainTableSelectedSubjects(mgr);
        historiesSelectionSubjects = getLearnTableSelectedSubjects(mgr);

        if (exportConfigSelectionSubjects.length === 1 && historiesSelectionSubjects.length >= 1) {
          try {
            return dialogs.showLoading(500, assignCloudHistoryExportConfigToSelectedHistories(mgr, exportConfigSelectionSubjects[0]).then(function () {
              _this.setEnabled(false); // disable this button once done.


              mgr.setUnassignCommandEnabled(true);
            })).promise();
          } catch (e) {
            return Promise.reject(new Error('No change for row without history id.'));
          }
        } else {
          return Promise.reject(new Error('Must have one item selected in the main database table and at least one item selected in the learn table'));
        }
      }
    });
    this.$mgr = mgr;
  };

  CloudHistoryExportAssignCommand.prototype = Object.create(Command.prototype);
  CloudHistoryExportAssignCommand.prototype.constructor = CloudHistoryExportAssignCommand;
  /**
   * Function called when the selection in the main table or the learn
   * table changes. The assign command should only be enabled when one item
   * is selected in the main table, at least one item is selected in the learn table, and an assign is capable of
   * being performed (at least one entry in the learn table selection is not currently assigned to the selected entry in
   * the main table).
   * @param learnModeSelectionState The learn mode selection state
   */

  CloudHistoryExportAssignCommand.prototype.tableSelectionChanged = function (learnModeSelectionState) {
    var mainTableSelectedSubjects = getMainTableSelectedSubjects(this.$mgr),
        onlyAutoExportSelected = mainTableSelectedSubjects.length === 1 && baja.hasType(mainTableSelectedSubjects[0], "cloudLink:CloudHistoryAutoExportConfig");

    if (onlyAutoExportSelected) {
      this.$displayName = cloudLinkLex.get('cloudHistoryExportLearn.assign.displayNameAuto');
      this.$description = cloudLinkLex.get('cloudHistoryExportLearn.assign.descriptionAuto');
    } else {
      this.$displayName = cloudLinkLex.get('cloudHistoryExportLearn.assign.displayName');
      this.$description = cloudLinkLex.get('cloudHistoryExportLearn.assign.description');
    }

    var learnTableSelection = mgrUtils.getLearnTableSelection(this.$mgr),
        selectedDiscoveryRows = learnTableSelection.getSelectedElements(this.$mgr.getLearnModel().getRows()),
        selectedDiscoveryRowsWithoutHistoryId = selectedDiscoveryRows.filter(function (row) {
      return !row.getSubject().getHistoryId;
    }),
        isValidRowSelectedInDiscovery = selectedDiscoveryRows.length >= 1 && selectedDiscoveryRowsWithoutHistoryId.length === 0;

    if (!learnModeSelectionState && mainTableSelectedSubjects.length === 1 && isValidRowSelectedInDiscovery) {
      var mainTableSelection = mainTableSelectedSubjects[0];

      for (var i = 0; i < selectedDiscoveryRows.length; i++) {
        if (selectedDiscoveryRows[i].getSubject().getHistoryId) {
          var historyId = selectedDiscoveryRows[i].getSubject().getHistoryId(),
              prevCloudHistoryConfig = this.$mgr.getCloudHistoryExportConfigFromHistoryId(historyId.toString()),
              prevAutoExportConfig = this.$mgr.getCloudHistoryAutoExportConfigFromHistoryId(historyId.toString());

          if (onlyAutoExportSelected && !prevAutoExportConfig || !onlyAutoExportSelected && !prevCloudHistoryConfig || !onlyAutoExportSelected && prevCloudHistoryConfig.getName() !== mainTableSelection.getName()) {
            this.setEnabled(true);
            return;
          }
        }
      }
    }

    this.setEnabled(false);
  };

  return CloudHistoryExportAssignCommand;
});
