/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Ashutosh Chaturvedi
 */

/**
 * API Status: **Private**
 * @module nmodule/cloudLink/rc/history/model/CloudHistoryExportLearnModel
 */
define(['baja!', 'lex!cloudLink', 'nmodule/cloudLink/rc/history/util/IconDisplayNameColumn', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/wb/table/tree/TreeTableModel'], function (baja, lexs, IconDisplayNameColumn, Column, TreeTableModel) {
  'use strict';

  var cloudLinkLex = lexs[0];
  /**
   * ConfigColumn shows the Cloud History Config to which the LocalDbHistory point
   * belongs to.
   * Contains the manager to fetch the CloudHistoryConfig for the HistoryId
   */

  var ConfigColumn = function ConfigColumn(name, params) {
    Column.apply(this, arguments);
    this.mgr = params.manager;
  };

  ConfigColumn.prototype = Object.create(Column.prototype);
  ConfigColumn.prototype.constructor = ConfigColumn;
  /**
   * The function returns the mapped Cloud History Export config for the history
   * Id. It makes a call to the manager's `getCloudHistoryExportConfigFromHistoryId`
   * function and pass the history id which returns the config for it.
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row from the discovery table
   * @returns {string} the mapped Cloud History Export Config
   */

  ConfigColumn.prototype.getValueFor = function (row) {
    var node = row.getSubject();

    if (node.getHistoryId) {
      var id = node.getHistoryId(),
          manager = this.mgr;
      var config = manager.getCloudHistoryExportConfigFromHistoryId(String(id));
      return config && config.getDisplayName() || '';
    }

    return '';
  };
  /**
   * ExcludeColumn shows the status of the history in the Cloud History Auto Export Config.
   * Contains the manager to fetch the CloudHistoryConfig for the HistoryId
   */


  var ExcludeColumn = function ExcludeColumn(name, params) {
    Column.apply(this, arguments);
    this.mgr = params.manager;
  };

  ExcludeColumn.prototype = Object.create(Column.prototype);
  ExcludeColumn.prototype.constructor = ExcludeColumn;
  /**
   * The function returns the status of the history in the auto export config.
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row from the discovery table
   * @returns {string} the mapped Cloud History Export Config
   */

  ExcludeColumn.prototype.getValueFor = function (row) {
    var node = row.getSubject();

    if (node.getHistoryId) {
      var id = node.getHistoryId(),
          manager = this.mgr;
      var config = manager.getCloudHistoryAutoExportConfigFromHistoryId(String(id));
      var assignedConfig = manager.getCloudHistoryExportConfigFromHistoryId(String(id));
      return config || assignedConfig ? cloudLinkLex.get('cloudHistoryExportLearn.excluded') : cloudLinkLex.get('cloudHistoryExportLearn.included');
    }

    return '';
  };
  /**
   * @private
   * @exports nmodule/cloudLink/rc/history/model/CloudHistoryExportLearnModel
   */


  var exports = {};
  /**
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */

  exports.make = function (mgr) {
    return TreeTableModel.make({
      columns: [new IconDisplayNameColumn('history', {
        displayName: cloudLinkLex.get('cloudHistoryExportLearn.history')
      }), new ConfigColumn('config', {
        displayName: cloudLinkLex.get('cloudHistoryExportLearn.assignedConfig'),
        manager: mgr
      }), new ExcludeColumn('exclude', {
        displayName: cloudLinkLex.get('cloudHistoryExportLearn.autoConfig'),
        manager: mgr
      })]
    });
  };

  return exports;
});
