/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/* eslint-env browser */

/**
 * API Status: **Private**
 * @module nmodule/cloudLinkForge/rc/DeviceRegistrationWidget
 */
define(['baja!', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'lex!cloudLinkForge', 'jquery', 'underscore', 'dialogs', 'hbs!nmodule/cloudLinkForge/rc/DeviceRegistrationWidgetTemplate', 'css!nmodule/cloudLinkForge/rc/DeviceRegistrationWidgetStyle'], function (baja, Widget, subscriberMixIn, lexicons, $, _, dialogs, devRegTemplate) {
  'use strict';

  var lex = lexicons[0],
      successImg = "<img src='/module/icons/x16/job/success.png'/>",
      failureImg = "<img src='/module/icons/x16/job/failed.png'/>",
      pendingImg = "<img class='devRegSpin' src='/module/icons/x16/refresh.png'/>";
  /**
   * A Widget that enables a device to be registered with the Cloud.
   *
   * @class
   * @alias module:nmodule/cloudLinkForge/rc/DeviceRegistrationWidget
   * @extends module:bajaux/Widget
   */

  var DeviceRegistrationWidget = function DeviceRegistrationWidget() {
    var that = this;
    Widget.apply(that, arguments);
    subscriberMixIn(that);
  };

  DeviceRegistrationWidget.prototype = new Widget();
  DeviceRegistrationWidget.prototype.constructor = DeviceRegistrationWidget;

  DeviceRegistrationWidget.prototype.doLoad = function (auth) {
    var that = this; //need to make sure there is a function of getConnectionInfo()

    function registerDevice() {
      return auth.rpc("getBajaVersion").then(function (bajaVersion) {
        var url = auth.getRegistrationUrl(),
            query = "";

        function addParam(name, val) {
          query += (query ? "&" : "?") + name + "=" + encodeURIComponent(val);
        }

        addParam("systemId", auth.getSystemId());
        addParam("systemType", auth.getSystemType());
        addParam("systemOwnershipCode", auth.getSystemOwnershipCode());
        addParam("systemPublicKey", auth.getSystemPublicKey());
        addParam("systemRegistrationState", auth.getRegistrationState().getTag());

        if (auth.getSystemGuid()) {
          addParam("systemGuid", auth.getSystemGuid());
        }

        addParam("bajaVersion", bajaVersion);

        if (auth.getGatewayId() !== "") {
          addParam("gatewayid", auth.getGatewayId());
        }

        url += query; // Seems to be some re-encoding issues with URLs and ORDs so workaround this for now.

        if (window.niagara.env.type === "wb") {
          window.niagara.env.hyperlink(url);
        } else {
          window.open(url, "_blank");
        }
      });
    }

    function reauthenticate() {
      auth.rpc("reauthenticate");
    } // When focus is put on the window, attempt a reauthentication if we're currently disconnected.
    // The system has to be registered and the authenticator enabled, otherwise reauthenticate()
    // is guaranteed to fail.


    $(window).on("focus", that.$reauthenticate = function () {
      if (auth.getRegistrationState().is("registered") && auth.getEnabled() && auth.getAuthenticationState().is("unauthenticated")) {
        reauthenticate();
      }
    });
    return that.getSubscriber().subscribe(auth).then(function () {
      var jq = that.jq(),
          regStatusJq,
          currentRegStatus,
          connEnabledJq,
          currentConnEnabled,
          connStateJq,
          currentConnStatus,
          systemTypeJq,
          gatewayIdJq,
          systemGuidJq,
          registerButtonJq,
          reauthenticateButtonJq,
          errorSpanJq;
      jq.html(devRegTemplate({
        title: lex.get("DeviceRegistrationWidget.title"),
        deviceRegistered: lex.get("DeviceRegistrationWidget.deviceRegistered"),
        connectorEnabled: lex.get("DeviceRegistrationWidget.connectorEnabled"),
        deviceConnected: lex.get("DeviceRegistrationWidget.deviceConnected"),
        systemType: lex.get("DeviceRegistrationWidget.systemType"),
        systemGuid: lex.get("DeviceRegistrationWidget.systemGuid"),
        registerDevice: lex.get("DeviceRegistrationWidget.registerDevice"),
        forceReauthenticate: lex.get("DeviceRegistrationWidget.forceReauthenticate"),
        gatewayId: lex.get("DeviceRegistrationWidget.gatewayId")
      }));
      jq.on("click", ".devRegRegister", function (e) {
        e.preventDefault();

        if (auth.getRegistrationState().is("registered") && auth.getAuthenticationState().is("authenticated")) {
          dialogs.showYesNo(lex.get("DeviceRegistrationWidget.alreadyRegistered")).yes(registerDevice); // TODO: This should either not be possible, or should reset the connector first, because
          // otherwise it WILL fail, if we actually attempt re-registration.
        } else {
          registerDevice();
        }
      });
      jq.on("click", ".devRegReauthenticate", function (e) {
        e.preventDefault();
        reauthenticate();
      });
      regStatusJq = jq.find(".devRegStatus");
      connEnabledJq = jq.find(".connEnabled");
      connStateJq = jq.find(".devRegConnState");
      systemTypeJq = jq.find(".devRegSystemType");
      gatewayIdJq = jq.find(".devRegGatewayId");
      systemGuidJq = jq.find(".devRegSystemGuid");
      registerButtonJq = jq.find(".devRegRegister");
      reauthenticateButtonJq = jq.find(".devRegReauthenticate");
      errorSpanJq = jq.find(".devRegError");

      function updateConnectionDetails() {
        var newRegStatus, newConnEnabled, newConnState; // This should show failure for both unregistered and registered but not provisioned states

        newRegStatus = auth.getRegistrationState().is("registered") ? successImg : failureImg;

        if (auth.getGatewayId() === '') {
          gatewayIdJq.parent().hide();
        } else {
          gatewayIdJq.text(auth.getGatewayId());
        }

        if (newRegStatus !== currentRegStatus) {
          regStatusJq.html(currentRegStatus = newRegStatus);
        }

        if (auth.getStatus().isDisabled()) {
          newConnEnabled = failureImg;
        } else {
          newConnEnabled = successImg;
        }

        if (newConnEnabled !== currentConnEnabled) {
          connEnabledJq.html(currentConnEnabled = newConnEnabled);
        }

        if (auth.getAuthenticationState().is("authenticated")) {
          newConnState = successImg;
        } else if (auth.getAuthenticationState().is("unauthenticated") || auth.getAuthenticationState().is("authenticationFailed")) {
          newConnState = failureImg;
        } else {
          newConnState = pendingImg;
        }

        if (newConnState !== currentConnStatus) {
          connStateJq.html(currentConnStatus = newConnState);
        }

        systemTypeJq.text(auth.getSystemType() || lex.get("DeviceRegistrationWidget.none"));
        systemGuidJq.text(auth.getSystemGuid() || lex.get("DeviceRegistrationWidget.none")); // Disable the "Register Device" button when registration is guaranteed to fail:
        // - is in fault
        // - missing system type
        // or when it is not needed, because the system is already authenticated
        // or when already registered
        // or auth is disabled

        if (auth.getStatus().isFault()) {
          registerButtonJq.attr("disabled", "disabled");
          errorSpanJq.text(lex.get("DeviceRegistrationWidget.inFault"));
        } else if (auth.getAuthenticationState().is("authenticated")) {
          registerButtonJq.attr("disabled", "disabled");
          errorSpanJq.text("");
        } else if (!auth.getSystemType() || auth.getSystemType().len === 0) {
          registerButtonJq.attr("disabled", "disabled");
          errorSpanJq.text(lex.get("DeviceRegistrationWidget.missingSystemType"));
        } else if (auth.getRegistrationState().is("registered") || !auth.getEnabled()) {
          registerButtonJq.attr("disabled", "disabled");
          errorSpanJq.text("");
        } else {
          registerButtonJq.removeAttr("disabled");
          errorSpanJq.text("");
        } // Disable the "Reauthenticate" button when we know beforehand that
        // there is no point in trying to connect:  not registered or not enabled


        if (!auth.getRegistrationState().is("registered") || !auth.getEnabled()) {
          reauthenticateButtonJq.attr("disabled", "disabled");
        } else {
          reauthenticateButtonJq.removeAttr("disabled");
        }
      }

      updateConnectionDetails();
      that.getSubscriber().attach("changed", _.throttle(updateConnectionDetails, 100)); // If the service is currently registered and enabled, but disconnected, then attempt reauthentication.

      if (auth.getRegistrationState().is("registered") && auth.getEnabled() && auth.getAuthenticationState().is("unauthenticated")) {
        // If we detect the registration part has failed then ask the user to register the device.
        that.getSubscriber().attach("topicFired", _.once(function (topic) {
          if (topic.getName() === "registrationFailed") {
            dialogs.showYesNo(lex.get("DeviceRegistrationWidget.forward")).yes(registerDevice);
          }
        }));
        reauthenticate();
      }
    });
  };

  DeviceRegistrationWidget.prototype.doDestroy = function () {
    if (this.$reauthenticate) {
      $(window).off("focus", this.$reauthenticate);
    }
  };

  return DeviceRegistrationWidget;
});
