function _toConsumableArray(r) { return _arrayWithoutHoles(r) || _iterableToArray(r) || _unsupportedIterableToArray(r) || _nonIterableSpread(); }
function _nonIterableSpread() { throw new TypeError("Invalid attempt to spread non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _iterableToArray(r) { if ("undefined" != typeof Symbol && null != r[Symbol.iterator] || null != r["@@iterator"]) return Array.from(r); }
function _arrayWithoutHoles(r) { if (Array.isArray(r)) return _arrayLikeToArray(r); }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*eslint-env browser */ /*jshint browser: true */
/*global niagara: false */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/tree/NavTree
 */
define(['baja!', 'log!nmodule.webEditors.rc.wb.tree.NavTree', 'jquery', 'Promise', 'underscore', 'bajaux/events', 'bajaux/dragdrop/dragDropUtils', 'bajaux/Widget', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/IconEditor', 'nmodule/webEditors/rc/util/htmlUtils', 'nmodule/webEditors/rc/util/Switchboard', 'nmodule/webEditors/rc/wb/mixin/TransferSupport', 'nmodule/webEditors/rc/wb/profile/selectionModeSettings', 'nmodule/webEditors/rc/wb/tree/TreeNode', 'hbs!nmodule/webEditors/rc/wb/template/NavTree', 'css!nmodule/webEditors/rc/wb/tree/NavTreeStyle'], function (baja, log, $, Promise, _, events, dragDropUtils, Widget, fe, feDialogs, BaseEditor, IconEditor, htmlUtils, Switchboard, TransferSupport, selectionModeSettings, TreeNode, tplNavTree) {
  'use strict';

  var DESTROY_EVENT = events.DESTROY_EVENT;
  var LOAD_EVENT = events.LOAD_EVENT;
  var MODIFY_EVENT = events.MODIFY_EVENT;
  var EXPAND_SPINNER_DELAY = 250;
  var HOVER_PRELOAD_DELAY = 200;
  var RIGHT_MOUSE_BUTTON = 3;
  var logError = log.severe.bind(log);
  var getSelectionModeFromEvent = selectionModeSettings.getSelectionModeFromEvent,
    TOGGLE_MODE = selectionModeSettings.TOGGLE_MODE,
    SELECT_MODE = selectionModeSettings.SELECT_MODE,
    SWATH_MODE = selectionModeSettings.SWATH_MODE;
  var contextMenuOnLongPress = htmlUtils.contextMenuOnLongPress;
  function widgetDefaults() {
    return {};
  }

  /*
   * TODO: maybe be smart about which node is most likely to be visited first?
   * TODO: limit and configure preload depth
   * TODO: turn red or something if kids fail to load
   */
  function isWb() {
    return typeof niagara !== 'undefined' && niagara.env && niagara.env.type === 'wb';
  }

  //TODO: fix drag highlighting in Workbench (NCCB-9185)
  //purposely break styling in Workbench when dragging, since we never get the
  //dragleave event to remove the styling
  var DROP_TARGET_CLASS = isWb() ? 'wbDropTarget' : 'dropTarget';

  ////////////////////////////////////////////////////////////////
  // Helper methods
  ////////////////////////////////////////////////////////////////

  /**
   * Resolves a promise with the supplied value
   * @param {Object} val
   * @returns {Promise}
   */
  function resolve(val) {
    return Promise.resolve(val);
  }

  /**
   * Rejects a promise with a new error using the supplied string value
   * @param {String} val
   * @returns {Promise}
   */
  function reject(val) {
    return Promise.reject(new Error(val));
  }

  /**
   * Unloads the kids of the supplied NavTree
   * @param {module:nmodule/webEditors/rc/wb/tree/NavTree} tree
   * @returns {Promise}
   */
  function unloadKids(tree) {
    var kidsList = tree.$getKidsList();
    var kidEditors = tree.$getKids();
    tree.value().$kidsLoaded = false;
    return kidEditors.destroyAll().then(function () {
      kidsList.empty();
    });
  }

  /**
   *
   * @param {module:nmodule/webEditors/rc/wb/tree/NavTree} tree
   * @param {Object} kid the object that represents kid NavTree to be built
   * @returns {Promise<module:nmodule/webEditors/rc/wb/tree/NavTree>}
   */
  function buildKid(tree, kid) {
    return tree.buildChildFor({
      value: kid,
      dom: $('<li/>').data('node', kid).appendTo(tree.$getKidsList()),
      type: tree.constructor,
      //TODO: be smarter about this
      properties: {
        expanded: false,
        loadKids: false,
        enableHoverPreload: tree.$enableHoverPreload,
        displayFilter: tree.getDisplayFilter()
      }
    });
  }

  /**
   * Loads the kids for the supplied NavTree
   * @param {module:nmodule/webEditors/rc/wb/tree/NavTree} tree
   * @param {baja.comm.Batch} batch
   * @returns {Promise<module:nmodule/webEditors/rc/wb/tree/NavTree>}
   */
  function loadKids(tree, batch) {
    var node = tree.value();
    return node.getKids({
      batch: batch
    }).then(function (kids) {
      return Promise.all(_.map(kids, function (kid) {
        var displayFilter = tree.getDisplayFilter();
        if (displayFilter) {
          if (displayFilter(node, kid)) {
            return buildKid(tree, kid);
          } else {
            return null;
          }
        } else {
          return buildKid(tree, kid);
        }
      })).then(function (kidTrees) {
        return _.compact(kidTrees);
      });
    });
  }

  /**
   *
   * @param {module:nmodule/webEditors/rc/wb/tree/NavTree} tree the tree that is to be compared to the selected
   * @param {module:nmodule/webEditors/rc/wb/tree/NavTree} selectedEd the tree to be selected
   */
  function setSelectedEditor(tree, selectedEd) {
    if (tree !== selectedEd) {
      tree.setSelected(false);
    }
    _.each(tree.$getKids(), function (kidEd) {
      setSelectedEditor(kidEd, selectedEd);
    });
  }

  /**
   * Returns the name of the kid NavTree
   * @param {module:nmodule/webEditors/rc/wb/tree/NavTree} kid
   * @returns {String}
   */
  function kidName(kid) {
    return kid.getName();
  }

  /**
   * Returns the value of the supplied TreeNode
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} node
   * @returns {*}
   */
  function toValue(node) {
    return node.value();
  }

  /**
   * Returns true if the type of the supplied value is a baja.iNavNode
   * @param {Object} value
   * @returns {boolean}
   */
  function isNavNode(value) {
    return baja.hasType(value, 'baja:INavNode');
  }

  /**
   * Returns true if the object is an instanceof TreeNode
   * @param {Object} value
   * @returns {boolean}
   */
  function isTreeNode(value) {
    return value instanceof TreeNode;
  }

  /**
   * Returns the nav ord of the supplied nav node
   * @param {module:baja/nav/NavNode} navNode
   * @returns {baja.Ord}
   */
  function toNavOrd(navNode) {
    return navNode.getNavOrd();
  }

  /**
   * Returns true if the tree node can be dragged
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} node
   * @returns {boolean}
   */
  function draggableOnlyFilter(node) {
    return node.isDraggable();
  }

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * Allows for displaying, expanding, and collapsing a single node in a tree.
   *
   * A single `NavTree` instance will show a node's icon and display value,
   * an expand/collapse button, and a list containing the node's children.
   *
   * The kid list will initially be collapsed, but when the `NavTree` first
   * loads, it will immediately kick off loading the children's values behind
   * the scenes. This should allow the list to be immediately expanded when the
   * user does get around to clicking the expand button to show the list. (If
   * the list is not finished loading when the user clicks, there will just
   * be a delay until it does finish.)
   *
   * Note that each individual node in the tree will receive its own `NavTree`
   * instance. With some clever overriding of `buildChildFor`, this should allow
   * for some granularity when expanding out your tree. Inline field editors?
   * Who knows.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/tree/NavTree
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @param {Object} [params] the object literal containing the constructor's arguments
   * @param {Object} [params.properties] the properties for the constructor
   * @param {Boolean} [params.properties.expanded=false] set to true if this tree should
   * be immediately expanded on first load
   * @param {Boolean} [params.properties.loadKids=true] set to false if this tree should
   * not attempt to load its children before it is expanded
   * @param {Boolean} [params.properties.enableHoverPreload=false] set to true if child
   * nodes should start to preload when the mouse hovers over the expand
   * button (to shave a bit of time off child node expansion)
   * @param {module:nmodule/webEditors/rc/wb/tree/NavTree~displayFilter} [params.properties.displayFilter]
   * @param {boolean} [params.properties.hideRoot] hides the root node and expands the root
   */
  var NavTree = function NavTree(params) {
    BaseEditor.call(this, {
      params: params,
      defaults: widgetDefaults()
    });
    var props = this.properties();
    var expanded = props.getValue('expanded');
    var loadKids = props.getValue('loadKids');
    var enableHoverPreload = props.getValue('enableHoverPreload');
    this.$expanded = expanded;
    this.$loadKids = loadKids !== false;
    this.$selected = false;
    this.$enableHoverPreload = enableHoverPreload;
    new Switchboard(this).allow('$setLoaded').oneAtATime().keyedOn(function (loaded) {
      return loaded;
    }).allow('$addKid').notWhile('$removeKid').keyedOn(kidName).allow('$removeKid').notWhile('$addKid').keyedOn(kidName).allow('$reorderKids').notWhile('$addKid').notWhile('$removeKid').allow('setSelectedPath').notWhile('$addKid');
    TransferSupport(this);
  };
  NavTree.prototype = Object.create(BaseEditor.prototype);
  NavTree.prototype.constructor = NavTree;
  NavTree.ACTIVATED_EVENT = 'navTree:activated';
  NavTree.COLLAPSED_EVENT = 'navTree:collapsed';
  NavTree.DESELECTED_EVENT = 'navTree:deselected';
  NavTree.EXPANDED_EVENT = 'navTree:expanded';
  NavTree.SELECTED_EVENT = 'navTree:selected';

  /**
   * Return the element for the expand/collapse button.
   * @private
   * @returns {jQuery}
   */
  NavTree.prototype.$getButton = function () {
    return this.jq().children('button');
  };

  /**
   * Return the element to load the node's display value.
   * @private
   * @returns {jQuery}
   */
  NavTree.prototype.$getDisplayElement = function () {
    return this.jq().children('.display').children('.displayName');
  };

  /**
   * Return the element to load the node's icon.
   * @private
   * @returns {jQuery}
   */
  NavTree.prototype.$getIconElement = function () {
    return this.jq().children('.display').children('.icon');
  };

  /**
   * Return the <ul> element that holds the list items to hold `NavTree`s for
   * the child nodes.
   * @private
   * @returns {jQuery}
   */
  NavTree.prototype.$getKidsList = function () {
    return this.jq().children('ul');
  };

  /**
   * @private
   * @returns {Array.<module:nmodule/webEditors/rc/wb/tree/NavTree>}
   */
  NavTree.prototype.$getKids = function () {
    return this.getChildEditors({
      dom: this.$getKidsList(),
      type: NavTree
    });
  };

  /**
   * Returns an array of all the visible nodes from this NavTree down recursively.
   * @private
   * @since Niagara 4.15
   * @returns {Array.<module:nmodule/webEditors/rc/wb/tree/NavTree>}
   */
  NavTree.prototype.$getAllKids = function () {
    var kids = [];
    this.$getKids().forEach(function (kid) {
      kids.push(kid);
      if (kid.$isExpanded()) {
        var grandKids = kid.$getAllKids();
        grandKids.forEach(function (grandKid) {
          kids.push(grandKid);
        });
      }
    });
    return kids;
  };

  /**
   * Given a kid node, find the `NavTree` that currently has that node loaded,
   * if any.
   * @private
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} node
   * @returns {module:nmodule/webEditors/rc/wb/tree/NavTree}
   */
  NavTree.prototype.$getKidEditorFor = function (node) {
    var kidDoms = this.$getKidsList().children();
    var ed;
    var i;
    for (i = 0; i < kidDoms.length; i++) {
      ed = $(kidDoms[i]).data('widget');
      if (ed && ed.value().equals(node)) {
        return ed;
      }
    }
  };

  /**
   * The amount of time to wait before automatically unloading children of
   * a collapsed node.
   * @private
   * @returns {number} milliseconds
   */
  NavTree.prototype.$getUnloadTimeout = function () {
    return 10000;
  };

  /**
   * Return whether the tree is currently in expanded mode or not.
   * @private
   * @returns {Boolean}
   */
  NavTree.prototype.$isExpanded = function () {
    return !!this.$expanded;
  };

  /**
   * Set the expanded status of the nav tree.
   *
   * If expanding, the child nodes of this node will be loaded if they are
   * not already, and then the kid list will be shown. After the kid list is
   * shown, kids of kids will start preloading immediately to speed up
   * subsequent drilldowns.
   *
   * If collapsing, the kid list will simply be hidden. Any kid editors will
   * not necessarily unload just from collapsing the tree.
   *
   * @private
   * @param {Boolean} expanded
   * @returns {Promise} promise to be resolved when the tree has fully
   * expanded or collapsed.
   */
  NavTree.prototype.$setExpanded = function (expanded) {
    var that = this;
    var kidsList = that.$getKidsList();
    var button = that.$getButton();
    var prom;
    var pending = true;
    var spinnerTicket;
    var event = NavTree[expanded ? 'EXPANDED_EVENT' : 'COLLAPSED_EVENT'];
    if (expanded === that.$isExpanded()) {
      return resolve();
    }
    that.$expanded = expanded;
    if (expanded) {
      spinnerTicket = setTimeout(function () {
        if (pending) {
          button.addClass('loading');
        }
      }, EXPAND_SPINNER_DELAY);
      prom = that.$setLoaded(true).then(function () {
        kidsList.show();
        if (that.$loadKids) {
          return that.$preloadGrandKids(true);
        }
      })["finally"](function () {
        button.removeClass('loading');
        pending = false;
        clearTimeout(spinnerTicket);
      });
    } else {
      kidsList.hide();
      setSelectedEditor(that, that);
      prom = resolve();
    }
    return prom.then(function () {
      that.trigger(event);
      return that.$updateButton();
    });
  };

  /**
   * Are my current children loaded and ready to expand?
   * @private
   * @returns {Boolean}
   */
  NavTree.prototype.$isLoaded = function () {
    return this.$loaded;
  };

  /**
   * Load up my children so that I can expand without delay. Or, unload them
   * to save memory. Note that if unloading kids, this tree will also be
   * collapsed (an expanded tree with zero children makes no sense). Kids
   * will be automatically re-loaded if the tree is expanded again, but
   * hopefully they will have been loaded well in advance of the expand button
   * being clicked (for quick response).
   *
   * @private
   * @param {Boolean} loaded
   * @param {baja.comm.Batch} [batch]
   * @returns {Promise} promise to be resolved when I have loaded or
   * unloaded all of my kids.
   */
  NavTree.prototype.$setLoaded = function (loaded, batch) {
    var that = this;
    var prom;

    //TODO: this is Switchboard behavior. come back and do it right
    if (loaded === that.$loaded) {
      return that.$$slProm || resolve();
    }
    if (that.$getKidsList().is(':visible')) {
      logError('warning: $setLoaded should never happen when list element ' + 'is visible');
    }
    that.$loaded = loaded;
    prom = that.$$slProm = (loaded ? loadKids(that, batch) : that.collapse().then(function () {
      return unloadKids(that);
    })).then(function () {
      delete that.$$slProm;
    });
    return prom;
  };

  /**
   * Are all of my kids in the loaded state? Or: are my grandkids loaded?
   * @private
   * @returns {Boolean}
   */
  NavTree.prototype.$isKidsLoaded = function () {
    return this.$kidsLoaded;
  };

  /**
   * Set the loaded state of all my children. Or: load or unload my grandkids.
   * This will typically be called when the tree is expanded, to try to keep
   * one step ahead of the user as he or she drills down through the nav tree.
   *
   * @private
   * @param {Boolean} loaded
   * @returns {Promise} promise to be resolved when all of my grandkids
   * are loaded or unloaded
   */
  NavTree.prototype.$preloadGrandKids = function (loaded) {
    var that = this;
    var kids = that.$getKids();
    if (that.$kidsLoaded === loaded) {
      return resolve();
    }
    that.$kidsLoaded = loaded;
    var batch = new baja.comm.Batch(),
      prom = Promise.all(_.map(kids, function (kid) {
        return kid.value().$preLoad && kid.$setLoaded(loaded, batch);
      }));
    batch.commit();
    return prom;
  };

  /**
   * Builds a new nav tree for the added node and appends it to the current list
   * of kids. Will be called when the backing TreeNode gets a new kid added
   * (the `added` event is emitted), and we must create a new tree in the DOM
   * to reflect the change. Note that the expand/collapse button will be enabled
   * if this is the first kid to be added.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} kid
   * @returns {Promise} promise to be resolved when the new editor is
   * added
   */
  NavTree.prototype.$addKid = function (kid) {
    var that = this;
    return buildKid(that, kid).then(function (kidTree) {
      that.$updateButton()["catch"](logError);
      return kidTree.$setLoaded(that.$isExpanded());
    });
  };

  /**
   * Removes the nav tree editor from the DOM when a kid node is removed. Will
   * be called when the backing TreeNode gets a kid removed (the `removed`
   * event is emitted), and we must remove the associated tree from the DOM
   * to reflect the change. Note that the expand/collapse button will be
   * disabled if this is the last kid to be removed.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} kid the removed
   * node (has already been removed from the tree when this method is called)
   * @returns {Promise} promise to be resolved when the nav tree editor
   * has been removed from the DOM
   */
  NavTree.prototype.$removeKid = function (kid) {
    var that = this;
    var kidEd = that.$getKidEditorFor(kid);
    if (kidEd) {
      var li = kidEd.jq();
      li.remove();
      return kidEd.destroy().then(function () {
        that.$updateButton()["catch"](logError);
      });
    } else {
      return resolve();
    }
  };

  /**
   * Updates the display element of the renamed kid. Note that the referenced
   * kid has already been updated in the tree; we're just updating the DOM to
   * reflect that.
   *
   * @private
   * @param {String} newName
   * @returns {Promise} promise to be resolved when the editor for the
   * renamed kid has had its display element updated
   */
  NavTree.prototype.$renameKid = function (newName) {
    var that = this;
    var node = that.value();
    return that.value().getKid(newName).then(function (kid) {
      var displayFilter = that.getDisplayFilter();
      var isDisplayed = !displayFilter || displayFilter(node, kid);
      if (isDisplayed) {
        var kidEd = that.$getKidEditorFor(kid);
        if (kidEd) {
          return kidEd.$updateDisplayAndIcon(kid);
        } else {
          return that.$addKid(kid).then(function () {
            return that.$reorderKids();
          });
        }
      } else {
        return that.$removeKid(kid);
      }
    });
  };

  /**
   * Shuffles around the order of child trees in the DOM to reflect changes in
   * nav child order.
   *
   * @private
   * @returns {Promise} promise to be resolved when the reordering is
   * complete
   */
  NavTree.prototype.$reorderKids = function () {
    var that = this;
    var node = this.value();
    var kidsList = this.$getKidsList();
    var kidDoms = kidsList.children();
    return node.getKids().then(function (kids) {
      //kids are newly ordered at this point. for each kid, find the existing
      //(out of order) DOM element for that kid, and map it into a correctly
      //ordered array of DOM elements.
      var orderedDoms = _.map(kids, function (kid) {
        var displayFilter = that.getDisplayFilter();
        var isDisplayed = !displayFilter || displayFilter(node, kid);
        if (isDisplayed) {
          for (var i = 0; i < kidDoms.length; i++) {
            if (kid.equals($(kidDoms[i]).data('widget').value())) {
              return kidDoms[i];
            }
          }
        }
      });
      orderedDoms = _.compact(orderedDoms);

      //detach the out-of-order elements and pop them back in in order.
      kidDoms.detach();
      kidsList.html($(orderedDoms));
    });
  };

  /**
   * Updates the CSS classes and disabled status of the expand/collapse button
   * to reflect whether the tree can be expanded or collapsed.
   *
   * @private
   * @returns {Promise}
   */
  NavTree.prototype.$updateButton = function () {
    var button = this.$getButton();
    var node = this.value();
    var loadKids = this.$loadKids;
    var expanded = this.$isExpanded();
    function doUpdate(disabled) {
      button.prop('disabled', disabled).toggleClass('expanded', !!expanded).toggleClass('collapsed', !expanded);
    }
    if (node && node.mayHaveKids()) {
      if (node.$kidsLoaded || loadKids) {
        return node.getKids().then(function (kids) {
          doUpdate(!kids.length);
        });
      } else {
        return Promise.resolve(doUpdate(false));
      }
    } else {
      return Promise.resolve(doUpdate(true));
    }
  };

  /**
   * Determine if a node should be displayed.
   * @callback module:nmodule/webEditors/rc/wb/tree/NavTree~displayFilter
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} parent
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} child
   * @returns {boolean} true if to be displayed, false other wise
   *
   */

  /**
   * Returns the display filter if one is registered.
   *
   * @returns {module:nmodule/webEditors/rc/wb/tree/NavTree~displayFilter}
   */
  NavTree.prototype.getDisplayFilter = function () {
    return this.properties().getValue('displayFilter');
  };

  /**
   * Return true if this nav tree is currently selected/highlighted by the user.
   *
   * @returns {Boolean}
   */
  NavTree.prototype.isSelected = function () {
    return this.$selected;
  };

  /**
   * Sets the nav tree's selected status. A second modified parameter allows
   * you to specify whether we are modifying an existing selection (e.g. by
   * holding the CTRL key) or making a brand new selection. The event triggered
   * will pass this as a parameter to the event handler.
   *
   * If setting selected to `false`, and the `NavTree` is already unselected,
   * no DOM change will occur and no event will be fired (why should one care?)
   * However setting selected to `true` when the node is already selected, it
   * will still fire an event. (Consider when you have several nodes selected
   * with `Ctrl` and click on one of the nodes already selected: we need to
   * fire the selected event so that we know to clear the rest of the
   * selection.)
   *
   * @param {Boolean} selected
   * @param {object} [params]
   * @param {Boolean} [params.modified] set to true if this should modify an
   * existing selection instead of starting a new one.
   * @param {Boolean} [params.silent] set to true if this should not fire a
   * `SELECTED_EVENT`.
   *
   */
  NavTree.prototype.setSelected = function (selected, params) {
    var that = this;
    var jq = that.jq();
    var eventName = selected ? NavTree.SELECTED_EVENT : NavTree.DESELECTED_EVENT;
    var modified = params && params.modified;
    var silent = params && params.silent;
    if (!selected && !that.$selected) {
      return;
    }
    that.$selected = selected;
    jq.toggleClass('selected', selected);
    if (!silent) {
      jq.trigger(eventName, [that, !!modified]);
      if (selected) {
        htmlUtils.suggestFocus(jq.children('.display'));
      }
    }
  };

  /**
   * Indicates that this node has been "activated" by the user, say by double-
   * clicking it. In contrast with "selected", say by a single click. Triggers
   * `NavTree.ACTIVATED_EVENT`.
   */
  NavTree.prototype.activate = function () {
    this.trigger(NavTree.ACTIVATED_EVENT);
  };

  /**
   * Performs a depth-first search through the nav tree and assembles all
   * selected nodes into an array.
   *
   * @returns {Array.<module:nmodule/webEditors/rc/wb/tree/TreeNode>} an array
   * of selected nodes
   */
  NavTree.prototype.getSelectedNodes = function () {
    var selectedNodes = this.isSelected() ? [this.value()] : [];
    _.each(this.$getKids(), function (kidEd) {
      var selectedKids = kidEd.getSelectedNodes();
      if (selectedKids) {
        selectedNodes = selectedNodes.concat(selectedKids);
      }
    });
    return selectedNodes;
  };

  /**
   * Will clear all selection for all nodes in the navTree.
   * @since Niagara 4.15
   * @param {Object} [params]
   * @param {boolean} [params.silent] Set this flag to true, to not fire `DESELECTED_EVENT`.
   */
  NavTree.prototype.clearSelection = function () {
    var params = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    this.$getKids().forEach(function (kid) {
      if (kid.isSelected()) {
        kid.setSelected(false, params);
      }
      kid.clearSelection(params);
    });
  };

  /**
   * Get an array of nodes that are currently visible, i.e., expanded, within
   * this tree. If the tree is collapsed, only the one currently loaded node
   * will be resolved.
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/wb/tree/TreeNode>>}
   */
  NavTree.prototype.getVisibleNodes = function () {
    var nodes = [this.value()];
    if (!this.$isExpanded()) {
      return Promise.resolve(nodes);
    } else {
      return Promise.all(_.map(this.$getKids(), function (kid) {
        return kid.getVisibleNodes();
      })).then(function (nodeArrays) {
        return nodes.concat(_.flatten(nodeArrays));
      });
    }
  };

  /**
   * Implementation for `NavMonitorSupport` - get the nav ORDs of all visible
   * tree nodes.
   * @see module:nmodule/webEditors/rc/wb/mixin/NavMonitorSupport
   * @returns {Promise.<Array.<baja.Ord>>}
   */
  NavTree.prototype.getTouchableOrds = function () {
    return this.getVisibleNodes().then(function (nodes) {
      return _.chain(nodes).map(toValue).filter(isNavNode).map(toNavOrd).value();
    });
  };

  /**
   * Expands out the tree, recursing down through the sub-nodes until the
   * specified node is found and selected, or selecting the parent node when
   * reaching an unrecognized node.
   *
   * The given path should be an array of node names. The first name in the path
   * should match the name of the node loaded into this tree, and the second
   * name in the path (if provided) should match one of the child tree nodes. It
   * will then recurse down until a match is found, or an unrecognized node is
   * reached. Each referenced tree node will have all of its child nodes loaded.
   *
   * @param {Array.<String>} path - the path of node names to expand and select
   * @param {Object} [params]
   * @param {Boolean} [params.modified] set to true if this should modify an existing selection instead of starting a new one.
   * @param {Boolean} [params.silent] set to true if this should not fire a `SELECTED_EVENT`.
   * @param {Boolean} [params.scrollIntoView=true] set to false if you want to leave the scrollbar where it is.
   * @param {Boolean} [params.expandOnly=false] set to true if this should not modify the selection, but just expand to the desired element.
   * @param {Boolean} [params.expandLastNode=false] set to true if the last node should be expanded.
   *
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/tree/NavTree>} promise to be resolved with
   * the NavTree instance after it has been found and tree nodes leading to it have been loaded and
   * expanded. To be rejected if the path does not resolve to a NavTree instance.
   */
  NavTree.prototype.setSelectedPath = function (path) {
    var _this = this;
    var params = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    var scrollIntoView = params.scrollIntoView !== false;
    var modified = params.modified,
      expandOnly = params.expandOnly,
      expandLastNode = params.expandLastNode;
    return this.traverseTo(path).then(function (_ref) {
      var tree = _ref.tree,
        success = _ref.success;
      if (tree) {
        if (!expandOnly) {
          tree.setSelected(true, params);
          if (!modified) {
            setSelectedEditor(_this.getRoot(), tree);
          }
        }
        if (scrollIntoView) {
          tree.jq()[0].scrollIntoView({
            block: "nearest",
            inline: "nearest"
          });
        }
      }
      if (!success) {
        throw new Error(path.join() + ' not found');
      }
      return Promise.resolve(expandLastNode && tree.expand()).then(function () {
        return tree;
      });
    });
  };

  /**
   * Traverse the NavTree to the specified path, loading all nodes along the way. If the path does
   * not resolve all the way to the final NavTree, it will still expand and load as far as it can.
   *
   * Unlike `setSelectedPath`, this will not set any nodes selected along the way. It just drills
   * down through the tree until the corresponding node is found.
   *
   * Resolves with `tree` and `success` values, where `tree` is the NavTree instance we successfully
   * traversed to. If the path does not resolve all the way, `success` will be false and `tree` will
   * be the last tree in the path that was found. If `path` is empty, this is considered as "don't
   * traverse anywhere" - `tree` will be undefined, and `success` will be true.
   *
   * @since Niagara 4.15
   *
   * @returns {Promise.<{ tree: module:nmodule/webEditors/rc/wb/tree/NavTree, success: boolean }>} promise
   * to be resolved with the NavTree instance after it has been found and tree nodes leading to it
   * have been loaded and expanded.
   */
  NavTree.prototype.traverseTo = function (path) {
    var _this2 = this;
    if (!_.isArray(path)) {
      return reject('Array required');
    }
    if (!path.length) {
      return resolve({
        tree: undefined,
        success: true
      });
    }
    var restOfPath = path.slice();
    var head = restOfPath.shift();
    var next = restOfPath[0];
    var node = this.value();
    if (node.getName() !== head) {
      return reject(); // Node not found
    }
    if (!next) {
      //no more kids to check. it's me!
      return Promise.resolve({
        tree: this,
        success: true
      });
    }
    return this.$setLoaded(true).then(function () {
      return node.getKid(next);
    }).then(function (kid) {
      var kidEd = _this2.$getKidEditorFor(kid);
      if (!kidEd) {
        return resolve({
          tree: _this2,
          success: false
        });
      }
      return kidEd.traverseTo(restOfPath).then(function (result) {
        return _this2.expand().then(function () {
          return result;
        });
      });
    })
    // eslint-disable-next-line promise/no-return-in-finally
    ["finally"](function () {
      return _this2.expand();
    });
  };

  /**
   * Collapses the tree.
   *
   * @returns {Promise}
   */
  NavTree.prototype.collapse = function () {
    return this.$setExpanded(false);
  };

  /**
   * Expands the tree.
   *
   * @returns {Promise}
   */
  NavTree.prototype.expand = function () {
    return this.$setExpanded(true);
  };

  /**
   * Adds event listeners to the backing `TreeNode`. Whenever the node has its
   * children renamed, reordered, added to or removed from, the corresponding
   * method on the `NavTree` will be called to keep the DOM up to date.
   *
   * This will automatically be called as soon as the node is loaded into the
   * `NavTree`.
   *
   * @see module:nmodule/webEditors/rc/wb/tree/NavTree#$addKid
   * @see module:nmodule/webEditors/rc/wb/tree/NavTree#$removeKid
   * @see module:nmodule/webEditors/rc/wb/tree/NavTree#$renameKid
   * @see module:nmodule/webEditors/rc/wb/tree/NavTree#$reorderKids
   */
  NavTree.prototype.subscribe = function () {
    var that = this;
    var node = that.value();
    node.on('added', that.$addedHandler = function (kid) {
      if (!that.$loaded) {
        return;
      }
      var displayFilter = that.getDisplayFilter();
      var isDisplayed = !displayFilter || displayFilter(node, kid);
      if (isDisplayed) {
        that.$addKid(kid)["catch"](logError);
      }
    }).on('removed', that.$removedHandler = function (kid) {
      if (!that.$loaded) {
        return;
      }
      that.$removeKid(kid)["catch"](logError);
    }).on('renamed', that.$renamedHandler = function (newName, oldName) {
      if (!that.$loaded) {
        return;
      }
      that.$renameKid(newName, oldName)["catch"](logError);
    }).on('reordered', that.$reorderedHandler = function () {
      if (!that.$loaded) {
        return;
      }
      that.$reorderKids(node)["catch"](logError);
    });
  };

  /**
   * Removes event listeners added in `subscribe()`. Will be called when the
   * `NavTree` is destroyed, or when an old node is unloaded.
   */
  NavTree.prototype.unsubscribe = function () {
    var that = this;
    var node = that.value();
    if (node) {
      node.removeListener('added', that.$addedHandler).removeListener('removed', that.$removedHandler).removeListener('renamed', that.$renamedHandler).removeListener('reordered', that.$reorderedHandler);
    }
  };

  /**
   * @since Niagara 4.15
   * @returns {module:nmodule/webEditors/rc/wb/tree/NavTree|undefined}
   */
  NavTree.prototype.getParent = function () {
    var parentWidget = Widget["in"](this.jq().parent().parent());
    if (parentWidget instanceof NavTree) {
      return parentWidget;
    }
  };

  /**
   * Gets the root of the supplied tree
   * @since Niagara 4.15
   * @returns {module:nmodule/webEditors/rc/wb/tree/NavTree}
   */
  NavTree.prototype.getRoot = function () {
    var parent = this.getParent();
    return parent instanceof NavTree ? parent.getRoot() : this;
  };

  /**
   * Sets up initial HTML for the nav tree node.
   * @param {JQuery} dom
   */
  NavTree.prototype.doInitialize = function (dom) {
    var _this3 = this;
    var that = this;
    var enableHoverPreload = that.$enableHoverPreload;
    var preloadTicket;
    var touchdown = false;
    var mouseupCancelled = false;
    dom.html(tplNavTree()).addClass('NavTree').toggleClass('hideRoot', !!that.properties().getValue('hideRoot'));
    dom.on([DESTROY_EVENT, LOAD_EVENT].join(' '), '.editor', false);
    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });

    //new tree node selected by the user.
    dom.on(NavTree.SELECTED_EVENT, function (e, ed, modified) {
      if (!modified) {
        //we're not just modifying an existing selection - so go up to the root
        //and wipe out all existing selections except for the one we just made.
        setSelectedEditor(_this3.getRoot(), ed);
      }
    });
    function cancelMouseup() {
      function uncancel() {
        mouseupCancelled = false;
        $(document).off('mouseup touchend', uncancel);
      }
      if (!mouseupCancelled) {
        $(document).on('mouseup touchend', uncancel);
      }
      mouseupCancelled = true;
    }
    function handleMouseSelection(e) {
      if (!that.value().isSelectable()) {
        return;
      }
      var isToggle = getSelectionModeFromEvent(e, false) === TOGGLE_MODE;
      var isSwath = getSelectionModeFromEvent(e, true) === SWATH_MODE;
      var which = e.which;
      var root = that.getRoot();
      var selected = root.getSelectedNodes();
      root.$isToggle = isToggle;
      if (which === RIGHT_MOUSE_BUTTON && that.isSelected()) {
        return;
      }
      if (which === RIGHT_MOUSE_BUTTON) {
        that.setSelected(true, {
          modified: isToggle
        });
        return;
      }
      if (isSwath && selected.length > 0 && !isToggle) {
        handleShiftSelection();
      } else {
        root.$mouseupUnSelect = that.isSelected();
        that.setSelected(isToggle ? !that.isSelected() : true, {
          modified: isToggle || that.isSelected()
        });
        if (!isToggle) {
          delete root.$prevShiftStart;
        }
      }
    }

    /**
     * Selects the correct nodes on a shift select
     * @since Niagara 4.15
     */
    function handleShiftSelection() {
      var root = that.getRoot();
      var prevShiftStart = root.$prevShiftStart;
      var allKids = root.$getAllKids();
      var select = false;
      allKids = [root].concat(_toConsumableArray(allKids));
      var selectedStart = allKids.contains(prevShiftStart) ? prevShiftStart : allKids.filter(function (kid) {
        return kid.isSelected();
      })[0];
      allKids.forEach(function (kid) {
        kid.setSelected(false, {
          modified: true
        });
      });
      root.$prevShiftStart = selectedStart;
      allKids.forEach(function (kid) {
        if (kid === that || kid === selectedStart) {
          select = selectedStart !== that ? !select : false;
          kid.setSelected(true, {
            modified: true
          });
        } else if (select) {
          kid.setSelected(true, {
            modified: true
          });
        }
      });
    }
    contextMenuOnLongPress(dom.children('.display'));
    dom.children('.display').on('touchstart', function (e) {
      var touches = e.originalEvent.touches;
      if (touches.length > 1) {
        //don't select/activate on a pinch zoom
        cancelMouseup();
      } else {
        touchdown = true;
      }
    }).on('touchmove', function () {
      cancelMouseup();
    }).on('mousedown contextmenu', function (e) {
      var isToggle = getSelectionModeFromEvent(e, false) === TOGGLE_MODE;
      var noSelected = !isToggle ? _this3.getRoot().getSelectedNodes().length : 0;
      if (!(that.isSelected() && noSelected <= 1 && !isToggle) && !mouseupCancelled) {
        handleMouseSelection(e);
        cancelMouseup();
      }
    }).on('mouseup', function (e) {
      var root = that.getRoot();
      var mouseupUnSelect = root.$mouseupUnSelect;
      var isToggle = root.$isToggle;
      delete root.$mouseupUnSelect;
      delete root.$isToggle;
      if (mouseupUnSelect) {
        that.setSelected(!isToggle, {
          modified: false
        });
      }
    }).on('touchend', function (e) {
      var multitouch = e.originalEvent.touches.length > 1;
      if (!mouseupCancelled && !multitouch && touchdown) {
        handleMouseSelection(e);
        if (getSelectionModeFromEvent(e) === SELECT_MODE) {
          that.trigger(NavTree.ACTIVATED_EVENT, that);
        }
        touchdown = false;
        return false;
      }
    }).on('dblclick', function (e) {
      if (getSelectionModeFromEvent(e, false) === TOGGLE_MODE) {
        return;
      }
      if (that.value().isSelectable()) {
        that.trigger(NavTree.ACTIVATED_EVENT, that);
        return false;
      }
    }).on('dragstart', function (e) {
      var isToggleMode = getSelectionModeFromEvent(e, false) === TOGGLE_MODE;
      if (touchdown) {
        that.setSelected(true, {
          modified: isToggleMode
        });
      }
      if (isToggleMode && (!e.changedTouches || !e.changedTouches.length)) {
        that.setSelected(true, {
          modified: isToggleMode
        });
      }
      var node = that.value();
      var values = _.map(_this3.getRoot().getSelectedNodes().filter(draggableOnlyFilter), function (node) {
        return node.value();
      });
      if (node.isDraggable()) {
        var clipboard = e.originalEvent.dataTransfer,
          mime = 'niagara/navnodes';
        dragDropUtils.toClipboard(clipboard, mime, values)["catch"](logError);
      }
    }).on('dragover dragenter', function (e) {
      if (that.value().isDropTarget()) {
        dom.children('.display').addClass(DROP_TARGET_CLASS);
        e.preventDefault();
      }
    }).on('dragend dragleave drop', function () {
      if (that.value().isDropTarget()) {
        dom.children('.display').removeClass(DROP_TARGET_CLASS);
      }
    }).on('drop', function (e) {
      var node = that.value();
      if (node.isDropTarget()) {
        var clipboard = e.originalEvent.dataTransfer;
        dragDropUtils.fromClipboard(clipboard).then(function (envelope) {
          return envelope.toValues();
        }).then(function (values) {
          return node.doDrop(values);
        })["catch"](feDialogs.error);
        return false;
      }
    });
    dom.children('button').on('click', function () {
      var p;
      if (that.$isExpanded()) {
        that.$disableTicket = setTimeout(function () {
          that.$unloadGrandKids()["catch"](logError);
        }, that.$getUnloadTimeout());
        p = that.collapse();
      } else {
        clearTimeout(that.$disableTicket);
        p = that.expand();
      }
      p["catch"](logError);
    });
    if (enableHoverPreload) {
      dom.children('button').on('mouseenter', function () {
        preloadTicket = setTimeout(function () {
          that.$setLoaded(true)["catch"](logError);
        }, HOVER_PRELOAD_DELAY);
      }).on('mouseleave', function () {
        clearTimeout(preloadTicket);
      });
    }
  };

  /**
   * @since Niagara 4.15
   * @private
   * @returns {Promise}
   */
  NavTree.prototype.$unloadGrandKids = function () {
    return this.$preloadGrandKids(false);
  };

  /**
   * Loads in a new `TreeNode`.
   *
   * *Important*: This node will be destroyed when this editor is destroyed.
   * Don't pass in `TreeNode`s you want to hold onto after you are finished
   * with the `NavTree`.
   *
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} value
   * @returns {Promise}
   */
  NavTree.prototype.doLoad = function (value) {
    var _this4 = this;
    if (!isTreeNode(value)) {
      return reject('TreeNode required');
    }
    if (value.isDestroyed()) {
      return reject('node already destroyed');
    }
    var that = this;
    var oldNode = that.$oldNode;
    if (value === oldNode) {
      return Promise.resolve();
    }
    that.$oldNode = value;
    var dom = that.jq();
    that.unsubscribe();
    that.$updateButton()["catch"](logError);
    that.subscribe();
    this.$loaded = false;
    return this.$getKids().destroyAll().then(function () {
      return oldNode && oldNode.destroy();
    }).then(function () {
      dom.children('.spacer').remove();
      dom.prepend(_.map(value.getFullPath().slice(1), function () {
        return '<div class="spacer"></div>';
      }).join(''));
      return Promise.all([value.activate(), _this4.$updateDisplayAndIcon(value)]);
    }).then(function () {
      if (that.$loadKids) {
        return that.$setLoaded(true);
      }
    }).then(function () {
      dom.children('.display').prop('draggable', !!value.isDraggable());
      var toExpand = false;
      var hideRoot = !!that.properties().getValue('hideRoot');
      if (that.$expanded || hideRoot) {
        that.$expanded = false;
        toExpand = true;
      }
      return that.$setExpanded(toExpand);
    });
  };

  /**
   * @param {boolean} readonly
   * @returns {Promise}
   * @since Niagara 4.15
   */
  NavTree.prototype.doReadonly = function (readonly) {
    return Promise.all(this.$getKids().map(function (kid) {
      return kid.setReadonly(readonly);
    }));
  };

  /**
   * @param {boolean} enabled
   * @returns {Promise}
   * @since Niagara 4.15
   */
  NavTree.prototype.doEnabled = function (enabled) {
    return Promise.all(this.$getKids().map(function (kid) {
      return kid.setEnabled(enabled);
    }));
  };

  /**
   * @private
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} treeNode
   * @returns {Promise} to be resolved after the display and icon elements have the correct values
   * for this TreeNode
   */
  NavTree.prototype.$updateDisplayAndIcon = function (treeNode) {
    var _this5 = this;
    return Promise.all([treeNode.toDisplay().then(function (display) {
      return _this5.$getDisplayElement().text(display);
    }), fe.buildFor({
      dom: this.$getIconElement(),
      value: treeNode.getIcon(),
      type: IconEditor
    })]);
  };

  /**
   * @returns {Array.<*>} the values of the selected nodes
   */
  NavTree.prototype.getSubject = function () {
    return _.map(this.getSelectedNodes(), function (node) {
      return node.value();
    });
  };

  /**
   * Removes the `NavTree` CSS class, unsubscribes for events from the
   * backing `TreeNode`, and destroys all child editors. Essentially, wipes
   * out everything from here down.
   * @returns {*}
   */
  NavTree.prototype.doDestroy = function () {
    var that = this;
    var dom = that.jq();
    var node = that.value();
    dom.removeClass('NavTree');
    dom.children('.expand, .display').off();
    that.unsubscribe();
    return that.getChildWidgets().destroyAll().then(function () {
      return node && node.destroy();
    });
  };
  return NavTree;
});
