/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/tree/BajaComplexTreeNode
 */
define(['baja!', 'lex!baja', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/fe/baja/util/slotUtils', 'nmodule/webEditors/rc/wb/tree/BajaSlotTreeNode', 'nmodule/webEditors/rc/wb/tree/TreeNode'], function (baja, lexs, _, compUtils, slotUtils, BajaSlotTreeNode, TreeNode) {
  'use strict';

  var isNotHidden = _.negate(slotUtils.isHidden);
  var getDisplayName = compUtils.getDisplayName;
  function isComplex(v) {
    return baja.hasType(v, 'baja:Complex');
  }
  function getName(complex) {
    var name = complex.getName();
    if (!name && baja.hasType(complex, 'baja:Station')) {
      var ord = complex.getNavOrd();
      if (ord) {
        name = ord.relativizeToSession();
      }
    }

    //use this for when loading an unmounted component in
    return String(name || complex.getType().getTypeName());
  }

  /**
   * TreeNode implementation that can handle instances of `baja:Complex`.
   *
   * This node will load one child node per slot contained in the `Complex`.
   * Each child node will be another `BajaComplexTreeNode` (if the slot is a
   * `Complex`) or a vanilla `TreeNode` for a `Simple` property, action, or
   * topic.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/tree/BajaComplexTreeNode
   * @param {baja.Complex} comp
   */
  var BajaComplexTreeNode = function BajaComplexTreeNode(comp) {
    if (!isComplex(comp)) {
      throw new Error('Complex required');
    }
    this.$complex = comp;
    this.$icon = comp.getType().getIcon().getImageUris();
    TreeNode.call(this, getName(comp), getDisplayName(comp));
  };
  BajaComplexTreeNode.prototype = Object.create(TreeNode.prototype);
  BajaComplexTreeNode.prototype.constructor = BajaComplexTreeNode;

  /**
   * Returns the backing `Complex` for this node.
   *
   * @returns {baja.Complex}
   */
  BajaComplexTreeNode.prototype.value = function () {
    return this.$complex;
  };

  /**
   * Returns the icon for the backing `Complex`'s type.
   *
   * @returns {Array.<String>}
   */
  BajaComplexTreeNode.prototype.getIcon = function () {
    return this.$icon;
  };

  /**
   * Loads slots on the `Complex` (if not already loaded) and instantiates a
   * child node for each slot.
   *
   * @param {Object} [params]
   * @param {baja.comm.Batch} [params.batch] a batch to use for the loadSlots
   * operation
   * @returns {Promise} promise to be resolved with an array of child
   * nodes
   */
  BajaComplexTreeNode.prototype.$loadKids = function (params) {
    var that = this,
      complex = that.value(),
      progressCallback = params && params.progressCallback || _.noop;
    var prom = complex.loadSlots({
      batch: params && params.batch
    }).then(function () {
      var slots = complex.getSlots().filter(isNotHidden).toArray();
      return _.map(slots, function (slot) {
        if (slot.isProperty() && slot.getType().isComplex()) {
          //TODO: this feels slightly goofy? refactor to makeChild method? or?
          return new that.constructor(complex.get(slot));
        } else {
          return new BajaSlotTreeNode(complex, slot);
        }
      });
    });
    progressCallback('commitReady');
    return prom;
  };

  //TODO: subscribe? attach events?

  return BajaComplexTreeNode;
});
