/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/mixin/ProposalSupport
 */
define(['bajaux/mixin/batchSaveMixin', 'bajaux/mixin/mixinUtils', 'Promise', 'nmodule/webEditors/rc/fe/baja/util/compUtils'], function (batchSaveMixin, mixinUtils, Promise, compUtils) {
  'use strict';

  var writeSlot = compUtils.writeSlot,
    applyMixin = mixinUtils.applyMixin,
    COMMIT_READY = batchSaveMixin.COMMIT_READY,
    MODIFIED_FROM_PROPOSAL = {},
    MIXIN_NAME = 'proposal',
    PROPOSED_VALUE_NAME = '$proposedValue';

  /**
   * This adds a "proposal" workflow to an editor that bypasses user interaction
   * and allows the user-entered value to be set programmatically.
   *
   * The use case envisioned is for PopOutCommand, which allows the detailed
   * editing of an object such as a Facets in a Property Sheet row. The user
   * click the popout button and makes changes to a new copy of the current
   * Facets value. Once the editing is complete, the newly-entered Facets object
   * is then "proposed" onto the inline editor in the Property Sheet row. No
   * changes will be committed at this time, but when that inline editor is
   * saved, the proposed changes will be applied regardless of the editor's
   * doSave behavior.
   *
   * @mixin
   * @extends module:bajaux/Widget
   * @exports nmodule/webEditors/rc/wb/mixin/ProposalSupport
   */
  var exports = {};

  /**
   * @returns {*|undefined} the proposed value, or undefined if none has been
   * proposed yet
   */
  exports.getProposedValue = function () {
    var props = this.properties();
    if (props.has(PROPOSED_VALUE_NAME)) {
      return props.getValue(PROPOSED_VALUE_NAME);
    }
  };

  /**
   * Propose a new value onto the editor. After the value is proposed, the
   * following changes occur:
   *
   * - `value()` will be unchanged
   * - `doRead` will resolve the proposed value
   * - `save()` (if this is a ComplexSlotEditor) will apply the proposed changes
   *   to the edited slot
   *
   * @param {*} proposedValue
   * @returns {Promise}
   */
  exports.propose = function (proposedValue) {
    var that = this;
    return Promise["try"](function () {
      return that.doLoad(proposedValue);
    }).then(function () {
      setProposedValue(that, proposedValue);
      return that.setModified(MODIFIED_FROM_PROPOSAL);
    });
  };
  function addProposalSupport(target) {
    if (!applyMixin(target, MIXIN_NAME, exports)) {
      return;
    }
    var setModified = target.setModified,
      doRead = target.doRead,
      load = target.load,
      saveToComplex = target.saveToComplex;
    target.setModified = function (modified) {
      if (modified && modified !== MODIFIED_FROM_PROPOSAL) {
        clearProposedValue(this);
      }
      return setModified.apply(this, arguments);
    };
    target.load = function () {
      clearProposedValue(this);
      return load.apply(this, arguments);
    };
    target.doRead = function () {
      var proposed = this.getProposedValue();
      return proposed !== undefined ? proposed : doRead.apply(this, arguments);
    };
    if (target.saveToComplex) {
      target.saveToComplex = function (readValue, params) {
        var proposed = this.getProposedValue(),
          progressCallback = params && params.progressCallback,
          promise;
        var complex = this.getComplex();
        var slot = complex && this.getSlot();
        if (proposed !== undefined && complex && slot) {
          promise = writeSlot(complex, slot, proposed, params);
          if (progressCallback) {
            progressCallback(COMMIT_READY);
          }
        } else {
          promise = saveToComplex.apply(this, arguments);
        }
        return promise;
      };
    }
  }
  function clearProposedValue(widget) {
    widget.properties().remove(PROPOSED_VALUE_NAME);
  }
  function setProposedValue(widget, proposedValue) {
    widget.properties().add({
      name: PROPOSED_VALUE_NAME,
      value: proposedValue,
      hidden: true,
      "transient": true
    });
  }
  return addProposalSupport;
});
