/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/mixin/ContextMenuSupport
 */
define(['bajaux/mixin/mixinUtils', 'jquery', 'Promise', 'nmodule/js/rc/asyncUtils/asyncUtils', 'nmodule/webEditors/rc/wb/menu/baseMenuAgentSet', 'nmodule/webEditors/rc/wb/menu/menuUtils'], function (mixinUtils, $, Promise, asyncUtils, baseMenuAgentSet, menuUtils) {
  'use strict';

  var applyMixin = mixinUtils.applyMixin;
  var forSubject = menuUtils.forSubject;
  var doRequire = asyncUtils.doRequire;
  var DEFAULT_MENU_ID = 'nmodule/webEditors/rc/wb/menu/CommandGroupContextMenu';
  var MIXIN_NAME = 'contextMenuSupport';
  function getSubject(w, e) {
    if (typeof w.getSubject === 'function') {
      return w.getSubject($(e.target));
    }
    return [w.value()];
  }
  var exports = {
    $toContextMenu: function $toContextMenu(e) {
      var _this = this;
      if (typeof this.toContextMenuCommandGroup === 'function') {
        return Promise.resolve(this.toContextMenuCommandGroup(e)).then(function (newCommandGroup) {
          if (!newCommandGroup) {
            return _this.$toContextMenuBase(e);
          }
          return newCommandGroup;
        });
      }
      return this.$toContextMenuBase(e);
    },
    $toContextMenuBase: function $toContextMenuBase(e) {
      var _this2 = this;
      return Promise.resolve(getSubject(this, e)).then(function (subject) {
        if (!subject) {
          return;
        }
        return forSubject(_this2, subject).then(function (group) {
          if (!group) {
            throw new Error('no commands apply to entire subject');
          }
          if (typeof _this2.updateMenuCommandGroup === 'function') {
            return _this2.updateMenuCommandGroup(group, subject);
          }
          return group;
        });
      });
    }
  };

  /**
   * This mixin adds support for context menus to a bajaux Widget.
   *
   * The target widget may optionally implement any of these functions:
   *
   * - `getContextMenuSelector()`: return a CSS selector for elements from which
   *   to listen for right click events. Defaults to `.contextMenu`.
   * - `getSubject(jq)` - this function will receive the jQuery element that is
   *   being right-clicked/contextmenu-ed. It should return an array consisting
   *   of the baja Values that would be represented by this widget, given a
   *   right click on that element. For instance, a right click on a `tr` in a
   *   Table would probably represent the Value stored at that row - or if
   *   multiple rows were ctrl-clicked, multiple values could be returned. If
   *   not implemented, the value `load()`ed into the Widget is always
   *   considered to be the subject.
   * - 'updateMenuCommandGroup(group, subject)' - this function is used to add additional
   *    commands to the context menu command group.
   * - 'toContextMenuCommandGroup(event)' - this function bypasses the standard method of creating the
   *    context menu and creates a new command group and commands.  The event that is being passed
   *    into can be used to determine the target and from that the command group to be used to build
   *    the menu.  If no command group is return, then the default command group will be used.  The
   *    command group returned from this is attached to the context menu automatically.  An empty
   *    command group can be returned if no context menu is desired.
   * - 'supportsContextMenuAccelerators' - Defaults to `false`, this function will enable checking for accelerators when
   *    the element which is listening for right clicks is in focus.
   * - 'contextMenuReadyOnMouseDown' - Defaults to `true`, when accelerators are supported, this function can be added to
   *    return `false` to avoid preparing the accelerators on mousedown. When the context menu commands are ready for reviewing the
   *    accelerator commands, fire a `webeditors:updatecontextmenu` event on the element which is listening for right clicks.
   *
   * @alias module:nmodule/webEditors/rc/wb/mixin/ContextMenuSupport
   * @param {module:bajaux/Widget} target
   * @param {object} params
   * @param {string} [params.menuModuleId=nmodule/webEditors/rc/wb/menu/CommandGroupContextMenu] a
   * custom subclass of `CommandGroupContextMenu` can be used instead if
   * desired.
   */
  function addContextMenuSupport(target, params) {
    if (!applyMixin(target, MIXIN_NAME, exports)) {
      return;
    }
    var menuId = params && params.menuModuleId || DEFAULT_MENU_ID;
    var _doInitialize = target.doInitialize;
    var _doDestroy = target.doDestroy;
    var menu;
    target.doInitialize = function (dom) {
      var _this3 = this;
      return Promise.resolve(_doInitialize.apply(this, arguments)).then(function () {
        return doRequire(menuId);
      }).then(function (Menu) {
        menu = new Menu(_this3);
        menu.arm();
        _this3.getContextMenu = function () {
          return menu;
        };
      });
    };
    target.doDestroy = function () {
      return Promise.all([_doDestroy.apply(this, arguments), typeof menu.destroy === 'function' && menu.destroy()]);
    };
  }

  /**
   * List of all ContextMenuSupport events.
   *
   * @alias module:nmodule/webEditors/rc/wb/mixin/ContextMenuSupport~events
   * @since Niagara 4.15
   */
  addContextMenuSupport.events = {
    /** When accelerators are supported, this event can be triggered to notify ContextMenuSupport that the list of commands
     *  is ready to process for accelerators.
     */
    UPDATE_CONTEXT_MENU: 'webeditors:updatecontextmenu'
  };
  return addContextMenuSupport;
});
