/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/mixin/AttachableSupport
 */
define(['bajaux/mixin/mixinUtils', 'Promise', 'nmodule/webEditors/rc/fe/baja/util/Attachable', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (mixinUtils, Promise, Attachable, typeUtils) {
  'use strict';

  var applyMixin = mixinUtils.applyMixin,
    isComponent = typeUtils.isComponent,
    MIXIN_NAME = 'attachable';
  function destroyOldAttachable(obj) {
    var oldAtt = obj.$attachable;
    if (oldAtt) {
      oldAtt.detach();
      delete obj.$attachable;
    }
  }

  /**
   * When a widget needs to load different mounted/subscribed components, it
   * will need to attach Baja event handlers to that component. However it is
   * very important to detach all event handlers when the widget is destroyed
   * or the component is unloaded.
   *
   * This mixin provides a way to manage that workflow; by providing a single
   * point of entry to attach Baja event handlers, those handlers can be
   * tracked and automatically detached when the component is unloaded or
   * the widget is destroyed.
   *
   * The widget receiving the mixin should implement a function named `attach`
   * that will accept an `Attachable` instance and attach handlers to that.
   * The `Attachable` will be cleaned up automatically.
   *
   * @alias module:nmodule/webEditors/rc/wb/mixin/AttachableSupport
   * @param {module:bajaux/Widget} target
   *
   * @example
   *   function MyWidget() {
   *     Widget.apply(this, arguments);
   *     addAttachableSupport(this);
   *   }
   *
   *   //apply the subscriberMixIn - unsubscribed components don't fire
   *   //Baja events.
   *   MyWidget = subscribable(MyWidget);
   *
   *   //...
   *
   *   MyWidget.prototype.attach = function (attachable) {
   *     var that = this;
   *     attachable.attach({
   *       flagsChanged: function (prop) {
   *         that.handleNewFlags(this.getFlags(prop));
   *       }
   *     });
   *   };
   */
  var addAttachableSupport = function addAttachableSupport(target) {
    if (!applyMixin(target, MIXIN_NAME)) {
      return;
    }
    var _doLoad = target.doLoad,
      _doDestroy = target.doDestroy;
    target.doLoad = function (value) {
      var that = this;
      destroyOldAttachable(that);
      return Promise.resolve(_doLoad.apply(that, arguments)).then(function () {
        if (isComponent(value) && typeof that.attach === 'function') {
          var att = new Attachable(value);
          that.attach(att);
          that.$attachable = att;
        }
      });
    };
    target.doDestroy = function () {
      var that = this;
      destroyOldAttachable(that);
      return _doDestroy.apply(that, arguments);
    };
  };
  return addAttachableSupport;
});
