/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/mgr/model/folderComponentSourceMixin
 */
define(['baja!', 'bajaux/mixin/mixinUtils', 'Promise', 'underscore', 'nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource'], function (baja, mixinUtils, Promise, _, ContainerComponentSource) {
  'use strict';

  var applyMixin = mixinUtils.applyMixin,
    MIXIN_NAME = 'FOLDER_SOURCE';

  /**
   * Return an array of children of the parent object, filtered according to the
   * supplied predicate. This will be the filter function supplied to the component
   * source this module is mixed into.
   *
   * @inner
   * @param parent The parent object.
   * @param filter A predicate function used to filter out the required instances.
   *
   * @returns {Array.<baja.Value>}
   */
  function getDirectChildren(parent, filter) {
    return parent.getSlots().properties().is('baja:Component').filter(function (slot) {
      return filter(slot, this.get(slot));
    }).toValueArray();
  }

  /**
   * Function to take a root component and return an array of it's children and the
   * children of its folders, recursively down to the leaves of the component tree.
   * The slots of the component structure should have been loaded in before this
   * function is called.
   *
   * @inner
   * @param {baja.Component} root - The component at the root of the hierarchy, such as a point device ext or a folder.
   * @param {Function} filter - A predicate filter function, used to determine which children to obtain.
   * @returns {Array.<baja.Component>}
   */
  function getFlattenedFolderHierarchy(root, filter) {
    /**
     * Function to recursively reduce a list of components in folders to an array with
     * the parent object, followed by the children.
     */
    function doFlatten(parent, result) {
      return _.reduce(getDirectChildren(parent, filter), function (memo, child) {
        memo.push(child);
        memo = doFlatten(child, memo);
        return memo;
      }, result);
    }
    return doFlatten(root, []);
  }

  /**
   * From a given root component, return all the folders that are direct or
   * indirect descendants of that root in a flat array. This is similar to
   * the result of `getFlattenedFolderHierarchy()` but with any non-folder
   * components removed from the result.
   *
   * @param {baja.Component} root
   * @param {baja.Type} folderType
   * @returns {Array.<baja.Component>}
   */
  function getChildFolderHierarchy(root, folderType) {
    return getFlattenedFolderHierarchy(root, function (c) {
      return c.getType().is(folderType);
    });
  }

  /**
   * Create `Attachable`s for the folder descendants of the given component.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource} source
   * @param {baja.Component} comp
   */
  function attachToChildFolderHierarchy(source, comp) {
    var folderType = source.$folderType,
      descendantFolders = getChildFolderHierarchy(comp, folderType);
    _.each(descendantFolders, function (f) {
      attachToFolder(source, f);
    });
  }

  /**
   * Remove the `Attachable`s from the folder descendants of the given component.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource} source
   * @param {baja.Component} comp
   */
  function detachFromChildFolderHierarchy(source, comp) {
    var folderType = source.$folderType,
      folders = getChildFolderHierarchy(comp, folderType);
    _.each(folders, function (f) {
      detachFromFolder(source, f);
    });
  }

  /**
   * Create a new Attachable for the folder we have loaded in, using the
   * same filter as the instance this code is mixed into. This will emit
   * the same events for the folders as for the root container.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource}  source
   * @param {baja.Component} folder
   */
  function attachToFolder(source, folder) {
    var att = source.makeAttachable(folder);
    source.$folderAttachables.push({
      folder: folder,
      attachable: att
    });
  }

  /**
   * Detach the `Attachable` from a single folder, removing it from the array.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource}  source
   * @param {baja.Component} folder
   */
  function detachFromFolder(source, folder) {
    var attached = _.findWhere(source.$folderAttachables, {
      folder: folder
    });
    if (attached) {
      attached.attachable.detach();
      source.$folderAttachables = _.without(source.$folderAttachables, attached);
    }
  }

  /**
   * Used to wrap the `Promise`s returned by a recursive invocation of `loadComponentStructure`
   * within a `Promise.all`.
   */
  function loadAllKids(parent, folderType, currentDepth, maxDepth, batch) {
    return loadComponentStructure({
      parent: parent,
      folderType: folderType,
      batch: batch,
      currentDepth: currentDepth,
      maxDepth: maxDepth
    });
  }

  /**
   * Recursive function to load the slots of a component and then, for any folders we find,
   * load the slots on those, and recurse down to the following levels until it reaches the
   * leaf levels of the folder structure. This takes two depth related parameters - `maxDepth`,
   * which is the maximum depth to load non-folder components underneath a given folder, and
   * `currentDepth`, which is decremented as the next level of non-folder components is loaded.
   * The current depth is reset to max depth each time a folder is encountered and this
   * function is recursed. The recursion is done via a call to `loadAllKids()` which wraps the
   * recursive invocations of this function for a set of siblings within a `Promise`.
   *
   * @param {Object} params - object containing information such as parent component and folder type.
   * @param {baja.Component} params.parent - the root component in the source container.
   * @param {String|Type} params.folderType - the typespec of the manager's folder type.
   * @param {baja.comm.Batch} [params.batch] - batch used when recursively loading sibling folders' slots.
   * @param {Number} params.currentDepth - the depth counter, decremented when loading non-folder components.
   * @param {Number} params.maxDepth - the maximum depth to load when encountering a non-folder component.
   *
   * @returns {Promise.<*>}
   */
  function loadComponentStructure(params) {
    var parent = params.parent,
      folderType = params.folderType,
      loadSlotsArgs = {};
    if (params.batch) {
      loadSlotsArgs.batch = params.batch;
    }
    return parent.loadSlots(loadSlotsArgs).then(function () {
      var promises,
        depth = params.currentDepth,
        maxDepth = params.maxDepth,
        batch = new baja.comm.Batch(),
        cursor = parent.getSlots().properties().is('baja:Component'),
        children = cursor.toValueArray();
      promises = Promise.all(children.map(function (child) {
        // Try to create a Promise to load the kids for the given child. If the child is
        // a folder, we reset the depth counter and start loading the components beneath
        // it. If the child is not a folder, we check the current depth and see if we
        // need to load any further.

        if (child.getType().is(folderType)) {
          return loadAllKids(child, folderType, maxDepth, maxDepth, batch);
        } else if (depth > 0) {
          return loadAllKids(child, folderType, depth - 1, maxDepth, batch);
        }
      }));
      batch.commit();
      return promises;
    });
  }

  /**
   * Detach ALL attached folders and discard the `Attachable`s. This is used in two
   * cases - either when the 'all descendants' command is deselected, or when the source
   * is being destroyed.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource} source
   */
  function detachFolderAttachables(source) {
    _.each(source.$folderAttachables, function (attached) {
      attached.attachable.detach();
    });
    source.$folderAttachables = [];
  }

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * A mixin type to be applied to a `ContainerComponentSource`. This will extend
   * the source with functionality to support the `AllDescendantsCommand`, with the
   * ability to represent a hierarchy of folders and components as a flat set of
   * rows within the table.
   *
   * @alias module:webEditors/rc/wb/mgr/model/folderComponentSourceMixin
   *
   * @mixin
   * @param {module:nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource} target -
   * The component source instance that will have the mixin applied to it.
   * @param {Object} params - the parameter object
   * @param {string|Type} params.folderType - The folder type for the manager
   */
  var exports = function exports(target, params) {
    var folderType, superDestroy;
    params = params || {};
    if (!(target instanceof ContainerComponentSource)) {
      throw new Error('Folder source mixin must be applied to a ContainerComponentSource type');
    }
    if (!applyMixin(target, MIXIN_NAME)) {
      return;
    }
    folderType = params.folderType;
    superDestroy = target.destroy;
    if (!folderType) {
      throw new Error('Parameters must specify a type spec string or Type instance for a folder type');
    }
    target.$folderType = folderType;
    target.$flattened = false;
    target.$folderAttachables = [];
    target.$folderComponentDepth = params.componentDepth !== undefined ? params.componentDepth : 0;

    /**
     * Set the depth to use when loading non-folder items. This is expected to be set to
     * the same depth as a depth subscriber, if the manager is using one.
     *
     * @param {Number} depth - the component depth to load to underneath a folder.
     */
    target.setComponentDepth = function (depth) {
      this.$folderComponentDepth = depth;
    };

    /**
     * Load a flattened folder structure for the current root component and all its
     * descendants. This will call the recursive function `loadFolderStructure()`.
     * The returned array will contain a flattened set of folders and the contents
     * of those folders, subject to the filter supplied in the constructor.
     *
     * @private
     * @returns {Promise.<Array.<baja.Component>>}
     */
    target.$loadFlattenedComponents = function () {
      var that = this,
        root = that.getContainer();

      // Return a promise that will traverse down the component tree, loading
      // the slots for the children of folders that we find.

      return loadComponentStructure({
        parent: root,
        folderType: that.$folderType,
        maxDepth: that.$folderComponentDepth,
        currentDepth: that.$folderComponentDepth
      }).then(function () {
        attachToChildFolderHierarchy(that, root);
      });
    };

    /**
     * Called when the 'all descendants' command is unselected. This will detach all
     * the `Attachable`s we have on the folders below the root container.
     *
     * @returns {Promise.<*>}
     */
    target.$unloadFlattenedComponents = function () {
      detachFolderAttachables(this);
      return Promise.resolve();
    };

    /**
     * Allows the manager to choose whether to receive either just the immediate child
     * components of the root subject, or a flattened folder hierarchy, with components
     * in folders directly or indirectly under the root.
     *
     * @param flattened boolean value indicating whether the getComponents() function
     * should return a flattened folder hierarchy, or just the first level children.
     *
     * @returns {Promise.<*>}
     */
    target.setFlattened = function (flattened) {
      this.$flattened = !!flattened;
      if (this.$flattened) {
        return this.$loadFlattenedComponents();
      } else {
        return this.$unloadFlattenedComponents();
      }
    };

    /**
     * Returns whether this component source is set to return just the immediate
     * children of the root subject component (default) or will return a flattened
     * folder hierarchy of all components in all descendant folders.
     *
     * @returns {boolean}
     */
    target.isFlattened = function () {
      return this.$flattened;
    };

    /**
     * Override of the base class. This will either return the direct children
     * of the container, or all the direct and indirect descendants in a flattened
     * array.
     *
     * @override
     * @returns {Array.<baja.Component>}
     */
    target.getComponents = function () {
      var that = this,
        root = that.getContainer();
      return that.$flattened ? getFlattenedFolderHierarchy(root, that.$filter) : getDirectChildren(root, that.$filter);
    };

    /**
     * Return all the folders that are directly or indirectly descendants of the root
     * component, without their non-folder contents. This can be used for passing the
     * folders to a depth subscriber when the all descendants command is selected.
     *
     * @returns {Array.<baja.Component>}
     */
    target.getFlattenedFolders = function () {
      var that = this,
        root = that.getContainer();
      return getChildFolderHierarchy(root, that.$folderType);
    };

    /**
     * Overrides the base `handleAdded` function, to handle the addition of new folders
     * when in the flattened state. This will attach to the new folder(s) and ensure
     * the 'added' event is emitted with the descendant components too, so they can be
     * loaded into the table model.
     *
     * @param {Array.<baja.Component>} values - the array of new component(s)
     */
    target.handleAdded = function (values) {
      var that = this,
        flattened;
      if (this.isFlattened()) {
        flattened = [];
        _.each(values, function (value) {
          flattened.push(value);

          // If we are adding a folder and set to flatten them, attach to the folder
          // and its descendant folders, then add all the descendant values into the
          // array of values to be passed in the event's parameters.

          if (value.getType().is(that.$folderType)) {
            attachToFolder(that, value);
            attachToChildFolderHierarchy(that, value);
            flattened = flattened.concat(getFlattenedFolderHierarchy(value, that.$filter));
          }
        });
        values = flattened;
      }
      that.emit('added', values);
    };

    /**
     * Overrides the `handleRemoved' function to handle the removal of folders when
     * in the flattened state. This will detach the folder (and sub folders), and
     * augment the values with the children of those folders, so they can be removed
     * from the table model.
     *
     * @override
     * @param {Array.<baja.Component>} values - the array of removed component(s)
     */
    target.handleRemoved = function (values) {
      var that = this,
        flattened;
      if (this.isFlattened()) {
        flattened = [];
        _.each(values, function (value) {
          flattened.push(value);
          if (value.getType().is(that.$folderType)) {
            // If we are flattened, detach from the folder and any descendant folders we
            // are attached to. We also then notify the descendant components in the array
            // of components passed with the 'removed' event.

            detachFromFolder(that, value);
            detachFromChildFolderHierarchy(that, value);
            flattened = flattened.concat(getFlattenedFolderHierarchy(value, that.$filter));
          }
        });
        values = flattened;
      }
      that.emit('removed', values);
    };

    /**
     * Override the default ContainerComponentSource destruction. In addition to the
     * Attachable on the root container, this will clean up the `Attachable`s that are
     * currently attached to the folders.
     */
    target.destroy = function () {
      detachFolderAttachables(this);
      delete this.$folderAttachables;
      superDestroy.apply(this, arguments);
    };
  };
  return exports;
});
