/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/mgr/commands/MatchCommand
 */
define(['baja!', 'bajaux/commands/Command', 'bajaux/mixin/mixinUtils', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/mgr/commands/MgrCommand', 'nmodule/webEditors/rc/wb/mgr/BatchComponentEditor', 'nmodule/webEditors/rc/wb/mgr/mgrUtils', 'nmodule/webEditors/rc/wb/mgr/model/MgrModel', 'nmodule/webEditors/rc/wb/table/model/source/ArrayComponentSource'], function (baja, Command, mixinUtils, Promise, _, feDialogs, MgrCommand, BatchComponentEditor, mgrUtils, MgrModel, ArrayComponentSource) {
  'use strict';

  var getLearnTableSelectedSubjects = mgrUtils.getLearnTableSelectedSubjects,
    getMainTableSelectedSubjects = mgrUtils.getMainTableSelectedSubjects,
    setProposedDiscoveryValues = mgrUtils.setProposedDiscoveryValues;
  var _MgrCommand$flags = MgrCommand.flags,
    ALL_BARS = _MgrCommand$flags.ALL_BARS,
    LEARN_CONTEXT_MENU = _MgrCommand$flags.LEARN_CONTEXT_MENU;

  /**
   * Display the `BatchComponentEditor` for the component being matched.
   * This will already have had the proposed values from the matched discovery
   * object set at the point this is called.
   *
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} mgr - the manager
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} editorModel - the manager model for the batch editor
   */
  function editInstance(mgr, editorModel) {
    var baseComponent;
    var subMgrModel;
    return mgr.getOrdBase().then(function (ordBase) {
      baseComponent = ordBase;
      return feDialogs.showFor({
        value: editorModel,
        properties: {
          ordBase: baseComponent
        },
        type: BatchComponentEditor
      });
    }).then(function (subModel) {
      subMgrModel = subModel;
      if (subModel) {
        var dbModel = mgr.getModel();
        var instances = _.invoke(subModel.getRows(), 'getSubject');
        return dbModel.postMatch(instances);
      }
    }).then(function () {
      return subMgrModel;
    });
  }

  /**
   * Match the discovered item to the component in the database. This will
   * set the proposed values from the discovery object in the same way as
   * the `AddCommand` does.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} mgr - the manager
   * @param {baja.Component} comp - the existing database component
   * @param {*} discovery - the discovery item to be matched to the existing component.
   * @param {boolean} [quick] if true then perform a quick match
   */
  function matchSelection(mgr, comp, discovery, quick) {
    var databaseModel = mgr.getModel(),
      editModel,
      row;
    editModel = new MgrModel({
      componentSource: new ArrayComponentSource([comp]),
      columns: databaseModel.getEditableColumns()
    });
    row = _.first(editModel.getRows());
    return setProposedDiscoveryValues(mgr, databaseModel, discovery, row, quick).then(function () {
      if (quick) {
        return;
      }
      return editInstance(mgr, editModel);
    });
  }

  /**
   * Command to match a discovered object to an existing database object. This requires
   * one subject to be selected in each of the tables.
   *
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/webEditors/rc/wb/mgr/commands/MatchCommand
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} mgr
   */
  var MatchCommand = function MatchCommand(mgr, params) {
    var _this = this;
    Command.call(this, _.extend({
      module: 'webEditors',
      lex: 'commands.mgr.match',
      enabled: false,
      flags: ALL_BARS | LEARN_CONTEXT_MENU,
      /**
       * Match the selected object in the discovery table to the selected object
       * in the database table.
       *
       * @alias module:nmodule/webEditors/rc/wb/mgr/commands/MatchCommand#invoke
       * @returns {Promise}
       */
      func: function func() {
        var mainSelection, learnSelection;
        if (!mixinUtils.hasMixin(mgr, 'MGR_LEARN')) {
          return Promise.resolve();
        }
        mainSelection = getMainTableSelectedSubjects(mgr);
        learnSelection = getLearnTableSelectedSubjects(mgr);
        if (mainSelection.length === 1 && learnSelection.length === 1) {
          return matchSelection(mgr, mainSelection[0], learnSelection[0], _this.$quick)["catch"](feDialogs.error);
        } else {
          return Promise.reject(new Error('Must have one item selected in the learn and database tables'));
        }
      }
    }, params));
    this.$mgr = mgr;
    this.$quick = params && params.quick;
  };
  MatchCommand.prototype = Object.create(Command.prototype);
  MatchCommand.prototype.constructor = MatchCommand;
  return MatchCommand;
});
