/**
 *  @copyright 2018 Tridium, Inc. All Rights Reserved.
 *  @author Tony Richards
 */

/*eslint-env browser */ /*jshint browser: true */

define(['Promise'], function (Promise) {
  'use strict';

  /**
   * API Status: **Private**
   *
   * A set of utility functions for the chunking
   *
   * @exports nmodule/webEditors/rc/util/chunkUtil
   */
  var chunkUtil = {};
  chunkUtil.ABORT_MESSAGE = "abort";

  /**
   * Like a regular ajax request, but provides caller the ability to have a
   * progress and check handler for chunked responses
   *
   * @param {String} uri the uri of the request
   * @param {Object} params the object literal for the method's optional arguments.
   * @param {Function} [params.progress] function to be called with an array of
   * Strings that are fully available to process
   * @param {Function} [params.checkAbort] return true from this function to
   * abort the ajax request, will get called on each progress call
   * @param {String} [params.delimiter] Delimiter used between records of the
   * chunked data; defaults to '\n'.
   * @returns {Promise} resolves to the array of Strings which have yet to be
   * processed though the progress callback. These last results are not sent to the `progress` callback
   * because once the request is completed, it can no longer be aborted.
   * @private
   */
  chunkUtil.ajax = function (uri, params) {
    // eslint-disable-next-line promise/avoid-new
    return new Promise(function (resolve, reject) {
      params = params || {};
      var progress = params.progress,
        checkAbort = params.checkAbort,
        lastDelimiterIndex = 0,
        progressIndex = 0,
        delimiter = params.delimiter || '\n';

      // It would be nice to be able to support new RegExp('\\r?\\n') as the
      // default delimiter, but String.prototype.lastIndexOf doesn't support
      // regular expressions, and it would also complicate the API so that it
      // would likely need to support either String or RegExp.

      var request = new XMLHttpRequest();
      request.addEventListener("load", function (e) {
        if (request.status >= 300) {
          reject(new Error(e.statusText));
          return;
        }
        // Finished
        var data = request.responseText;
        if (lastDelimiterIndex > 0) {
          var lastUnprocessedChunks = data.substring(lastDelimiterIndex + 1).split("\n");
          resolve(lastUnprocessedChunks);
        } else {
          if (!data || !data.length) {
            resolve([]);
          } else {
            resolve(data.split("\n"));
          }
        }
      });
      request.addEventListener("progress", function () {
        var text = request.responseText,
          startString = lastDelimiterIndex === 0 ? 0 : 1;
        if (progress) {
          var lastDelimiter = text.lastIndexOf(delimiter);
          var progressDone;
          if (lastDelimiterIndex !== lastDelimiter && lastDelimiter !== -1) {
            var newResponse = text.substring(lastDelimiterIndex + startString, lastDelimiter);
            lastDelimiterIndex = lastDelimiter;
            progressDone = Promise.resolve(progress(newResponse.split(delimiter), progressIndex));
            progressIndex++;
          } else {
            progressDone = Promise.resolve();
          }
          return progressDone.then(function () {
            // Give one final chance to look at data (so abort after calling progress)
            if (checkAbort && checkAbort()) {
              request.abort();
              reject(new Error(chunkUtil.ABORT_MESSAGE));
            }
          });
        }
      });
      request.addEventListener("error", function (e) {
        reject(new Error(e));
      });
      request.open("GET", uri);
      request.send();
    });
  };
  return chunkUtil;
});
