/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/* jshint browser: true */
define(['baja!', 'log!nmodule.webEditors.rc.transform.exportUtils', 'Promise', 'underscore', 'bajaux/Widget', 'nmodule/js/rc/asyncUtils/asyncUtils', 'nmodule/export/rc/TransformOperation', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/servlets/registry', 'nmodule/webEditors/rc/servlets/userData', 'nmodule/webEditors/rc/transform/transformer/ExporterTransformer', 'nmodule/webEditors/rc/wb/profile/ServletViewWidget'], function (baja, log, Promise, _, Widget, asyncUtils, TransformOperation, compUtils, typeUtils, registry, userData, ExporterTransformer, ServletViewWidget) {
  'use strict';

  var doRequire = asyncUtils.doRequire,
    getNavOrd = compUtils.getNavOrd,
    getTypes = registry.getTypes,
    getTypeInfo = registry.getTypeInfo,
    isComponent = typeUtils.isComponent,
    flatten = _.flatten,
    pluck = _.pluck;
  var logSevere = log.severe.bind(log);

  /**
   * API Status: **Private**
   * @exports nmodule/webEditors/rc/transform/exportUtils
   */
  var exports = {};
  var EXPORT_USER_CONFIGURATION = 'EXPORT_USER_CONFIGURATION';

  /**
   * @return {Promise.<Array.<module:nmodule/export/rc/ExportDestinationType>>}
   */
  exports.getAvailableDestinations = function () {
    return getTypes({
      targetType: 'export:IJavaScriptExportDestination',
      showAbstract: false,
      showInterface: false
    }).then(function (results) {
      return getTypeInfo(pluck(results, 'type'));
    }).then(function (typeInfos) {
      var js = pluck(typeInfos, 'js'),
        deps = flatten(pluck(js, 'deps')),
        ids = pluck(js, 'id');
      return doRequire(ids, {
        deps: deps
      }).then(function (destinationCtors) {
        return destinationCtors.map(function (Destination) {
          return new Destination();
        }).filter(isValid);
      });
    });
  };
  function isValid(destination) {
    try {
      destination.checkValid();
      return true;
    } catch (err) {
      return false;
    }
  }

  /**
   * Find _all_ available transform operations for the object being transformed.
   *
   * @param {*} object
   * @returns {Promise.<Array.<module:nmodule/export/rc/TransformOperation>>}
   */
  exports.getAllTransformOperations = function (object) {
    return Promise.all([exports.getTransformOperationsFromProvider(object), exports.getTransformOperationsAsAgents(object)]).then(flatten);
  };

  //TODO: dup with ServletViewTransformOperationProvider
  /**
   * Find all transformations on the station for this object.
   *
   * @param {*|baja.Object} object
   * @return {Promise.<Array.<module:nmodule/export/rc/TransformOperation>>} array
   * of available transform operations. Will be empty if object not a
   * baja.Object or no transform operations found.
   */
  exports.getTransformOperationsAsAgents = function (object) {
    if (!baja.hasType(object)) {
      return Promise.resolve([]);
    }
    function toTransformOp(exporterInfo) {
      return new TransformOperation(new ExporterTransformer(exporterInfo), object);
    }
    return $getExporters(object).then(function (exporters) {
      return exporters.map(toTransformOp);
    });
  };

  /**
   * Find all transform operations provided by the given provider.
   *
   * @param {module:nmodule/export/rc/TransformOperationProvider|*} provider
   * @returns {Promise.<Array.<module:nmodule/export/rc/TransformOperation>>} array
   * of provided transform operations. Will be empty if object is not actually
   * a TransformOperationProvider.
   */
  exports.getTransformOperationsFromProvider = function (provider) {
    if (!provider || typeof provider.getTransformOperations !== 'function') {
      return Promise.resolve([]);
    }
    return Promise.resolve(provider.getTransformOperations());
  };

  /**
   * Get the file name for this destination type.
   *
   * @param {module:nmodule/export/rc/TransformOperation} transformOp The transform operation this destination type is a target for
   * @param {module:nmodule/export/rc/ExportDestinationType} [destination] The current export destination type
   * @returns {string} A file name for the transform operation
   */
  exports.getFileName = function (transformOp, destination) {
    var transformer = transformOp && transformOp.getTransformer(),
      fileExt = transformer && transformer.getFileExtension(),
      fileName = transformer && _.isFunction(transformer.getFileName) && transformer.getFileName(transformOp.getTransformedObject(), destination) || 'file';
    return fileExt ? fileName + '.' + fileExt : fileName;
  };

  /**
   * Get the file path for this destination type.
   *
   * @param {module:nmodule/export/rc/TransformOperation} transformOp The transform operation this destination type is a target for
   * @param {module:nmodule/export/rc/ExportDestinationType} [destination] The current export destination type
   * @returns {string} A file path for the transform operation and destination
   */
  exports.getFilePath = function (transformOp, destination) {
    var transformer = transformOp && transformOp.getTransformer(),
      filePath = transformer && _.isFunction(transformer.getFilePath) && transformer.getFilePath(transformOp.getTransformedObject(), destination) || '';
    return filePath || "";
  };

  /**
   * Use an RPC call to perform an export and save the results to a file, all
   * server-side.
   *
   * @param {baja.Ord|String} ordToExport ORD to the object that is to be
   * exported
   * @param {baja.Ord|String} ordToFile desired ORD to the file to be written
   * or overwritten
   * @param {String} exporterTypeSpec type spec of the `baja:Exporter` to use
   * @param {object} [cx={}] export context
   * @returns {Promise}
   */
  exports.exportToStationFile = function (ordToExport, ordToFile, exporterTypeSpec) {
    var cx = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
    return baja.rpc({
      typeSpec: 'export:ExporterRpc',
      method: 'exportToFile',
      args: [ordToExport, ordToFile, exporterTypeSpec].map(String).concat(_.mapObject(cx, function (value) {
        return JSON.stringify(baja.bson.encodeValue(value));
      }))
    });
  };

  /**
   * @private
   * @param {baja.Object} value
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/transform/exportUtils~ExporterInfo>>}
   */
  function $getExporters(value) {
    if (!value) {
      return Promise.resolve([]);
    }
    if (value instanceof baja.Ord) {
      return $getExportersForOrd(value);
    } else if ($isMounted(value)) {
      return $getExportersForOrd(value.getOrdInSession());
    } else {
      return getNavOrd(value).then(function (navOrd) {
        if (navOrd) {
          return $getExportersForOrd(navOrd);
        } else {
          return $getExportersForType(value.getType().getTypeSpec());
        }
      });
    }
  }

  /**
   * @private
   */
  function $getExportersForOrd(ord) {
    if (!ord) {
      return Promise.resolve([]);
    }
    return baja.rpc({
      typeSpec: "export:ExporterRpc",
      method: 'getExportersForOrd',
      args: [ord.toString()]
    });
  }

  /**
   * @private
   */
  function $getExportersForType(typeSpec) {
    if (!typeSpec) {
      return Promise.resolve([]);
    }
    return baja.rpc({
      typeSpec: "export:ExporterRpc",
      method: 'getExportersForType',
      args: [typeSpec]
    });
  }

  /**
   * @private
   * @param {baja.Component|*} value
   */
  function $isMounted(value) {
    return isComponent(value) && value.isMounted();
  }

  /**
   * @typedef {object} module:nmodule/webEditors/rc/transform/exportUtils~ExporterInfo
   * @property {string} typeSpec The typeSpec of the Exporter
   * @property {string} id The id of the Exporter
   * @property {string} displayName The displayName of the Exporter
   * @property {string} icon The icon of the Exporter
   * @property {string} props Additional meta information about the Exporter
   * @property {string} props.mimeType Mime Type of the Exporter
   * @property {string} props.fileExt File extension of the Exporter
   * @property {string} props.responseType Response Type of the exported data
   */

  /**
   * Get an object specifying the value of the transform operation to be stored
   * in user data.
   *
   * @param {module:nmodule/export/rc/TransformOperation} transformer
   * @returns {*}
   */
  function transformerPersistenceObject(transformer) {
    if (!transformer) {
      return {};
    } else {
      return {
        transformOp: transformer.getDisplayName()
      };
    }
  }

  /**
   * Get an object specifying the value of the destination type to be stored in
   * user data.
   *
   * @param {module:nmodule/export/rc/ExportDestinationType} destination
   * @returns {*}
   */
  function destinationPersistenceObject(destination) {
    if (!destination) {
      return {};
    } else {
      return {
        destination: destination.getDisplayName()
      };
    }
  }

  /**
   * @typedef {Object} module:nmodule/webEditors/rc/transform/exportUtils~PersistenceObject
   * @property {String} transformOp the selected transform operation's display name
   * @property {String} destination the selected destination type's display name
   * @property {Object.<String, Object.<String, String>>} transformers mapping of transform operation
   * display name to configuration objects. Each configuration object is a mapping of
   * slot names to BSON encoded values.
   */

  /**
   *
   * @param {object.<string, object.<string, baja.Value>>} transformerConfigMap mapping
   * of transform operation display name to configuration objects. Each configuration
   * object is a mapping of slot names to baja values.
   * @returns {object.<string, object.<string, string>>}
   */
  function persistTransformerConfigMap(transformerConfigMap) {
    return _.mapObject(transformerConfigMap, userData.toPersistenceObject);
  }

  /**
   *
   * @param {module:nmodule/export/rc/TransformOperation} transformOperation
   * @param {module:nmodule/export/rc/ExportDestinationType} destination
   * @param {object.<string, object.<string, baja.Value>>} transformerConfigMap mapping
   * of transform operation display name to configuration objects. Each configuration
   * object is a mapping of slot names to baja values.
   * @returns {module:nmodule/webEditors/rc/transform/exportUtils~PersistenceObject}
   */
  function persistenceObject(transformOperation, destination, transformerConfigMap) {
    var transformerObject = transformerPersistenceObject(transformOperation);
    var destinationObject = destinationPersistenceObject(destination);
    var componentObject = persistTransformerConfigMap(transformerConfigMap);
    var merge = _.extend({}, transformerObject, destinationObject);
    merge.transformers = componentObject;
    return merge;
  }

  /**
   * Gets a JSON and BSON encoded string of the user data configuration pieces
   * passed in.
   * @param {module:nmodule/export/rc/TransformOperation} transformer
   * @param {module:nmodule/export/rc/ExportDestinationType} destination
   * @param {object.<string, object.<string, baja.Value>>} transformerConfigMap
   * @returns {string}
   */
  function userDataString(transformer, destination, transformerConfigMap) {
    return JSON.stringify(persistenceObject(transformer, destination, transformerConfigMap));
  }

  /**
   * Given the transform operation, destination type, and object containing data
   * about various transform operation configs, persist the user export state to
   * the station.
   * @param {module:nmodule/export/rc/TransformOperation} transformer
   * @param {module:nmodule/export/rc/ExportDestinationType} destination
   * @param {object.<string, object.<string, baja.Value>>} transformers
   * @returns {Promise}
   */
  exports.persistUserConfiguration = function (transformer, destination, transformers) {
    var userConfig = userDataString(transformer, destination, transformers);
    return userData.put(EXPORT_USER_CONFIGURATION, userConfig);
  };

  /**
   * Recursively decode baja values from a nested object structure containing
   * bson values.
   * @param {object.<string, *>} object mapping of strings to either baja values
   * or objects which contain baja values
   * @returns {Promise.<object>}
   */
  function mapFromBsonAsync(object) {
    return baja.bson.decodeAsync(object).then(function (bson) {
      if (bson !== null) {
        return bson;
      }
      return Promise.all(Object.keys(object).map(function (key) {
        return mapFromBsonAsync(object[key]).then(function (bson) {
          object[key] = bson;
        });
      })).then(function () {
        return object;
      });
    });
  }

  /**
   * @typedef {Object} module:nmodule/webEditors/rc/transform/exportUtils~UserConfigurationObject
   * @property {String} transformOp the selected transform operation's display name
   * @property {String} destination the selected destination type's display name
   * @property {Object.<String, Object.<String, *>>} transformers mapping of transform operation
   * display name to configuration objects. Each configuration object is a baja value or an
   * object literal containing more baja values.
   */

  /**
   * If user data is stored, returns an object with destination field indicating
   * user destination type, transformOp field indicating user transform operation,
   * and transformers field indicating given configuration options that user has
   * chosen on various transformers.
   *
   * @returns {Promise.<module:nmodule/webEditors/rc/transform/exportUtils~UserConfigurationObject>}
   */
  exports.getUserConfiguration = function () {
    return userData.get(EXPORT_USER_CONFIGURATION).then(function (foundData) {
      var parseData = foundData && JSON.parse(foundData);
      if (!parseData) {
        return {};
      }
      var transformers = parseData.transformers;
      return mapFromBsonAsync(transformers).then(function (obj) {
        parseData.transformers = obj;
        return parseData;
      });
    })["catch"](function (err) {
      logSevere(err);
      return {};
    });
  };

  /**
   *
   * @param {module:nmodule/export/rc/TransformOperation} transformOperation
   * @returns {boolean}
   */
  exports.isTransformerSupplier = function (transformOperation) {
    var transformer = transformOperation && transformOperation.getTransformer();
    return !(transformer && _.isFunction(transformer.isSupplier) && !transformer.isSupplier());
  };

  /**
   * Find all available TransformOperations for the subject being exported.
   *
   * @param {module:nmodule/export/rc/TransformOperationProvider|*} subject
   * @returns {Promise.<baja.OrderedMap>}
   * @since Niagara 4.13
   */
  exports.makeOperationMap = function (subject) {
    return getOperations(subject).then(function (ops) {
      var map = new baja.OrderedMap();
      ops.forEach(function (op) {
        return map.put(op.getTransformer().getDisplayName(), op);
      });
      return map;
    });
  };

  /**
   * @param {*} subject
   * @returns {Promise.<Array.<module:nmodule/export/rc/TransformOperation>>}
   */
  function getOperations(subject) {
    var transformsOfSubject = exports.getAllTransformOperations(subject);
    if (subject instanceof Widget) {
      if (subject instanceof ServletViewWidget) {
        return transformsOfSubject;
      } else {
        return Promise.all([transformsOfSubject, exports.getAllTransformOperations(subject.value())]).then(flatten);
      }
    }
    return transformsOfSubject;
  }
  return exports;
});
