/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/mixin/DataMixin
 */
define(['bajaux/mixin/mixinUtils', 'underscore', 'nmodule/js/rc/tinyevents/tinyevents'], function (mixinUtils, _, tinyevents) {
  'use strict';

  var applyMixin = mixinUtils.applyMixin;
  var MIXIN_NAME = 'data';

  /**
   * Mixin to allow getting/setting arbitrary data on an object and receive
   * events when the data changes.
   *
   * @mixin
   * @alias module:nmodule/webEditors/rc/mixin/DataMixin
   * @param {object} target the object to receive the mixin
   * @param {object} [initialData] pass an initial set of data if desired
   * @example
   * DataMixin(obj);
   * obj.data('foo', 'bar');
   * obj.data('foo'); // 'bar'
   */
  var DataMixin = function DataMixin(target, initialData) {
    if (!applyMixin(target, MIXIN_NAME, DataMixin.prototype)) {
      return;
    }
    target.$data = _.extend({}, initialData);
  };

  /**
   * Get or set any arbitrary data on this object.
   *
   * If setting a value on an object with `tinyevents` on it, a `dataChanged`
   * event will be emitted with the key and new value.
   *
   * @param {String} key
   * @param {*} [value] the value to set; omit the second argument for a get
   * @returns {undefined|Array.<*>|Thenable} if `tinyevents` is on the object, the
   * results of its `emit()` call for the `dataChanged` event will be returned
   */
  DataMixin.prototype.data = function (key, value) {
    var data = this.$data;
    if (arguments.length === 1) {
      return data[key];
    } else {
      data[key] = value;
      if (tinyevents.isOn(this)) {
        return this.emit('dataChanged', key, value);
      }
    }
  };

  /**
   * Return an array of key names for the metadata values set on the row
   * via the `data()` method.
   *
   * @returns {Array.<String>}
   */
  DataMixin.prototype.getDataKeys = function () {
    return Object.keys(this.$data);
  };

  /**
   * Delete the metadata value with the given key name.
   *
   * @param {String} key
   */
  DataMixin.prototype.deleteData = function (key) {
    delete this.$data[key];
  };
  return DataMixin;
});
