function _toConsumableArray(r) { return _arrayWithoutHoles(r) || _iterableToArray(r) || _unsupportedIterableToArray(r) || _nonIterableSpread(); }
function _nonIterableSpread() { throw new TypeError("Invalid attempt to spread non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _iterableToArray(r) { if ("undefined" != typeof Symbol && null != r[Symbol.iterator] || null != r["@@iterator"]) return Array.from(r); }
function _arrayWithoutHoles(r) { if (Array.isArray(r)) return _arrayLikeToArray(r); }
function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'lex!js,webEditors', 'log!nmodule.webEditors.rc.fe.feDialogs', 'dialogs', 'jquery', 'Promise', 'underscore', 'bajaux/events', 'bajaux/Widget', 'bajaux/commands/Command', 'bajaux/util/ErrorDetailsWidget', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/BaseWidget', 'nmodule/webEditors/rc/fe/ValueWithSummaryWidget', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (baja, lexs, log, dialogs, $, Promise, _, events, Widget, Command, ErrorDetailsWidget, fe, BaseWidget, ValueWithSummaryWidget, compUtils, typeUtils) {
  'use strict';

  var noop = _.noop,
    omit = _.omit,
    once = _.once,
    pick = _.pick;
  var LOAD_EVENT = events.LOAD_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT;
  var VALUE_READY_EVENT = BaseWidget.VALUE_READY_EVENT;
  var _lexs = _slicedToArray(lexs, 2),
    jsLex = _lexs[0],
    webEditorsLex = _lexs[1];
  var isComplex = typeUtils.isComplex;
  var logError = log.severe.bind(log);
  var DEFAULT_DELAY = 200;
  var ENTER_KEY = 13;

  /**
   * Functions for showing field editors in modal dialogs. Useful for prompting
   * the user to enter values, edit individual slots, and fire actions.
   *
   * @exports nmodule/webEditors/rc/fe/feDialogs
   */
  var feDialogs = {};

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  //TODO: check for operator flag

  /**
   * Ensures that a mounted `component` and Action `slot` param are present.
   * If an actionArgument is provided ensure its at least a BValue.
   *
   * @private
   * @inner
   * @param {Object} params
   */
  function validateActionParams(params) {
    params = params || {};
    var component = params.component;
    if (!baja.hasType(component, 'baja:Component')) {
      throw new Error('component required');
    }
    if (!component.isMounted()) {
      throw new Error('component must be mounted');
    }
    var slot = component.getSlot(params.slot);
    if (!slot || !slot.isAction()) {
      throw new Error('Action slot required');
    }
    var actionArgument = params.actionArgument;
    if (actionArgument !== undefined && (!baja.hasType(actionArgument) || !actionArgument.getType().isValue())) {
      throw new Error('Action Arguments must be a Value');
    }
  }

  /**
   * Checks for `CONFIRM_REQUIRED` flag and shows confirmation dialog if
   * needed.
   *
   * @private
   * @inner
   * @param {baja.Complex} comp
   * @param {baja.Slot|String} slot
   * @returns {Promise} promise to be resolved if no confirmation was
   * needed or the user did confirm that the action should be invoked. Rejected
   * if the user did not confirm invocation.
   */
  function confirmInvoke(comp, slot) {
    // eslint-disable-next-line promise/avoid-new
    return new Promise(function (resolve, reject) {
      if (!(comp.getFlags(slot) & baja.Flags.CONFIRM_REQUIRED)) {
        return resolve();
      }
      var display = comp.getDisplayName(slot);
      dialogs.showOkCancel({
        title: webEditorsLex.get('dialogs.confirmInvoke.title', display),
        text: webEditorsLex.get('dialogs.confirmInvoke.content', display)
      }).ok(resolve).cancel(reject);
    });
  }

  /**
   * Build the editor in the given dialog.
   *
   * As the editor is created, initialized and loaded, progress events with
   * those same names will be passed to the given progress handler. This way,
   * someone using `feDialogs.showFor` can get callbacks for the actual editor
   * instance as it is created, and add event handlers on it, for instance.
   *
   * @inner
   * @param {Object} params fe params
   * @param {JQuery} contentDiv
   * @param {Function} progress
   * @param {Dialog} dlg The Dialog instance.
   * @returns {*}
   */
  function buildEditor(params, contentDiv, progress, dlg) {
    var parent = contentDiv.parent();
    var _params = params,
      summary = _params.summary;
    contentDiv.detach();
    if (summary) {
      params = omit(params, 'summary');
    }
    var value;
    return fe.params(params).then(function (feParams) {
      value = feParams.getValueToLoad();
      var makeForParams = feParams;
      if (summary) {
        var _value = feParams.value;
        makeForParams = {
          type: ValueWithSummaryWidget,
          value: _value,
          properties: {
            getConfig: function getConfig() {
              return feParams;
            },
            summary: summary
          }
        };
      }
      return fe.makeFor(makeForParams);
    }).then(function (ed) {
      progress('created', ed);
      return ed.initialize(contentDiv).then(function () {
        ed.$dlg = dlg;
        progress('initialized', ed);
        return ed.load(value);
      }).then(function () {
        progress('loaded', ed);
        contentDiv.prependTo(parent);
        return ed;
      }, function (err) {
        throw err;
      });
    });
  }
  function readAndDestroy(ed, shouldSave, onSaveError) {
    var modified = ed.isModified();
    return Promise.resolve(shouldSave && ed.save())["catch"](function (err) {
      var errorProm = onSaveError ? Promise.resolve(onSaveError(err)) : feDialogs.error(err);
      return errorProm["catch"](logError).then(function () {
        throw err; //failed to validate - keep dialog open
      });
    }).then(function () {
      return ed.read();
    }).then(function (value) {
      return Promise.resolve(modified && emitValueReady(ed, value)).then(function () {
        return ed.destroy()["catch"](logError);
      }).then(function () {
        return value;
      });
    });
  }

  /**
   * @param {module:bajaux/Widget} ed
   * @param {*} value
   * @returns {Promise}
   */
  function emitValueReady(ed, value) {
    return ed.emitAndWait(VALUE_READY_EVENT, value);
  }

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * Shows a field editor in a dialog.
   *
   * When the user clicks OK, the editor will be saved, committing any changes.
   * The value that the user entered will be read from the editor and used to
   * resolve the promise.
   *
   * @param {Object} params params to be passed to `fe.buildFor()`.
   * @param {jQuery} [params.dom] if your widget type should be instantiated
   * into a specific kind of DOM element, it can be passed in as a parameter.
   * Note that the given element will be appended to the dialog element itself,
   * so do not pass in an element that is already parented. If omitted, a `div`
   * will be created and used.
   * @param {String} [params.title] title for the dialog
   * @param {String} [params.summary] (since Niagara 4.14) provide this to display a more detailed
   * textual description about this prompt, e.g.: a description of its purpose, instructions on how
   * to use the editor shown, any additional information the user may want.
   * @param {Number} [params.delay=200] delay in ms to wait before showing a
   * loading spinner. The spinner will disappear when the field editor has
   * finished initializing and loading.
   * @param {boolean} [params.save] set to false to specify that the dialog
   * should *not* be saved on clicking OK - only the current value will be read
   * from the editor and used to resolve the promise.
   * @param {Function} [params.onSaveError] when this function is set and save
   * is set to true, the error will be handed off to this method. Otherwise,
   * when save is true feDialogs will show whatever error caused by the save in
   * a different dialog. This may optionally return a promise.
   * @param {Function} [params.progressCallback] pass a progress callback to
   * receive notifications as the editor being shown goes through the stages
   * of its life cycle (`created`, `initialized`, `loaded`), as well as whenever
   * the editor is validated (`invalid`, `valid`).
   * @param {Array.<module:dialogs~Button|string>} [params.buttons] as of Niagara 4.12,
   * custom buttons can be specified. See examples for details.
   * @param {Object.<string, Function>} [params.on] as of Niagara 4.12, custom handlers for
   * `bajaux` events (and only `bajaux` events) can be specified. This is an object literal
   * where the keys are `bajaux` event names and the values are event handler functions.
   * See examples for details.
   * @param {Function} [params.validate] as of Niagara 4.14, specify a custom validate function
   * to ensure that the entered value is valid. If the user enters an invalid value, the OK button
   * will be disabled. This function should throw an Error, return a rejected Promise, or return or
   * resolve `false` to fail validation.
   * @returns {Promise} promise to be resolved when the user has entered
   * a value into the field editor and clicked OK, or rejected if the field
   * could not be read. The promise will be resolved with the value that the
   * user entered (or `null` if Cancel was clicked).
   *
   * @example
   *   feDialogs.showFor({
   *     value: 'enter a string here (max 50 chars)',
   *     properties: { max: 50 },
   *     progressCallback: function (msg, arg) {
   *       switch(msg) {
   *       case 'created':     return console.log('editor created', arg);
   *       case 'initialized': return console.log('editor initialized', arg.jq());
   *       case 'loaded':      return console.log('editor loaded', arg.value());
   *       case 'invalid':     return console.log('validation error', arg);
   *       case 'valid':       return console.log('value is valid', arg);
   *       }
   *     }
   *   })
   *   .then(function (str) {
   *     if (str === null) {
   *       console.log('you clicked cancel');
   *     } else {
   *       console.log('you entered: ' + str);
   *     }
   *   });
   *
   * @example
   * <caption>Specify custom button handlers. If the user clicks one of these custom buttons,
   * the showFor promise will be resolved with the value resolved by its handler.</caption>
   *
   * feDialogs.showFor({
   *   value: 'enter a string',
   *   buttons: [ {
   *     name: 'uppercase',
   *     displayName: 'Uppercase It',
   *     handler: (dialog, event, editor) => {
   *       // the arguments to the button handler are: the Dialog instance, the click event,
   *       // and the editor being shown in the dialog.
   *       // call `dialog.keepOpen` if you are not ready for the dialog to close. the dialog will
   *       // stay open and the promise will not be resolved yet.
   *
   *       dialog.keepOpen();
   *       return editor.read().then((string) => editor.load(string.toUpperCase());
   *     }
   *   }, {
   *     name: 'lowercase',
   *     displayName: 'Lowercase It',
   *     handler: (dialog, event, editor) => {
   *       dialog.keepOpen();
   *       return editor.read().then((string) => editor.load(string.toLowerCase()));
   *     }
   *   }, {
   *     // default 'ok' behavior is to read the value and resolve the promise. you don't have to
   *     // specify a handler to do this.
   *     name: 'ok'
   *   } ]
   * });
   *
   * @example
   * <caption>The strings 'ok', 'cancel', 'yes', and 'no' are special - you can include them in the buttons
   * parameter to get their default behavior.</caption>
   *
   * // only show the OK button, and resolve the promise with the entered value. 'yes' works the same.
   * feDialogs.showFor({ value: 'enter a string', buttons: [ 'ok' ] });
   *
   * // only show the Cancel button, and resolve the promise with null. 'no' works the same.
   * feDialogs.showFor({ value: 'your changes will not be used', buttons: [ 'cancel' ] });
   *
   * @example
   * <caption>The buttons parameter can be an object literal, where the values are button
   * definitions or handler functions.</caption>
   *
   * feDialogs.showFor({
   *   value: 'Value to Edit',
   *   buttons: {
   *     ok: () => 'my custom ok result', // the value can be just a handler function. the default display name will be used.
   *     cancel: {
   *       displayName: "Never Mind"
   *       // omit the handler, and default handler for "cancel" will resolve null.
   *     },
   *     yes: {}, // just an empty object will use the default display name and default handler.
   *     no: {
   *       handler: () => 'user clicked "no"' // include a handler to override the default handler.
   *     },
   *     delete: {
   *       // for anything other than 'ok', 'cancel', 'yes', or 'no', you'll need to provide a
   *       // display name - or else just the button name will be used.
   *       displayName: 'Delete Everything',
   *       handler: () => deleteEverything()
   *     },
   *     retry: shouldShowRetryButton() && { // falsy values will cause the button _not_ to be shown.
   *       displayName: 'Try Again',
   *       handler: () => letUserTryAgain()
   *     }
   *   }
   * });
   *
   * @example
   * <caption>Use the 'on' parameter to respond to any bajaux events that are triggered by the
   * editor.</caption>
   *
   * const { MODIFY_EVENT } = events;
   * feDialogs.showFor({
   *   value: 'edit me',
   *   properties: { max: 10 },
   *   on: {
   *     [MODIFY_EVENT]: (dialog, event, editor) {
   *       return editor.validate()
   *         .catch(() => alert('no more than 10 characters pls'));
   *     }
   *   }
   * });
   */
  feDialogs.showFor = function showFor(params) {
    params = baja.objectify(params, 'value');
    var _params2 = params,
      delay = _params2.delay,
      dom = _params2.dom,
      _params2$on = _params2.on,
      on = _params2$on === void 0 ? {} : _params2$on,
      progressCallback = _params2.progressCallback,
      summary = _params2.summary,
      title = _params2.title,
      validate = _params2.validate;
    var contentDiv = dom || $('<div/>');
    if (contentDiv.parent().length) {
      return Promise.reject(new Error('element already parented'));
    }
    return normalizeButtons(params.buttons || ['ok', 'cancel'])
    // eslint-disable-next-line promise/avoid-new
    .then(function (buttons) {
      return new Promise(function (resolve, reject) {
        var editor;

        // when using a summary, the value editor is a child of a ValueWithSummaryWidget so will not
        // appear until the loaded event. we have to rapid-fire them after the editor becomes available.
        var innerEdAppeared = summary && once(function () {
          var inner = getEditorForCaller(editor);
          if (progressCallback) {
            progressCallback('created', inner);
            progressCallback('initialized', inner);
            progressCallback('loaded', inner);
          }
          if (validate) {
            applyValidator(inner, validate);
          }
        });
        var progress = function progress(event, ed) {
          if (event === 'created') {
            editor = ed;
            if (validate && !summary) {
              applyValidator(ed, validate);
            }
          }
          if (innerEdAppeared) {
            if (event === 'loaded') {
              innerEdAppeared();
              innerEdAppeared = null; // allow later progress messages to propagate as normal
            }
          } else {
            progressCallback && progressCallback(event, getEditorForCaller(ed));
          }
        };
        var firstShown = once(function () {
          /* wait until the content is visible then toggle its visibility
          off and on to work around iOS -webkit-touch-scrolling issue */
          contentDiv.toggle(0);
          contentDiv.toggle(0);
          return editor && editor.requestFocus && editor.requestFocus();
        });
        dialogs.show({
          buttons: withoutHandlers(buttons),
          delay: delay || DEFAULT_DELAY,
          title: title,
          layout: function layout() {
            //layout the editor when the dialog lays out
            firstShown();
            return editor && editor.layout();
          },
          content: function content(dlg, _content) {
            contentDiv.appendTo(_content);
            contentDiv.on(LOAD_EVENT + ' ' + MODIFY_EVENT, function (e, ed) {
              if (ed === editor) {
                editor.validate().then(function (value) {
                  var okJq = dlg.buttonJq('ok');
                  if (okJq) {
                    okJq.attr('title', '');
                    dlg.enableButton("ok");
                  }
                  progress('valid', value);
                }, function (err) {
                  var okJq = dlg.buttonJq('ok');
                  if (okJq) {
                    okJq.attr('title', String(err));
                    dlg.disableButton("ok");
                  }
                  progress('invalid', err);
                });
              }
            });
            contentDiv.on('keyup', function (e) {
              /*
              Previously code was put in to check for a 'keydown' before adding a checking
              on 'keyup' on the 'ENTER_KEY'.  But the reason for that change was lost, and
              it was causing some dialogs to not close when an ENTER_KEY was pressed.  From
              what can be tell putting this back, is not causing an issue, but if a future
              issue with ENTER_KEY and closing dialogs comes up, we might need to revisit this
              code.
               */
              if (e.which === ENTER_KEY) {
                Widget["in"](contentDiv).validate().then(function () {
                  return dlg.click('ok');
                })["catch"](noop);
              }
            });
            Object.keys(on).forEach(function (eventName) {
              var handler = on[eventName];
              contentDiv.on(eventName, function (e, ed) {
                var _this = this;
                for (var _len = arguments.length, rest = new Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
                  rest[_key - 2] = arguments[_key];
                }
                if (ed !== editor) {
                  return;
                }
                Promise["try"](function () {
                  return handler.apply(_this, [dlg, e, getEditorForCaller(ed)].concat(rest));
                })["catch"](logError);
              });
            });
            return buildEditor(params, contentDiv, progress, dlg).then(function (ed) {
              contentDiv.on(VALUE_READY_EVENT, function (e, value) {
                emitValueReady(ed, value).then(function () {
                  return ed.destroy()["catch"](logError);
                }).then(function () {
                  dlg.close();
                  resolve(value);
                })["catch"](logError);
              });
              buttons.forEach(function (btn) {
                var handler = getButtonHandler(btn, ed, params);
                dlg.on(btn.name, function (dlg) {
                  var _arguments = arguments,
                    _this2 = this;
                  var keepOpen;
                  dlg.keepOpen = function () {
                    keepOpen = true;
                  };
                  return Promise["try"](function () {
                    return handler.apply(_this2, [].concat(_toConsumableArray(_arguments), [getEditorForCaller(ed)]));
                  }).then(function (result) {
                    if (keepOpen) {
                      throw new Error();
                    }
                    delete dlg.keepOpen;
                    return ed.destroy()["catch"](logError).then(function () {
                      return resolve(result);
                    });
                  }, function (err) {
                    logError(err);
                    throw err;
                  });
                });
              });
            })["catch"](function (err) {
              _content.text(String(err));
              reject(err);
            });
          }
        });
      });
    });
  };
  function getButtonHandler(btn, ed, params) {
    var handler = btn.handler;
    switch (btn.name) {
      case 'ok':
      case 'yes':
        return handler || function () {
          var save = params.save,
            onSaveError = params.onSaveError;
          var shouldSave = save !== false;
          return readAndDestroy(ed, shouldSave, onSaveError);
        };
      case 'cancel':
      case 'no':
        return handler || function () {
          return null;
        };
      default:
        return handler || function () {};
    }
  }

  /**
   * @param {Array.<module:dialogs~Button|string|bajaux/commands/Command>|object} buttons
   * @returns {Promise.<Array.<module:dialogs~Button>>}
   */
  function normalizeButtons(buttons) {
    if (!Array.isArray(buttons)) {
      buttons = Object.keys(buttons).map(function (name) {
        var button = buttons[name];
        if (!button) {
          return;
        }
        if (typeof button === 'function') {
          button = {
            handler: button
          };
        }
        button.name = name;
        return button;
      });
    }
    var cmdCount = 0;
    return Promise.all(buttons.filter(function (b) {
      return !!b;
    }).map(function (btn) {
      if (btn instanceof Command) {
        return btn.toDisplayName().then(function (displayName) {
          return {
            name: 'cmd' + cmdCount++,
            displayName: displayName,
            handler: function handler() {
              return btn.invoke();
            }
          };
        });
      }
      if (typeof btn === 'string') {
        btn = {
          name: btn
        };
      }
      var _btn = btn,
        name = _btn.name,
        displayName = _btn.displayName;
      if (!displayName) {
        switch (name) {
          case 'ok':
          case 'cancel':
          case 'yes':
          case 'no':
            btn.displayName = jsLex.get('dialogs.' + name);
            break;
          default:
            btn.displayName = name;
        }
      }
      return btn;
    }));
  }
  function withoutHandlers(buttons) {
    return buttons.map(function (btn) {
      if (typeof btn === 'string') {
        btn = {
          name: btn
        };
      }
      return omit(btn, 'handler');
    });
  }

  /**
   * Show an editor in a dialog, similar to `showFor`, but with the added
   * expectation that the editor represents a one-time interaction, like a
   * button click, after which the dialog can be immediately closed. In other
   * words, the "click ok to close" functionality is embedded in the editor
   * itself. Only a Cancel button will be shown in the dialog itself.
   *
   * In order for the dialog to close, the shown editor must trigger a
   * `feDialogs.VALUE_READY_EVENT`, optionally with a read value. When this
   * event is triggered, the dialog will be closed and the promise resolved
   * with the value passed to the event trigger.
   *
   * @param {Object} params params to be passed to `fe.buildFor`
   * @param {String} [params.title] title for the dialog
   * @param {Number} [params.delay=200] delay in ms to wait before showing a
   * loading spinner. The spinner will disappear when the field editor has
   * finished initializing and loading.
   * @param {Function} [params.progressCallback] pass a progress callback to
   * receive notifications as the editor being shown goes through the stages
   * of its life cycle (created, initialized, loaded).
   * @returns {Promise} promise to be resolved when the editor has
   * triggered its own value event. It will be resolved with any value passed
   * to the event trigger, or with `null` if Cancel was clicked.
   *
   * @example
   *   <caption>Trigger a VALUE_READY_EVENT to cause the dialog to be closed.
   *   </caption>
   *
   * // ...
   * MyEditor.prototype.doInitialize = function (dom) {
   *   dom.on('click', 'button', function () {
   *     dom.trigger(feDialogs.VALUE_READY_EVENT, [ 'my value' ]);
   *   });
   * };
   * //...
   *
   * feDialogs.selfClosing({
   *   type: MyEditor
   * }}
   *   .then(function (value) {
   *     if (value === 'my value') {
   *       //success!
   *     }
   *   });
   */
  feDialogs.selfClosing = function (params) {
    params = baja.objectify(params, 'value');
    var progress = params.progressCallback || $.noop;
    var delay = params.delay;
    if (typeof delay === 'undefined') {
      delay = DEFAULT_DELAY;
    }

    // eslint-disable-next-line promise/avoid-new
    return new Promise(function (resolve, reject) {
      dialogs.showCancel({
        delay: delay,
        title: params.title,
        content: function content(dlg, _content2) {
          var contentDiv = $('<div/>').appendTo(_content2);
          dlg.cancel(function () {
            return resolve(null);
          });
          buildEditor(params, contentDiv, progress).then(function (ed) {
            contentDiv.on(VALUE_READY_EVENT, function (e, value) {
              ed.destroy()["finally"](function () {
                dlg.close();
                resolve(value);
              });
            });
          })["catch"](reject);
        }
      });
    });
  };

  /**
   * Invoke an action on a mounted component. If the action requires a
   * parameter, a field editor dialog will be shown to retrieve that argument
   * from the user.
   *
   * @param {Object} params
   * @param {baja.Component} params.component the component on which to invoke
   * the action. Must be mounted.
   * @param {String|baja.Slot} params.slot the action slot to invoke. Must be
   * a valid Action slot.
   * @param {baja.Value} [params.actionArgument] Starting in Niagara 4.10, this
   * action argument can be used instead of showing a dialog to obtain
   * the argument.
   * @returns {Promise} promise to be resolved with the action return
   * value if the action was successfully invoked, resolved with `null` if
   * the user clicked Cancel, or rejected if the parameters were invalid or the
   * action could not be invoked.
   */
  feDialogs.action = function action(params) {
    try {
      validateActionParams(params);
    } catch (e) {
      return Promise.reject(e);
    }
    function performInvocation(comp, slot, actionArgument) {
      var actionArgumentProvided = actionArgument !== undefined;
      var valueParam;
      return Promise.resolve(actionArgumentProvided || compUtils.resolveActionParams(comp, slot)).then(function (params) {
        if (actionArgumentProvided) {
          return actionArgument;
        }
        if (params === undefined) {
          return;
        }
        var value = params.value,
          title = params.title,
          type = params.type,
          properties = params.properties;
        valueParam = value;
        return feDialogs.showFor({
          value: value,
          title: title,
          type: type,
          properties: properties
        });
      }).then(function (readValue) {
        if (readValue === null) {
          // user clicked cancel to parameter dialog
          return null;
        }
        return comp.invoke({
          //complexes are always edit by ref.
          value: isComplex(valueParam) ? valueParam : readValue,
          slot: slot
        });
      });
    }
    var comp = params.component;
    var slot = params.slot;
    var actionArgument = params.actionArgument;
    return confirmInvoke(comp, slot).then(function () {
      return performInvocation(comp, slot, actionArgument);
    }, function () {
      return /* invocation canceled */null;
    })["catch"](function (err) {
      return feDialogs.error(err).then(function () {
        throw err;
      });
    });
  };

  /**
   * A simple mechanism for editing multiple properties at once.
   *
   * @param {module:nmodule/webEditors/rc/fe/feDialogs~EditableJSON|baja.Complex} props a JSON object representing the
   * properties to edit. May be nested. You can also simply pass a Complex to edit the slots of
   * that Complex. See examples.
   * @param {object} [params]
   * @param {baja.Component} [params.ordBase] if field editors may need to resolve ORDs themselves,
   * pass `ordBase` to allow them to successfully resolve ORDs even when offline.
   * @param {string} [params.title] optional dialog title
   * @param {string} [params.summary] optional summary details for dialog
   * @param {Function} [params.validate] optional validation function to ensure that the entered
   * value is valid. If the user enters an invalid value, the OK button will be disabled. This
   * function should throw an Error, return a rejected Promise, or return or resolve `false` to fail
   * validation. It will receive a JSON object or a Complex, depending on what was passed as the
   * `props` argument.
   * @returns {Promise.<object|null>} the newly entered values, or null if user clicked Cancel
   * @since Niagara 4.14
   *
   * @example
   * <caption>Edit a simple key->value map.</caption>
   * return feDialogs.props({ foo: 'bar' }); // resolves an object with user-edited "foo" property
   *
   * @example
   * <caption>Edit a nested key->value map.</caption>
   * return feDialogs.props({
   *   user: { value: { firstName: 'Moe', lastName: 'Howard' } }
   * }); // resolves an object with a "user" property containing user-editor "firstName" and "lastName"
   *
   * @example
   * <caption>Specify display name</caption>
   * return feDialogs.props({
   *   foo: { displayName: 'Your foo value', value: 'bar' }
   * }); // resolves an object with user-entered "foo" property, but user was shown customized display name
   *
   * @example
   * <caption>Use display name from lexicon</caption>
   * return feDialogs.props({
   *   overrideValue: { displayName: '%lexicon(control:override.value)%', value: 0 }
   * }); // resolves an object with user-entered "overrideValue" property but user was shown "Override Value" from lexicon
   *
   * @example
   * <caption>Specify flags</caption>
   * return feDialogs.props({
   *   hidden: { value: 'hiddenValue', hidden: true },
   *   readonly: { value: 'readonlyValue', readonly: true }
   * }); // resolves an object with "hidden" and "readonly" values. User did not see "hidden" and was not able to edit "readonly".
   *
   * @example
   * <caption>Specify properties</caption>
   * return feDialogs.props({
   *   percent: { value: 0, properties: { min: 0, max: 100 } }
   * }); // resolves an object with "percent" value. Editor was validated to be between 0 and 100.
   *
   * @example
   * <caption>Edit a Complex directly</caption>
   * const comp = baja.$('control:NumericWritable');
   * return feDialogs.props(comp); // please note that the input Complex will be *saved* when the user clicks OK
   *
   * @example
   * <caption>Specify title and summary</caption>
   * return feDialogs.props({ foo: 'bar' }, {
   *   title: 'Foo Dialog',
   *   summary: 'Please enter your foo value'
   * });
   */
  feDialogs.props = function (props) {
    var params = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    var title = params.title,
      summary = params.summary,
      validate = params.validate;
    return feDialogs.showFor({
      type: 'nmodule/webEditors/rc/fe/JSONPropertySheet',
      // no circular dependency
      value: props,
      formFactor: 'max',
      readonly: params.readonly,
      title: title,
      summary: summary,
      validate: validate,
      properties: pick(params, 'ordBase'),
      dom: $('<div class="-t-feDialogs-props-dialog"></div>')
    });
  };

  /**
   * When the user passes a `summary` parameter the actual editor shown in the dialog will be a
   * ValueSummaryWidget. But the user's own handlers for bajaux events and button clicks don't want
   * to receive a ValueSummaryWidget - that's internal details - if the caller showFor()s a String,
   * they want to interact with a StringEditor. Give them that StringEditor.
   *
   * @param {module:bajaux/Widget} ed the editor shown in the dialog
   * @returns {module:bajaux/Widget} the editor the caller expects to interact with
   */
  function getEditorForCaller(ed) {
    return ed instanceof ValueWithSummaryWidget ? ed.getInnerValueEditor() : ed;
  }
  function applyValidator(ed, validate) {
    ed.validators().add(function (v) {
      return Promise["try"](function () {
        return validate(v);
      }).then(function (result) {
        if (result === false) {
          throw new Error();
        }
      });
    });
  }

  /**
   * Show details about an error.
   *
   * @param {Error|*} err
   * @param {Object} [params]
   * @param {String} [params.title]
   * @param {module:bajaux/commands/Command} [params.command] An optional Command to help display information on which Command failed
   * @param {String} [params.messageSummary] An optional messageSummary to prepend to the Error.
   * @returns {Promise}
   */
  feDialogs.error = function (err) {
    var params = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    logError(err);
    return ErrorDetailsWidget.dialog(err, params);
  };
  return feDialogs;
});

/**
 * JSON format for data intended to be shown to, and edited by, the user using a Property Sheet.
 * This allows for more intuitive user editing of nested/annotated data, using the familiar Property
 * Sheet interface, without requiring the developer to translate a bunch of non-Niagara data into a
 * Component and back.
 *
 * @typedef {Object.<string, module:nmodule/webEditors/rc/fe/feDialogs~EditableJSON|baja.Complex|*>} module:nmodule/webEditors/rc/fe/feDialogs~EditableJSON
 * @property {string|number|boolean} value the value to be edited
 * @property {string} [displayName] display name for the edited value, in lexicon format. If
 * omitted, the string key will be used.
 * @property {boolean} [hidden] true if this value should be hidden from the user for editing, but
 * still included in the results
 * @property {boolean} [readonly] true if this value should be readonly
 * @property {Object.<string, *>} [properties] any additional bajaux properties that should be
 * applied to the field editor for this value
 *
 * @see module:nmodule/webEditors/rc/fe/feDialogs#props
 */
