function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'lex!', 'Promise', 'underscore', 'nmodule/webEditors/rc/util/textUtils'], function (baja, lexjs, Promise, _, textUtils) {
  'use strict';

  var every = _.every;
  var toFriendly = textUtils.toFriendly;
  var knownBadTypes = {};

  /**
   * Is this string a valid baja type spec?
   *
   * @inner
   * @type {RegExp}
   */
  var TYPE_SPEC_REGEX = /^[A-Za-z][A-Za-z0-9_]*:[A-Z][A-Za-z0-9_]*$/;

  /**
   * Search through the type for its contract info for the given slot.
   *
   * @inner
   * @param {Type} type
   * @param {String|baja.Slot} slot
   * @returns {Object}
   * @throws {Error} if the slot could not be found on the type
   */
  function findInContract(type, slot) {
    if (!slot) {
      throw new Error('slot required');
    }
    var inType = type,
      slotName = String(slot),
      contract,
      obj,
      i;
    while (type && (contract = type.getContract())) {
      for (i = 0; i < contract.length; i++) {
        obj = contract[i];
        if (obj.n === slotName) {
          return obj;
        }
      }
      type = type.getSuperType();
    }
    throw new Error('slot ' + slotName + ' not found on ' + inType + ' or any supertype');
  }

  /**
   * Get the unique slot names in the contract.
   *
   * @inner
   * @param {Type} type
   * @param {Function} [filter] an optional filter
   * @returns {Array.<String>}
   */
  function getSlotNamesInContract(type, filter) {
    if (!type.isComplex()) {
      throw new Error('Complex Type required');
    }
    var results = [];
    var contract, obj, i;
    while (type && (contract = type.getContract())) {
      for (i = 0; i < contract.length; i++) {
        obj = contract[i];
        if (!filter || filter(obj)) {
          var name = obj.n;
          if (!results.includes(name)) {
            results.push(name);
          }
        }
      }
      type = type.getSuperType();
    }
    return results;
  }
  function getType(value) {
    return baja.hasType(value) ? value.getType() : value.constructor;
  }

  /**
   * API Status: **Private**
   * @exports nmodule/webEditors/rc/fe/baja/util/typeUtils
   */
  var exports = {};

  /**
   * @param {Array.<*>} values
   * @returns {boolean} true if all the values are the same Niagara Type, or
   * instances of the same constructor.
   */
  exports.allSameType = function (values) {
    if (!values.length) {
      return true;
    }
    var type = getType(values[0]);
    for (var i = 1; i < values.length; i++) {
      if (getType(values[i]) !== type) {
        return false;
      }
    }
    return true;
  };

  //TODO: BajaScript should eventually give a nice function for this.
  /**
   * Return the slot display name for a certain slot on a type.
   *
   * @param {Type} type
   * @param {String|baja.Slot} slotName
   * @returns {String} slot display name
   * @throws {Error} if slot was not found on the Type
   */
  exports.getSlotDisplayName = function (type, slotName) {
    return findInContract(type, slotName).dn;
  };

  /**
   * Return the default value for a certain slot on a type.
   *
   * @param {Type} type
   * @param {String|baja.Slot} slotName
   * @returns {baja.Value | undefined}
   * @throws {Error} if slot value was not found on the Type
   */
  exports.getSlotDefaultValue = function (type, slotName) {
    var obj = findInContract(type, String(slotName));
    if (obj.v) {
      return baja.bson.decodeValue(obj.v);
    }
    throw new Error('no default value found, slot: ' + slotName + ' is an action/topic on ' + type);
  };

  //TODO: BajaScript should eventually give a nice function for this.
  /**
   * Return the slot facets for a certain slot on a type.
   *
   * @param {Type} type
   * @param {String|baja.Slot} slotName
   * @returns {baja.Facets}
   * @throws {Error} if slot was not found on the Type
   */
  exports.getSlotFacets = function (type, slotName) {
    var obj = findInContract(type, String(slotName)),
      facets = baja.Facets.DEFAULT;
    return obj.x ? facets.decodeFromString(obj.x, baja.Simple.$unsafeDecode) : facets;
  };

  /**
   * Return the Type of the given slot. If the slot is an action or topic,
   * return null.
   *
   * @param {Type} type
   * @param {String|baja.Slot} slotName
   * @returns {Type|null} the Type of the given slot, or `null` if it's not a
   * Property
   * @throws {Error} if slot was not found on the Type
   */
  exports.getSlotType = function (type, slotName) {
    var obj = findInContract(type, String(slotName));
    if (obj.v) {
      return baja.lt(obj.v.t);
    }
    return null;
  };

  /**
   * Return all the unique slot names for a given Type.
   *
   * @param {Type} type
   * @returns {Array.<String>} the slot names.
   * @since Niagara 4.12
   */
  exports.getSlotNames = function (type) {
    return getSlotNamesInContract(type);
  };

  /**
   * Return all the unique property names for a given Type.
   *
   * @param {Type} type
   * @returns {Array.<String>} the property names.
   * @since Niagara 4.12
   */
  exports.getPropertyNames = function (type) {
    return getSlotNamesInContract(type, function (obj) {
      return obj.st === 'p';
    });
  };

  /**
   * Return all the unique action names for a given Type.
   *
   * @param {Type} type
   * @returns {Array.<String>} the action names.
   * @since Niagara 4.12
   */
  exports.getActionNames = function (type) {
    return getSlotNamesInContract(type, function (obj) {
      return obj.st === 'a';
    });
  };

  /**
   * Return all the unique topic names for a given Type.
   *
   * @param {Type} type
   * @returns {Array.<String>} the topic names.
   * @since Niagara 4.12
   */
  exports.getTopicNames = function (type) {
    return getSlotNamesInContract(type, function (obj) {
      return obj.st === 't';
    });
  };

  /**
   * Flatten out a type's supertype chain into an array of Types, `baja:Object`
   * first.
   *
   * @param {Type} type
   * @returns {Array.<Type>}
   */
  exports.getSuperTypeChain = function (type) {
    var arr = [];
    while (type) {
      arr.push(type);
      type = type.getSuperType();
    }
    return arr.reverse();
  };

  /**
   * Get a unique slot name for adding a new child component to a parent.
   *
   * @param {baja.Component} comp a component to which we're adding a slot
   * @param {String|baja.Simple|Type} [typeSpec] if given, the unique name will
   * match the type name
   * @returns {String} a unique name to use to add a new slot
   */
  exports.getUniqueSlotName = function (comp, typeSpec) {
    if (!typeSpec) {
      return comp.getUniqueName('newSlot');
    }
    var type = baja.lt(String(typeSpec));
    if (!type) {
      throw new Error('type ' + typeSpec + ' not found or not imported');
    }
    return comp.getUniqueName(type.getTypeName());
  };

  /**
   * Convenience method wrapping `baja.importTypes` in a promise.
   *
   * @param {Array.<String>} [types] type array. Type specs can also be passed
   * as direct parameters.
   *
   * @returns {Promise}
   *
   * @example
   *   <caption>Import an array of type specs</caption>
   *
   *   var specs = [ 'baja:Double', 'baja:String', 'baja:Time' ];
   *   typeUtils.importTypes(specs)
   *     .then(function (types) {
   *       //array of types
   *     });
   *
   * @example
   *   <caption>Import type specs as parameters</caption>
   *
   *   typeUtils.importTypes('baja:Double', 'baja:String', 'baja:Time')
   *     .then(([ doubleType, stringType, timeType ]) => {
   *       //types as parameters
   *     });
   */
  exports.importTypes = function (types) {
    var isArray = Array.isArray(types),
      arr = isArray ? types : Array.prototype.slice.call(arguments);
    return Promise["try"](function () {
      _.each(arr, function (typeSpec) {
        if (knownBadTypes[typeSpec]) {
          //if we know we've got a bad Type, save ourselves the network call
          throw new Error('unknown type: ' + typeSpec);
        }
      });
      if (!arr[0]) {
        //nothing to resolve
        return [];
      } else {
        return baja.importTypes({
          typeSpecs: arr
        })["catch"](function (err) {
          _.each(err, function (val, typeSpec) {
            knownBadTypes[typeSpec] = true;
          });
          throw err;
        });
      }
    });
  };

  /**
   * @param {*} obj
   * @returns {Boolean} true if the object is an instance of `baja:Value`.
   */
  exports.isValue = function (obj) {
    return baja.hasType(obj, 'baja:Value');
  };

  /**
   * @param {*} obj
   * @returns {Boolean} true if the object is an instance of `baja:Simple`.
   */
  exports.isSimple = function (obj) {
    return baja.hasType(obj, 'baja:Simple');
  };

  /**
   * Return true if the object is an instance of `baja:Number`. Note that all
   * of these implement `valueOf()` and so can be used in mathematical
   * calculations.
   *
   * @param {*} obj
   * @returns {Boolean}
   */
  exports.isNumber = function (obj) {
    return baja.hasType(obj, 'baja:Number');
  };

  /**
   * @param {Object} obj
   * @returns {Boolean} true if the object is an instance of `baja:Complex`.
   */
  exports.isComplex = function (obj) {
    return baja.hasType(obj, 'baja:Complex');
  };

  /**
   * @param {*} obj
   * @returns {Boolean} true if the object is an instance of `baja:Component`.
   */
  exports.isComponent = function (obj) {
    return baja.hasType(obj, 'baja:Component');
  };

  /**
   * @param {*} obj
   * @returns {boolean} true if object is a Knob
   */
  exports.isKnob = function (obj) {
    return obj && !baja.hasType(obj) && _.isFunction(obj.getSourceSlotName) && _.isFunction(obj.getSourceComponent);
  };

  /**
   * @param {*} obj
   * @returns {boolean} true if object is a relation knob
   */
  exports.isRelationKnob = function (obj) {
    return obj && _.isFunction(obj.getRelationId) && _.isFunction(obj.getEndpointComponent);
  };

  /**
   * Return true if the parameter is a type, or type spec.
   *
   * @param {Object|String} type
   * @returns {Boolean} true if the object is an instance of a baja type, or
   * a string that looks like a baja type spec
   */
  exports.isType = function (type) {
    if (type) {
      return TYPE_SPEC_REGEX.test(type) || typeof type.getTypeSpec === 'function';
    }
    return false;
  };

  /**
   * @param {string|Type|Function} type a baja type or type spec, or constructor function
   * @param {Array.<*>} arr
   * @returns {boolean} true if given an array, and all members are instances of the specified type
   * @since Niagara 4.14
   */
  exports.isArrayOf = function (type, arr) {
    if (!Array.isArray(arr)) {
      return false;
    }

    // special checks for primitives
    if (type === Number) {
      type = 'baja:Double';
    } else if (type === Boolean) {
      type = 'baja:Boolean';
    } else if (type === String) {
      type = 'baja:String';
    }
    if (exports.isType(type)) {
      return every(arr, function (v) {
        return baja.hasType(v, type);
      });
    } else if (typeof type === 'function') {
      return every(arr, function (v) {
        return v instanceof type;
      });
    } else {
      throw new Error('type required');
    }
  };

  /**
   * Check to see if one constructor is a subclass of another constructor.
   *
   * @param {Function|Object} superClass constructor function, or object whose
   * constructor to check
   * @param {Function|Object} subClass constructor function, or object whose
   * constructor to check
   * @returns {Boolean} true if subClass inherits from superClass, or if they
   * are the same constructor
   */
  exports.isAssignableFrom = function (superClass, subClass) {
    if (!superClass || !subClass) {
      return false;
    }
    var superCtor = typeof superClass === 'function' ? superClass : superClass.constructor,
      subCtor = typeof subClass === 'function' ? subClass : subClass.constructor;
    return Object.create(subCtor.prototype) instanceof superCtor;
  };

  /**
   * Creates a `baja:TypeSpec` instance from the given type spec string.
   *
   * @param {baja.Value|String|Type} str type spec (to be toStringed)
   * @returns {baja.Value} a `baja:TypeSpec` instance, suitable for adding to
   * a Component
   */
  exports.toTypeSpec = function (str) {
    return baja.$('baja:TypeSpec', String(str));
  };

  /**
   * Returns a Promise that will resolve to a suitable display
   * name string for the given type. This will try to obtain a string
   * from the lexicon for the module or, if no entry is found, it
   * will return a 'friendly' string from the name contained in
   * the type spec.
   *
   * @param {Type|String} typeSpec
   * @returns {Promise.<String>}
   */
  exports.getTypeDisplayName = function (typeSpec) {
    var str = String(typeSpec);
    if (str === 'null') {
      return Promise.resolve(str);
    }
    var type = baja.lt(str);
    var _ref = type ? [type.getModuleName(), type.getTypeName()] : String(typeSpec).split(':'),
      _ref2 = _slicedToArray(_ref, 2),
      module = _ref2[0],
      name = _ref2[1];
    return lexjs.module(module).then(function (lex) {
      return lex.get(name + '.displayName') || toFriendly(name);
    })["catch"](function () {
      return toFriendly(name);
    });
  };
  return exports;
});
