/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */
/*jshint devel: true */
/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/compat/RelevantStringEditor
 */
define(['jquery', 'nmodule/webEditors/rc/fe/baja/StringEditor', 'nmodule/js/rc/jquery/Relevant-Dropdowns/js/jquery.relevant-dropdown'], function ($, StringEditor) {
  'use strict';

  /**
   * Extension of StringEditor to shim in `datalist` support via the
   * Relevant Dropdowns jQuery plugin:
   *
   * http://css-tricks.com/relevant-dropdowns-polyfill-for-datalist/
   *
   * This editor will probably never be removed - it allows for custom rendering
   * of `datalist`s that we'll sometimes need but will probably never be
   * supported in browsers. Use the following Properties:
   *
   * - `shimDatalist`: true by default, which forces the use of a datalist shim,
   *   even if the browser does have native `datalist` support. It's best to
   *   leave this on for consistency across browsers. Set to false to force
   *   the use of native browser datalist functionality. (This property
   *   supersedes `relevant-force`.)
   * - `filterWhileTyping`: true by default. the usual `datalist` behavior is to
   *   filter down the displayed options to only those that match the string
   *   currently typed in. Set this to `false` to show *all* options no matter
   *   what string is currently entered. Useful for matching `DynamicEnum`
   *   behavior in WB. (This property supersedes `relevant-filter`.)
   *
   * @class
   * @alias module:nmodule/webEditors/rc/fe/baja/compat/RelevantStringEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/StringEditor
   * @see module:nmodule/js/rc/jquery/Relevant-Dropdowns/js/jquery.relevant-dropdown
   */
  var RelevantStringEditor = function RelevantStringEditor() {
    StringEditor.apply(this, arguments);
  };
  RelevantStringEditor.prototype = Object.create(StringEditor.prototype);
  RelevantStringEditor.prototype.constructor = RelevantStringEditor;
  RelevantStringEditor.prototype.$getDatalistElement = function () {
    return this.jq().children('datalist, .datalist');
  };
  RelevantStringEditor.prototype.$isDatalistShowing = function () {
    var dl = $('#' + this.$getTextElement().attr('list'));
    return dl.length && dl.css('display') !== 'none' && parseFloat(dl.css('opacity')) > 0;
  };

  /**
   * Set up Relevant dropdown on input if a datalist is present.
   */
  RelevantStringEditor.prototype.$rebuildHtml = function () {
    var that = this,
      dom = that.jq();

    //clean up handlers from previous Relevant call
    that.$destroyDatalist();
    dom.children('input, ul').off();
    StringEditor.prototype.$rebuildHtml.apply(this, arguments);
    var datalist = dom.children('datalist'),
      input = dom.children('input');
    if (datalist.length) {
      //if the user is pressing enter to select a value from the dropdown,
      //swallow it once, so it doesn't cause a form submission.
      //This also applies to the escape key so the dialog esc close will not fire while the datalist is showing.
      input.on('keyup keydown keypress', function (e) {
        if (e.keyCode === 13 || e.keyCode === 27) {
          //enter or escape
          if (that.$isDatalistShowing()) {
            return false;
          }
        }
      });
      var props = that.properties();
      var shimDatalist = props.getValue('shimDatalist');
      var filterWhileTyping = props.getValue('filterWhileTyping');
      var relevantForce = props.getValue('relevant-force');
      var relevantFilter = props.getValue('relevant-filter');
      if (shimDatalist !== false) {
        var force = shimDatalist || relevantForce !== false;
        var showAll = filterWhileTyping === false || relevantFilter === false;
        this.$relevant(input, {
          datalist: datalist,
          force: force,
          showAll: showAll
        });
      }
    }
  };

  /**
   * @private
   * @param {JQuery} input
   * @param {object} relevantParams
   */
  RelevantStringEditor.prototype.$relevant = function (input, _ref) {
    var _this = this;
    var datalist = _ref.datalist,
      force = _ref.force,
      showAll = _ref.showAll;
    input.relevantDropdown({
      force: force,
      showAll: showAll,
      fadeOutSpeed: 100,
      change: function change() {
        var textElement = _this.$getTextElement();
        //ensure we get modify events when selecting from datalist w/ enter key
        textElement.trigger('input');
        if (_this.properties().getValue('hideText')) {
          textElement.blur();
        }
        _this.trigger(RelevantStringEditor.events.LIST_ACTION_PERFORMED, textElement.val());
      },
      datalist: datalist
    });
  };
  RelevantStringEditor.prototype.$getDatalistElement = function () {
    return $('#' + this.$getTextElement().attr('list'));
  };
  RelevantStringEditor.prototype.$destroyDatalist = function () {
    var list = this.$getDatalistElement();
    list.off().remove();
  };
  RelevantStringEditor.prototype.doDestroy = function () {
    this.$destroyDatalist();
    this.jq().children('input, ul').off();
    return StringEditor.prototype.doDestroy.apply(this, arguments);
  };

  /**
   * List of all RelevantStringEditor events.
   *
   * @alias module:nmodule/webEditors/rc/fe/baja/compat/RelevantStringEditor~events
   */
  RelevantStringEditor.events = {
    /** Triggers when an item from the dropdown is picked by the user. */
    LIST_ACTION_PERFORMED: 'webeditors:listactionperformed'
  };
  return RelevantStringEditor;
});
