function _typeof(o) { "@babel/helpers - typeof"; return _typeof = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function (o) { return typeof o; } : function (o) { return o && "function" == typeof Symbol && o.constructor === Symbol && o !== Symbol.prototype ? "symbol" : typeof o; }, _typeof(o); }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/StringEditor
 */
define(['baja!', 'lex!webEditors', 'jquery', 'underscore', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/util/htmlUtils', 'hbs!nmodule/webEditors/rc/fe/baja/template/StringEditor'], function (baja, lexs, $, _, BaseEditor, htmlUtils, tplStringEditor) {
  'use strict';

  var DEFAULT_FIELD_WIDTH = 40,
    selectOnFocus = htmlUtils.selectOnFocus,
    selectAll = htmlUtils.selectAll,
    webEditorsLex = lexs[0];
  function getNumber(num) {
    return baja.hasType(num, 'baja:Number') && num.valueOf();
  }
  function inRange(ed, val) {
    var length = val.length,
      props = ed.properties(),
      max = getNumber(props.getValue('max')),
      min = getNumber(props.getValue('min'));
    if (typeof max === 'number' && length > max) {
      throw new Error('length ' + length + ' > ' + max);
    } else if (typeof min === 'number' && length < min) {
      throw new Error('length ' + length + ' < ' + min);
    }
  }
  function validSlotName(ed, val) {
    var mustBeSlotName = ed.properties().getValue('mustBeSlotName');
    if (mustBeSlotName && !baja.SlotPath.isValidName(val)) {
      throw new Error('Illegal name "' + val + '"');
    }
  }
  function throwItemNotInDatalistValidationError(val) {
    throw new Error(webEditorsLex.get('StringEditor.itemNotInDatalistError', val));
  }
  function validateInDatalist(ed, val) {
    var datalistItemsOnly = ed.properties().getValue('datalistItemsOnly'),
      datalist = ed.$getDatalist();
    if (!datalistItemsOnly) {
      return;
    }
    if (!datalist) {
      throwItemNotInDatalistValidationError(val);
    }
    if (typeof datalist === 'string') {
      if (!datalist.split(';').contains(val)) {
        throwItemNotInDatalistValidationError(val);
      }
      return;
    }
    if (datalist[val] === undefined) {
      throwItemNotInDatalistValidationError(val);
    }
  }
  var widgetDefaults = function widgetDefaults() {
    return {
      keyName: 'StringEditor',
      properties: {
        inputType: {
          value: 'text',
          hidden: true,
          readonly: true
        }
      }
    };
  };

  /**
   * A field editor for working with strings.
   *
   * It accepts the following Properties:
   *
   * - `datalist`: (string|object) a semicolon-separated list of strings to use
   *   to generate an HTML `datalist` element. Alternatively, it can be an
   *   object literal from values to display strings; the display strings will
   *   be used to create the `datalist`, but `doRead()` will resolve the actual
   *   value if it corresponds to the entered display string. Useful for enum
   *   values with display tags, etc.
   * - `datalistItemsOnly`: {boolean} if true, the widget will ensure the value
   *   currently entered is in the items provided by the datalist.
   * - `inputType`: (string) when not in multiline mode, set the `input` tag's
   *   `type` attribute. Defaults to `text`.
   * - `max`: (number) max length for the input.
   * - `min`: (number) min length for the input.
   * - `multiLine`: (boolean) if true, editor will be a `textarea`.
   * - `mustBeSlotName`: (boolean) input value must be a valid slot name.
   * - `placeholder`: (string) if given, the `placeholder` attribute of the
   *   text input will be set.
   * - `escapeNewLines`: {boolean} Starting in Niagara 4.15, if true, any newline characters will show up as `\n` to the end user.
   *   Also, if they type `\` then `n`, this will be saved as another newline character `\n`. Defaults to `false`.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/StringEditor
   *
   * @param {baja.Facets} [params.facets] if a semicolon-separated string
   * `datalist` facet is provided, a `<datalist>` element will be appended for
   * autocomplete purposes.
   */
  var StringEditor = function StringEditor(params) {
    BaseEditor.call(this, {
      params: params,
      defaults: widgetDefaults()
    });

    /**
     * A StringEditor will fail to validate under the following conditions:
     * - `datalistItemsOnly` property is set to true and the following met:
     *   - For a string datalist, item is not in the datalist.
     *   - For an object datalist value, it will fail to validate if the value
     *     does not correspond to neither a key or a display value from the
     *     datalist object.
     * - `max` property is set and the current string value is too long
     * - `min` property is set and the current string value is too short
     * - `mustBeSlotName` property is set and the current string value is not a
     *   valid slot name
     * @method module:nmodule/webEditors/rc/fe/baja/StringEditor#validate
     */
    var validators = this.validators();
    validators.add(function (val) {
      inRange(this, val);
    });
    validators.add(function (val) {
      validSlotName(this, val);
    });
    validators.add(function (val) {
      validateInDatalist(this, val);
    });
    this.$prevText = '';
  };
  StringEditor.prototype = Object.create(BaseEditor.prototype);
  StringEditor.prototype.constructor = StringEditor;
  StringEditor.prototype.$isMultiLine = function () {
    return this.properties().getValue('multiLine');
  };
  StringEditor.prototype.$getTextElement = function () {
    var tag = this.$isMultiLine() ? 'textarea' : 'input';
    return this.jq().children(tag);
  };
  StringEditor.prototype.$getDatalistElement = function () {
    return this.jq().children('datalist');
  };
  StringEditor.prototype.$getDatalist = function () {
    return this.properties().getValue('datalist');
  };
  StringEditor.prototype.$selectAll = function () {
    selectAll(this.$getTextElement());
  };
  StringEditor.prototype.requestFocus = function () {
    BaseEditor.prototype.requestFocus.apply(this, arguments);
    this.$selectAll();
  };

  /**
   * Creates a text element. If the `multiLine` facet is set, will create a
   * `<textarea>` element (setting the `cols` property to the
   * `fieldWidth` facet, if present). Otherwise, creates
   * `<input type="text">`.
   *
   * @param {JQuery} dom
   */
  StringEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on('input', 'input, textarea', function () {
      var val = this.value;
      if (val !== that.$prevText) {
        that.setModified(true);
      }
      that.$prevText = val;
    }).on('keyup', 'textarea', false);
    selectOnFocus(that, 'input', function () {
      return that.$selectAll();
    });
    return that.$rebuildHtml();
  };

  /**
   * Wipe and rebuild the contents of this editor's DOM. Why implement this
   * separately instead of directly in `doInitialize`? When the `datalist`
   * property changes, the safest/easiest thing to do is just to wipe and
   * rebuild with the correct `datalist` tag. This also gives subclasses
   * (namely, `RelevantStringEditor` a chance to get their own hooks in.
   *
   * @protected
   */
  StringEditor.prototype.$rebuildHtml = function () {
    var that = this,
      props = that.properties(),
      autocomplete = props.getValue('autocomplete'),
      datalist = that.$getDatalist(),
      fieldWidth = props.getValue('fieldWidth'),
      inputType = props.getValue('inputType') || 'text',
      placeholder = props.getValue('placeholder'),
      multiLine = that.$isMultiLine(),
      listId;
    if (autocomplete === 'off' && inputType === 'password') {
      autocomplete = 'new-password';
    } else if (autocomplete !== 'off') {
      autocomplete = '';
    }
    if (typeof fieldWidth !== 'number') {
      fieldWidth = DEFAULT_FIELD_WIDTH;
    }
    if (datalist) {
      if (typeof datalist === 'string') {
        datalist = datalist.split(';');
      } else {
        datalist = _.map(datalist, _.constant);
      }
      listId = that.generateId();
    }
    that.jq().html(tplStringEditor({
      multiLine: multiLine,
      fieldWidth: fieldWidth,
      inputType: inputType,
      autocomplete: autocomplete,
      placeholder: placeholder,
      datalist: datalist,
      listId: listId,
      value: that.$getTextElement().val()
    }));
  };

  /**
   * Loads the given string into the text element. If value is not already
   * a string, it will be toString-ed (or have `encodeToString` called if a
   * `baja:Simple`.
   *
   * @param {String|baja.Simple} val
   */
  StringEditor.prototype.doLoad = function (val) {
    var str = this.$valueToString(val),
      datalist = this.$getDatalist();
    if (datalist && _typeof(datalist) === 'object') {
      str = datalist[str] || str;
    }
    this.$prevText = str;
    this.$getTextElement().val(str);
  };

  /**
   * Reads the currently entered string from the text element. (If a value of
   * type `baja:Simple` was loaded, an instance of the same Type will be
   * decoded from the entered string and returned.)
   *
   * @returns {String|baja.Simple|Promise.<baja.Simple>} the currently entered string or the decoded
   * value if the value is a simple
   */
  StringEditor.prototype.doRead = function () {
    var value = this.value(),
      str = this.$getTextElement().val(),
      datalist = this.$getDatalist();
    if (datalist && _typeof(datalist) === 'object') {
      str = _.invert(datalist)[str] || str;
    }
    if (this.properties().getValue('escapeNewLines')) {
      str = str.replace(/(\\n)/gm, "\n");
    }
    if (!baja.hasType(value, 'baja:Simple')) {
      return str;
    }
    return baja.$(value.getType()).decodeAsync(str);
  };
  StringEditor.prototype.doChanged = function (prop) {
    if (prop === 'datalist') {
      return this.$rebuildHtml();
    }
  };

  /**
   * Enables or disables the text element.
   *
   * @param {Boolean} enabled
   */
  StringEditor.prototype.doEnabled = function (enabled) {
    this.$getTextElement().prop('disabled', !enabled);
  };

  /**
   * Sets the text element to readonly / not readonly.
   *
   * @param {Boolean} readonly
   */
  StringEditor.prototype.doReadonly = function (readonly) {
    this.$getTextElement().prop('readonly', readonly);
  };

  /**
   * @private
   * @param {String|baja.Simple} val
   * @returns {String}
   */
  StringEditor.prototype.$valueToString = function (val) {
    if (baja.hasType(val, 'baja:Simple')) {
      val = val.encodeToString();
    } else {
      val = String(val);
    }
    if (this.properties().getValue('escapeNewLines')) {
      val = val.replace(/(\r\n|\n|\r)/g, "\\n");
    }
    return val;
  };
  return StringEditor;
});
