/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/FacetsEditor
 */
define(['baja!', 'lex!webEditors', 'log!nmodule.webEditors.rc.fe.baja.FacetsEditor', 'bajaux/Properties', 'bajaux/Widget', 'jquery', 'Promise', 'underscore', 'bajaux/events', 'bajaux/mixin/responsiveMixIn', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/BaseWidget', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor', 'nmodule/webEditors/rc/fe/baja/FacetsRowEditor', 'nmodule/webEditors/rc/wb/options/MruButton', 'nmodule/webEditors/rc/wb/options/MruOptions', 'hbs!nmodule/webEditors/rc/fe/baja/template/FacetsEditor'], function (baja, lexs, log, Properties, Widget, $, Promise, _, events, responsiveMixIn, fe, BaseWidget, BaseEditor, DisplayOnlyEditor, FacetsRowEditor, MruButton, MruOptions, tplFacetsEditor) {
  'use strict';

  var MODIFY_EVENT = events.MODIFY_EVENT,
    DESTROY_EVENT = events.DESTROY_EVENT,
    ENABLE_EVENT = events.ENABLE_EVENT,
    DISABLE_EVENT = events.DISABLE_EVENT,
    READONLY_EVENT = events.READONLY_EVENT,
    WRITABLE_EVENT = events.WRITABLE_EVENT,
    VALUE_READY_EVENT = BaseWidget.VALUE_READY_EVENT,
    ROW_REMOVED_EVENT = FacetsRowEditor.ROW_REMOVED_EVENT,
    MRU_NAME = 'facetsFE',
    webEditorsLex = lexs[0],
    logError = log.severe.bind(log);

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Builds and returns new row.
   *
   * @param {module:nmodule/webEditors/rc/fe/baja/FacetsEditor} ed
   * @param {Object} params the object literal containing the method's arguments
   * @param {String} params.key the key for the new row
   * @param {baja.Value} params.value the value for the new row
   * @returns {Promise}
   */
  function buildTableRow(ed, _ref) {
    var key = _ref.key,
      value = _ref.value;
    var tr = $('<tr class="ux-table-row"/>');
    var uxFieldEditor = ed.$getUxFieldEditor(key);
    return ed.buildChildFor({
      dom: tr,
      type: FacetsRowEditor,
      value: {
        key: key,
        value: value
      },
      properties: Properties.extend(ed.properties().subset(['allowAdd', 'allowChangeType', 'allowDelete', 'allowRename', 'hideDefaultKeys']), uxFieldEditor && {
        uxFieldEditor: uxFieldEditor
      })
    });
  }

  /**
   * Adds a new row to the facets editor.
   *
   * @param {module:nmodule/webEditors/rc/fe/baja/FacetsEditor} ed
   * @param {String} key the key for the new row
   * @param {baja.Value} value the value for the new row
   * @returns {Promise} promise to be resolved after a new table row
   * has been created, loaded with editors for key, type, and value, and
   * appended to the editor DOM
   */
  function addRow(ed, key, value) {
    return buildTableRow(ed, {
      key: key,
      value: value
    }).then(function (tr) {
      ed.jq().children('table').children('tbody').append(tr.jq());
      return ed.setModified(true);
    });
  }

  /**
   * When a row editor is destroyed, will completely remove its DOM element
   * and set the editor to modified.
   *
   * @param {module:nmodule/webEditors/rc/fe/baja/FacetsEditor} ed
   * @param {JQuery} tr the table row to remove
   */
  function removeRow(ed, tr) {
    tr.remove();
    ed.setModified(true);
  }

  ////////////////////////////////////////////////////////////////
  // FacetsEditor definition
  ////////////////////////////////////////////////////////////////

  /**
   * A field editor for working with `Facets` values.
   *
   * A `FacetsEditor` supports the following `bajaux` `Properties`:
   *
   * - `allowAdd`: set to `false` to prevent new facets from being added.
   * - `allowChangeType`: set to `false` to prevent facet types from being
   *   changed.
   * - `allowDelete`: set to `false` to prevent facets from being removed.
   * - `allowRename`: set to `false` to prevent facet names from being changed.
   * - `showHeader`: set to `false` to prevent the header from being shown.
   * - `uxFieldEditors`: set this to a `baja.Facets` instance where the keys
   *   correspond to the keys in the edited `Facets`. The values should be
   *   type specs or RequireJS module IDs. This allows you to specify exactly
   *   what kind of editor to use to edit facets.
   * - `mru`: by default mru name will be `FacetsFE`, user can override with his mru name.
   *   This will let user have control over the mru options.
   * - `hideDefaultKeys`: set to `false`. One can hide Facets Key suggestions if this flag is set to `true`.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/FacetsEditor
   */
  var FacetsEditor = function FacetsEditor(params) {
    var that = this;
    BaseEditor.call(that, _.extend({
      keyName: 'FacetsEditor'
    }, params));
    if (that.getFormFactor() === Widget.formfactor.mini) {
      that.$makeDisplayOnly();
    }
    responsiveMixIn(that, {
      'editor-FacetsEditor-mobile': {
        maxWidth: 600
      }
    });
    that.on(VALUE_READY_EVENT, function (facets) {
      return that.$saveMru(facets);
    });
  };
  FacetsEditor.prototype = Object.create(BaseEditor.prototype);
  FacetsEditor.prototype.constructor = FacetsEditor;
  FacetsEditor.prototype.$makeDisplayOnly = function () {
    DisplayOnlyEditor.$mixin(this);
  };

  /**
   * Return true if this editor will allow you to add new facets.
   *
   * @private
   * @returns {boolean}
   */
  FacetsEditor.prototype.$isAllowAdd = function () {
    return this.properties().getValue('allowAdd') !== false;
  };

  /**
   * Return true if this editor will allow you change the types of facet values.
   *
   * @private
   * @returns {boolean}
   */
  FacetsEditor.prototype.$isAllowChangeType = function () {
    return this.properties().getValue('allowChangeType') !== false;
  };

  /**
   * Return true if this editor will allow you to delete facets.
   *
   * @private
   * @returns {boolean}
   */
  FacetsEditor.prototype.$isAllowDelete = function () {
    return this.properties().getValue('allowDelete') !== false;
  };

  /**
   * @private
   * @returns {string} Returns user supplied MRU if available, otherwise `FacetsFE`.
   */
  FacetsEditor.prototype.$getMruName = function () {
    return this.properties().getValue('mru') || MRU_NAME;
  };

  /**
   * Return true if this editor will allow you to rename facets. Automatically
   * false if `allowAdd` is false (how can I add a new facet if I can't
   * rename it?)
   *
   * @private
   * @returns {boolean}
   */
  FacetsEditor.prototype.$isAllowRename = function () {
    return this.$isAllowAdd() && this.properties().getValue('allowRename') !== false;
  };

  /**
   * Return true if this editor should show the header.
   *
   * @private
   * @returns {boolean}
   */
  FacetsEditor.prototype.$isShowHeader = function () {
    return this.properties().getValue('showHeader') !== false;
  };

  /**
   * If the editor specifies a particular editor for a particular key, retrieve
   * that type spec or RequireJS module ID.
   *
   * @private
   * @param {String} key
   * @returns {String}
   */
  FacetsEditor.prototype.$getUxFieldEditor = function (key) {
    var facets = this.properties().getValue('uxFieldEditors');
    return baja.hasType(facets, 'baja:Facets') && facets.get(key);
  };
  FacetsEditor.prototype.$getRows = function () {
    return this.getChildWidgets({
      type: FacetsRowEditor
    });
  };

  /**
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/options/MruButton}
   */
  FacetsEditor.prototype.$getMruButton = function () {
    return this.jq().find('.mru').data('widget');
  };

  /**
   * Build the MRU button to load in the selected facets, and to use
   * `toString()`s as display values. Will not build an MruButton if
   * `allowAdd=false`.
   * @private
   * @returns {Promise}
   */
  FacetsEditor.prototype.$buildMruButton = function () {
    var that = this;
    if (!that.$isAllowAdd()) {
      return Promise.resolve();
    }
    var select = function select(str) {
      return decodeAsync(str).then(function (facets) {
        return that.load(facets);
      }).then(function () {
        return that.setModified(true) || null;
      })["catch"](logError);
    };
    return MruOptions.make(that.$getMruName()).then(function (options) {
      return fe.buildFor({
        value: {
          options: options,
          select: select,
          toDisplayString: toDisplayString
        },
        type: MruButton,
        dom: that.jq().find('.mru')
      });
    });
  };

  /**
   * Creates a table to hold the array of facet values.
   *
   * @param {JQuery} dom
   */
  FacetsEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.html(tplFacetsEditor({
      headers: {
        key: webEditorsLex.get('FacetsEditor.key'),
        type: webEditorsLex.get('FacetsEditor.type'),
        value: webEditorsLex.get('FacetsEditor.value')
      },
      allowAdd: that.$isAllowAdd(),
      showHeader: that.$isShowHeader(),
      allowChangeType: that.$isAllowChangeType()
    }));
    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });
    dom.on([ENABLE_EVENT, DISABLE_EVENT, READONLY_EVENT, WRITABLE_EVENT, DESTROY_EVENT].join(' '), '.editor', false);
    dom.on('click', 'button.add', function (e) {
      var button = $(e.target);

      //prevent double-click tomfoolery
      button.prop('disabled', true);
      addRow(that, '', '').then(function () {
        button.prop('disabled', false);
      })["catch"](logError);
    });
    dom.on(ROW_REMOVED_EVENT, '.editor', function (e, ed) {
      ed.destroy().then(function () {
        removeRow(that, $(e.target));
      })["catch"](logError);
      return false;
    });
    return !baja.isOffline() && that.$buildMruButton();
  };

  /**
   * Loads the `Facets` into the editor, creating a table row for each key
   * in the `Facets`. Each row has field editors for key, type, and value.
   *
   * @param {baja.Facets} value
   * @returns {Promise} promise to be resolved when all the field editors
   * have finished loading into the DOM
   */
  FacetsEditor.prototype.doLoad = function (value) {
    var that = this;
    return Promise.all(value.getKeys().map(function (key) {
      return buildTableRow(that, {
        key: baja.SlotPath.unescape(key),
        value: value.get(key)
      });
    })).then(function (eds) {
      var elems = _.map(eds, function (ed) {
        return ed.jq();
      });
      that.jq().children('table').children('tbody').html(elems);
      that.jq().find('input').addClass('nofocus');
    });
  };

  /**
   * Reads key and value editors from each row and assembles them into a
   * `Facets` instance.
   *
   * @returns {Promise} promise to be resolved with a `baja.Facets`
   * instance read from key and value editors, or rejected in the case of
   * duplicate keys or invalid values
   */
  FacetsEditor.prototype.doRead = function () {
    var that = this;
    return that.$getRows().readAll().then(function (rowValues) {
      var keys = [],
        values = [];
      _.each(rowValues, function (rowValue) {
        var key = rowValue.key,
          value = rowValue.value;
        if (keys.indexOf(key) >= 0) {
          throw new Error('duplicate key "' + key + '"');
        } else {
          keys.push(baja.SlotPath.escape(key));
          values.push(value);
        }
      });
      return baja.Facets.make(keys, values);
    });
  };
  FacetsEditor.prototype.$saveRowMrus = function (facets) {
    return Promise.all(this.$getRows().map(function (row) {
      return row.$getKeyEditor().read().then(function (key) {
        return row.emitAndWait(VALUE_READY_EVENT, facets.get(key));
      });
    }));
  };

  /**
   * @param {baja.Facets} facets
   * @returns {Promise}
   */
  FacetsEditor.prototype.$saveMru = function (facets) {
    if (baja.isOffline()) {
      return Promise.resolve();
    }
    var that = this;
    return Promise.all([this.$saveRowMrus(facets), MruOptions.make(that.$getMruName()).then(function (mru) {
      mru.add(facets.encodeToString());
      return mru.save();
    })]);
  };

  /**
   * When this editor is saved, add the entered facets to the FacetsEditor MRU
   * history.
   * @param {baja.Facets} facets
   * @returns {Promise}
   */
  FacetsEditor.prototype.doSave = function (facets) {
    return this.$saveMru(facets);
  };

  /**
   * Returns the string representation of the `Facets` object, passing the
   * `Facets` object (itself) as the context for the toString() function.
   *
   * @returns {String|Promise.<String>}
   */
  FacetsEditor.prototype.valueToString = function (facets) {
    if (baja.hasType(facets, 'baja:Facets')) {
      var obj = _.extend(this.properties().toValueMap(), facets.toObject());
      delete obj.radix;
      return facets.toString(obj);
    } else {
      return DisplayOnlyEditor.prototype.valueToString.apply(this, arguments);
    }
  };

  /**
   * Enables or disables all sub-editors (keys, types, values) and the add
   * button.
   *
   * @param {Boolean} enabled
   * @returns {Promise} promise to be resolved when all child editors
   * are enabled/disabled
   */
  FacetsEditor.prototype.doEnabled = function (enabled) {
    this.jq().find('button.add').prop('disabled', this.isReadonly() || !enabled);
    return this.getChildWidgets().setAllEnabled(enabled);
  };

  /**
   * Sets all sub-editors (keys, types, values) readonly/writable and
   * enables/disables the add button.
   *
   * @param {Boolean} readonly
   * @returns {Promise} promise to be resolved when all child editors
   * are readonly/writable
   */
  FacetsEditor.prototype.doReadonly = function (readonly) {
    this.jq().find('button.add').prop('disabled', !this.isEnabled() || readonly);
    return this.getChildWidgets().setAllReadonly(readonly);
  };

  /**
   * Destroy all facet rows.
   *
   * @returns {Promise}
   */
  FacetsEditor.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };
  function decodeAsync(str) {
    return Promise.resolve(baja.Facets.DEFAULT.decodeAsync(str));
  }
  function toDisplayString(str) {
    return decodeAsync(str).then(String);
  }
  return FacetsEditor;
});
